package gov.va.nvap.server.endpoint.vista.link;

import gov.va.med.exception.FoundationsException;
import gov.va.med.vistalink.adapter.cci.VistaLinkConnection;
import gov.va.med.vistalink.rpc.RpcRequest;
import gov.va.med.vistalink.rpc.RpcRequestFactory;
import gov.va.med.vistalink.rpc.RpcRequestParams;
import gov.va.med.vistalink.rpc.RpcResponse;
import gov.va.med.vistalink.security.VistaKernelPrincipalImpl;
import gov.va.nvap.common.auth.context.LoginContextExt;
import gov.va.nvap.common.endpoint.EndpointException;
import gov.va.nvap.common.validation.NullChecker;
import java.net.SocketTimeoutException;

import java.util.List;
import java.util.Map;

import javax.resource.ResourceException;
import javax.security.auth.callback.CallbackHandler;
import javax.security.auth.login.Configuration;
import javax.security.auth.login.LoginContext;
import javax.security.auth.login.LoginException;
import org.apache.commons.lang.exception.ExceptionUtils;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.springframework.beans.factory.annotation.Required;

/**
 * VistALink Endpoint to make RPC calls into VistA systems.
 * 
 * @author Asha Amritraj
 */
public class VistALinkEndpoint implements
		gov.va.nvap.common.endpoint.Endpoint<Object, String> {

        private static final Log LOG = LogFactory.getLog(VistALinkEndpoint.class);
	private CallbackHandler callbackHandler;
	private Configuration configuration;
	private String loginContextName;
	private Map<Integer, String> parameterTypes;
	private String rpcContext;
	private String rpcName;
	private int timeout;
        private int retries;
        
	public String getRpcContext() {
		return this.rpcContext;
	}

	public String getRpcName() {
		return this.rpcName;
	}

	public int getTimeout() {
		return this.timeout;
	}
        
        public int getRetries() {
		return this.retries;
	}

	@Override
        public String invoke(final Object args) throws EndpointException {
            return this.invoke(args, 0);
        }
        
	public String invoke(final Object args, final int retriesUsed) throws EndpointException {

		VistaLinkConnection connection = null;
		LoginContext loginContext = null;
		try {
			// Login using the access/verify code set in the callback handler
			loginContext = new LoginContextExt(this.loginContextName,
					this.callbackHandler, this.configuration);
			loginContext.login();
			final VistaKernelPrincipalImpl userPrincipal = VistaKernelPrincipalImpl
					.getKernelPrincipal(loginContext.getSubject());
			connection = userPrincipal.getAuthenticatedConnection();
			connection.setTimeOut(this.timeout);
			final RpcRequest rpcRequest = RpcRequestFactory.getRpcRequest(
					this.rpcContext, this.rpcName);
			final RpcRequestParams params = rpcRequest.getParams();

			if (List.class.isInstance(args)) {
				final List<?> argList = (List<?>) args;
				for (int i = 0; i < argList.size(); i++) {
					params.setParam(i + 1, this.parameterTypes.get(i + 1),
							argList.get(i));
				}
			} else if (String.class.isInstance(args)) {
				final String argValue = (String) args;
				params.setParam(1, this.parameterTypes.get(1), argValue);
			}

			final RpcResponse rpcResponse = connection.executeRPC(rpcRequest);
			final String results = rpcResponse.getResults();
			return results;
		} catch (final LoginException ex) {
			throw new RuntimeException(ex);
		} catch (final FoundationsException ex) {
                    if(ExceptionUtils.getRootCause(ex) instanceof SocketTimeoutException) {
                        if(retriesUsed < retries) {
                            return this.invoke(args, retriesUsed+1);
                        }
                        else {
                            LOG.warn("Exhausted " + retriesUsed + " retries, socket timeout while contacting VistA.");
                            throw new RuntimeException(ex);
                        }
                    }
                    else { throw new RuntimeException(ex); }
		} finally {
			if (loginContext != null && NullChecker.isNotEmpty(loginContext)) {
				try {
                                    // Always logout
                                    loginContext.logout();
				} catch (final LoginException ex) {
                                    LOG.warn("Login issue while contacting VistA. Details: " + ex.getMessage());
                                    // Ignore Exception!!
				}
			}
			if (connection != null && NullChecker.isNotEmpty(connection)) {
				try {
                                    connection.close();
				} catch (final ResourceException ex) {
                                    LOG.warn("Problem while contacting VistA. Details: " + ex.getMessage());
				    // Ignore Exception!!
				}
			}
		}
	}

	@Required
	public void setCallbackHandler(final CallbackHandler callbackHandler) {
		this.callbackHandler = callbackHandler;
	}

	@Required
	public void setConfiguration(final Configuration configuration) {
		this.configuration = configuration;
	}

	@Required
	public void setLoginContextName(final String loginContextName) {
		this.loginContextName = loginContextName;
	}

	@Required
	public void setParameterTypes(final Map<Integer, String> parameterTypes) {
		this.parameterTypes = parameterTypes;
	}

	@Required
	public void setRpcContext(final String rpcContext) {
		this.rpcContext = rpcContext;
	}

	@Required
	public void setRpcName(final String rpcName) {
		this.rpcName = rpcName;
	}

	@Required
	public void setTimeout(final int timeout) {
		this.timeout = timeout;
	}
        
        @Required
	public void setRetries(final int retries) {
		this.retries = retries;
	}
}
