package gov.va.nvap.server.endpoint.pip;

import gov.va.med.esr.webservices.jaxws.schemas.GetEESummaryRequest;
import gov.va.med.esr.webservices.jaxws.schemas.GetEESummaryResponse;
import gov.va.nvap.common.validation.NullChecker;
import gov.va.nvap.service.esr.EnrollmentService;
import gov.va.nvap.svc.consentmgmt.PIPInterface;
import gov.va.nvap.svc.consentmgmt.stub.FacilityResolver;
import gov.va.nvap.svc.consentmgmt.stub.data.Facility;
import gov.va.nvap.svc.consentmgmt.stub.data.Location;

import java.util.Locale;
import java.util.Map;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.springframework.beans.factory.annotation.Required;

/**
 * Resolve Patient's Preferred Facility by calling the ESR webservice.
 * 
 * @author PII
 * 
 */
public class FacilityResolverImpl implements FacilityResolver {

	private static Log LOG = LogFactory.getLog(FacilityResolverImpl.class);

	private static String extractLocationCode(String userId) {
		// UserID format for VHA User :
		// <VHA><3CharLocationCode><First5LastName><First1FirstName>
		// Ex:        - 3 Char Location code is ISW, for one, PII !!
		// In other words this method should extract characters between 3-6 from
		// the string. -- Ugly approach

		if (NullChecker.isNullOrEmpty(userId)) {
			return ""; // empty
		}
		// Location codes are stored in UC in the DB
		userId = userId.trim().toUpperCase(Locale.ENGLISH);
		// userids are at-least 6 chars in length // Note: "VHA"<LOC> is 6
		// chars.
		if (userId.startsWith("VHA") && (userId.length() >= 6)) {
			return userId.substring(3, 6);
		}
		return "";
	}

	private EnrollmentService enrollmentService;
	private PIPInterface pip;

	private Map<String, String> testUsersAndFacilities;

	private GetEESummaryRequest buildRequest(final String patientICN) {
		final GetEESummaryRequest request = new GetEESummaryRequest();
		request.setKey(patientICN);
		request.setKeyType("VPID");
		request.setRequestName("preferredFacility");
		return request;
	}

	private String extractPreferredFacility(final GetEESummaryResponse response) {
		String patientPreferredFacility = "";

		if ((response != null) && (response.getSummary() != null)
				&& (response.getSummary().getDemographics() != null)) {
			final String facilityString = response.getSummary()
					.getDemographics().getPreferredFacility();
			if (facilityString == null) {
				return patientPreferredFacility;
			}
			final String[] facilityTokens = facilityString.split("-");
			if ((facilityTokens != null) && (facilityTokens.length == 2)) {
				patientPreferredFacility = facilityTokens[0].trim();
			}
		}
		return patientPreferredFacility;
	}

	public String getFacility(final String patientIcn) {
		return this.getFacilityId(patientIcn);
	}

	private Facility getFacilityByVHAUserId(final String userId) {
		final String locationCode = FacilityResolverImpl
				.extractLocationCode(userId);
		Facility fac = null;
		if (!NullChecker.isNullOrEmpty(locationCode)) {
			Location loc = null;
			try {
				loc = this.pip.getLocationByCode(locationCode);
			} catch (final Exception e) {
				// Swallowing exception, but that is ok. We want a failure in
				// obtaining Location to be translated into a
				// null Location object.
				FacilityResolverImpl.LOG
						.error("Exception thrown trying to retrieve Location by LocationCode["
								+ locationCode + "]. Setting Location to null");
			}
			if (loc != null) {
				fac = loc.getFacility();
			}
		}
		return fac;
	}

	private String getFacilityId(final String patientICN) {
		final String facilityId = this.getPatientPreferredFacility(patientICN);
		if (NullChecker.isNullOrEmpty(facilityId)) {
			// go to MPI
		}
		return facilityId;
	}

	private String getFacilityStationIdByVHAUserId(final String userId) {
		final Facility fac = this.getFacilityByVHAUserId(userId);
		String facilityStationId = "";
		if (fac != null) {
			facilityStationId = fac.getFacilityStation();
		}
		return facilityStationId;
	}

	/**
	 * Returns Facility given a UserId
	 */
	public String getFaciltyStationIdByUserId(final String userId) {
		if (this.isTestUser(userId)) {
			if (FacilityResolverImpl.LOG.isInfoEnabled()) {
				FacilityResolverImpl.LOG
						.info("UserId["
								+ userId
								+ "], is a test user. Retrieving station id from facility-config.xml file");
			}
			return this.testUsersAndFacilities.get(userId);
		}
		if (FacilityResolverImpl.LOG.isInfoEnabled()) {
			FacilityResolverImpl.LOG.info("UserId[" + userId
					+ "], is a not a test user. Retrieving station id from DB");
		}
		return this.getFacilityStationIdByVHAUserId(userId);
	}

	@Override
	public String getPatientPreferredFacility(final String patientICN) {
		String patientPreferredFacility = "";
		// build Request
		final GetEESummaryRequest request = this.buildRequest(patientICN);
		// invoke service
		final GetEESummaryResponse response = this.invokeWS(request);
		// Get patient's preferred facility
		patientPreferredFacility = this.extractPreferredFacility(response);
		return patientPreferredFacility;
	}

	@Override
	public String getUserFacility(final String userId) {
		return this.getFaciltyStationIdByUserId(userId);
	}

	private GetEESummaryResponse invokeWS(final GetEESummaryRequest request) {
		final GetEESummaryResponse response = this.enrollmentService
				.getEESummaryRequest(request);
		return response;
	}

	// private
	private boolean isTestUser(final String userId) {
		return !NullChecker.isNullOrEmpty(userId)
				&& (this.testUsersAndFacilities != null)
				&& this.testUsersAndFacilities.keySet().contains(userId);
	}

	@Required
	public void setEnrollmentService(final EnrollmentService enrollmentService) {
		this.enrollmentService = enrollmentService;
	}

	@Required
	public void setPip(final PIPInterface pip) {
		this.pip = pip;
	}

	@Required
	public void setTestUsersAndFacilities(
			final Map<String, String> testUsersAndFacilities) {
		this.testUsersAndFacilities = testUsersAndFacilities;
	}
}
