package gov.va.nvap.server.endpoint.announce;

import gov.va.med.nhin.adapter.announce.BatchAnnounceType;
import gov.va.nvap.common.validation.NullChecker;
import gov.va.nvap.privacy.OrganizationType;
import gov.va.nvap.server.endpoint.adapter.announce.transformer.AnnouncementToAnnouncePatientRequest;
import gov.va.nvap.service.pdq.PatientDemographics;
import gov.va.nvap.service.pdq.PatientDemographicsQuery;
import gov.va.nvap.service.pdq.PatientDemographicsResponse;
import gov.va.nvap.service.pdq.PdqService;
import gov.va.nvap.svc.consentmgmt.PIPInterface;
import gov.va.nvap.svc.consentmgmt.stub.adapter.announce.data.Announcement;
import gov.va.nvap.svc.consentmgmt.stub.adapter.announce.data.AnnouncementOrg;
import gov.va.nvap.svc.consentmgmt.stub.adapter.announce.data.AnnouncerInterface;
import gov.va.nvap.svc.consentmgmt.stub.adapter.announce.data.PatientAnnouncer;
import gov.va.nvap.svc.consentmgmt.stub.adapter.announce.jaxb.AnnouncePatientRequest;
import gov.va.nvap.svc.consentmgmt.stub.adapter.announce.jaxb.AuthorizedOrganizationType;
import gov.va.nvap.svc.consentmgmt.stub.adapter.announce.jaxb.PatientType;
import gov.va.nvap.svc.consentmgmt.stub.adapter.announce.jaxb.UserType;
import gov.va.nvap.svc.consentmgmt.stub.data.Organization;

import java.util.ArrayList;
import java.util.Collection;
import java.util.Date;
import java.util.List;

import javax.annotation.Resource;
import javax.ejb.ActivationConfigProperty;
import javax.ejb.EJB;
import javax.ejb.MessageDriven;
import javax.ejb.MessageDrivenContext;
import javax.ejb.TransactionAttribute;
import javax.ejb.TransactionAttributeType;
import javax.jms.Message;
import javax.jms.MessageListener;
import javax.jms.ObjectMessage;

/**
 * Message-Driven Bean implementation class for: BatchAnnouncer
 * 
 */
@MessageDriven(mappedName = "jms.vap.PatientAnnouncerBatchQueue", activationConfig = {
		@ActivationConfigProperty(propertyName = "acknowledgeMode", propertyValue = "Auto-acknowledge"),
		@ActivationConfigProperty(propertyName = "destinationType", propertyValue = "javax.jms.Queue") })
@TransactionAttribute(TransactionAttributeType.REQUIRES_NEW)
public class BatchAnnouncerMessageConsumer implements MessageListener {

	private static final int MAX_BATCH_SIZE = 1000;

	@EJB(name = "AnnouncerService", mappedName = "AnnouncerService")
	private AnnouncerInterface announcer;

	@EJB(name = "PatientAnnouncer", mappedName = "PatientAnnouncer")
	private PatientAnnouncer patientAnnouncer;
    
    @EJB(name = "PipService", mappedName = "PipService")
    private PIPInterface pip;
    
    @EJB(name = "PdqService", mappedName = "PdqService")
    private PdqService pdqService;

	private void announce(final String[] batchIds, final String userName,
			final String facility, final int maxBatchSize) throws Exception {
		try {
			for (final String batchId : batchIds) {
				Collection<Announcement> announcements = null;
				int firstResult = 0;
				do {
					announcements = this.announcer
							.getUnscheduledAnnouncementsByBatchId(batchId,
									maxBatchSize, firstResult);
					for (final Announcement announcement : announcements) {
                        // Construct the AnnouncePatientRequest object
						final AnnouncePatientRequest announcePatientRequest = 
                                new AnnouncementToAnnouncePatientRequest()
								.transform(announcement);
                        
						final List<String> organizations = new ArrayList<String>();
                        if (announcement.getTargetOrganization() != null) {
                            organizations.add(announcement.getTargetOrganization()
                                    .getOrgNumber());
                        } else {
                            // Add target organizations to request for ANNOUNCEMENT_ORGS
                            Collection<AnnouncementOrg> announceOrgs = 
                                    pip.getAnnouncementOrgsByAnnouncement(announcement);
                            for (final AnnouncementOrg announceOrg : announceOrgs) {
                                Organization organization = announceOrg.getTargetOrganization();
                                final AuthorizedOrganizationType authorizedOrganization = 
                                        new AuthorizedOrganizationType();
                                authorizedOrganization.setOrgNumber(organization
                                        .getOrgNumber());
                                authorizedOrganization.setOrgName(organization
                                        .getOrgName());
                                authorizedOrganization.setOrgOid(organization
                                        .getOrgOid());
                                announcePatientRequest.getAuthorizedOrganizations()
                                        .add(authorizedOrganization);
                            }
                        }

						final PatientDemographicsQuery query = new PatientDemographicsQuery();
						String icn = announcement.getPatientIen();
                        if (NullChecker.isNullOrEmpty(icn)) {
                            icn = announcement.getPatientConsentDir().getPatientIen();
                        }
						query.setPatientId(icn);
						try{
                            PatientDemographicsResponse patientDemographicsResponse = pdqService.getPatientDemographics(query); 
                            PatientDemographics demographics = patientDemographicsResponse.getPatientDemographics();
                            announcement.setFirstName(demographics.getFirstName());
                            announcement.setLastName(demographics.getLastName());
                            announcement.setSsn(demographics.getSsn());
                            announcement.setPatientIen(icn);
                        } catch(Exception e){
                            //fail silently because we have lots of patient data to try to get for this batch and just because we can't get one patient's data, doesn't mean we won't get others
                        }
                        
                        /*
						 * patientType.setLastName(demogprahics.getLastName());
						 * patientType
						 * .setGivenName(demogprahics.getFirstName());
						 * patientType.setSsn(demogprahics.getSsn());
						 */
						final PatientType patientType = new PatientType();
						patientType.setPatientId(icn);
						announcePatientRequest.setPatient(patientType);

						final UserType userType = new UserType();
						userType.setUserId(userName);
						userType.setFacilityNumber(facility);
						announcePatientRequest.setUser(userType);

						announcement.setScheduledTs(new Date());
						this.announcer.storeAnnouncement(announcement);
						// Call asynchronous announce
						// TODO: Fix Transaction Issues
						// Call announce Async. only after the store because the
						// other thread also updates the same record.
						this.patientAnnouncer
								.announceAsync(announcePatientRequest);
						
					}
					
					firstResult++;
					
				} while ((null != announcements)
						&& (announcements.size() == BatchAnnouncerMessageConsumer.MAX_BATCH_SIZE));
			}
		} catch (final Throwable t) {
			throw new Exception(t);
		}
	}

	/**
	 * @see MessageListener#onMessage(Message)
	 */
	@Override
	public void onMessage(final Message message) {
		// Unpack the message
		final ObjectMessage msg = (ObjectMessage) message;
		try {
			final BatchAnnounceType announcement = (BatchAnnounceType) msg
					.getObject();
			final int maxBatchSize = announcement.getMaxBatchSize();
			this.announce(
					announcement.getBatchIds(),
					announcement.getUser(),
					announcement.getFacility(),
					maxBatchSize < 1 ? BatchAnnouncerMessageConsumer.MAX_BATCH_SIZE
							: maxBatchSize);
		} catch (final Exception e) {
			throw new RuntimeException(e);
		}
	}

	@Resource
	public void setContext(final MessageDrivenContext context) {
	}

}
