package gov.va.nvap.server.endpoint.adapter.announce;

import gov.va.med.nhin.adapter.announce.AnnouncePatient;
import gov.va.med.nhin.adapter.announce.AnnouncePatient.Facilities;
import gov.va.med.nhin.adapter.announce.UserInfoType;
import gov.va.nvap.common.jaxb.JaxbUtil;
import gov.va.nvap.common.validation.Assert;
import gov.va.nvap.common.validation.NullChecker;
import gov.va.nvap.svc.consentmgmt.PIPInterface;
import gov.va.nvap.svc.consentmgmt.stub.adapter.announce.data.Announcement;
import gov.va.nvap.svc.consentmgmt.stub.adapter.announce.data.AnnouncerInterface;
import gov.va.nvap.svc.consentmgmt.stub.adapter.announce.data.PatientAnnouncer;
import gov.va.nvap.svc.consentmgmt.stub.adapter.announce.data.PatientAnnouncerException;
import gov.va.nvap.svc.consentmgmt.stub.adapter.announce.jaxb.AnnouncePatientRequest;
import gov.va.nvap.svc.consentmgmt.stub.adapter.announce.jaxb.AnnouncePatientResponse;
import gov.va.nvap.svc.consentmgmt.stub.adapter.announce.jaxb.AuthorizedOrganizationType;
import gov.va.nvap.svc.consentmgmt.stub.data.ConsentDirective;
import gov.va.nvap.svc.consentmgmt.stub.data.Organization;
import gov.va.nvap.svc.consentmgmt.stub.adapter.announce.data.AnnouncementOrg;

import java.util.ArrayList;
import java.util.Collection;
import java.util.Date;
import java.util.List;
import org.apache.commons.codec.digest.DigestUtils;
import org.apache.commons.lang.time.DateFormatUtils;

import org.springframework.beans.factory.annotation.Required;
import org.springframework.jms.core.JmsTemplate;
import org.springframework.ws.client.core.WebServiceTemplate;

/**
 * Announce patient on the NwHIN through the NwHIN Adapter.
 * 
 * @author Asha Amritraj
 * @author Irakli Kakushadze
 * @author Zack Peterson
 */
public class PatientAnnouncerEndpoint implements PatientAnnouncer {

	/**
	 * The announcer interface for creating batches of announcements and doing
	 * asynchronous announcements.
	 */
	AnnouncerInterface announcer;
	/**
	 * Spring JMS Template to send messages to the
	 * PatientAnnouncerForwardingQueue.
	 */
	JmsTemplate jmsTemplate;
	/**
	 * An interceptor to audit the announce.
	 * 
	 * PatientAnnouncerAuditInterceptor auditInterceptor;
	 */
	/**
	 * Convert to XML for JMS message.
	 */
	JaxbUtil patientAnnouncerJaxbHelper;
	/**
	 * The Adapter AnnouncePatient webservice template.
	 */
	WebServiceTemplate patientAnnounceWebserviceTemplate;
	/**
	 * The Policy information point, to be able to store announcements
	 * associated with a consent directive id.
	 */
	PIPInterface pipInterface;

	@Override
	public AnnouncePatientResponse announce(
			final AnnouncePatientRequest announcePatientRequest)
			throws PatientAnnouncerException {
		Assert.assertNotEmpty(announcePatientRequest, "AnnoucePatientRequest cannot be empty!");
		Assert.assertNotEmpty(announcePatientRequest.getAuthorizedOrganizations(), "Authorized organizations cannot be empty!");
		Assert.assertNotEmpty(announcePatientRequest.getPatient(), "Patient cannot be empty!");
		Assert.assertNotEmpty(announcePatientRequest.getPatient().getPatientId(), "Patient ID cannot be empty!");
		Assert.assertNotEmpty(announcePatientRequest.getUser(), "User cannot be empty!");
		Assert.assertNotEmpty(announcePatientRequest.getUser().getUserId(), "User ID cannot be empty!");
        
		// Announce for authorized organizations
		final AnnouncePatientResponse response = new AnnouncePatientResponse();

        final AnnouncePatient announcePatient = this
                .constructAnnouncePatientMessage(announcePatientRequest);
        /*
         * try { this.auditInterceptor.intercept(announcePatientRequest); }
         * catch (final InterceptorException ex) { throw new
         * PatientAnnouncerException(ex); }
         */

        /* Commented out for testing Batch Announcements - 10/01/2012 - Anand Srivastava */
        final gov.va.med.nhin.adapter.announce.AnnouncePatientResponse adapterResponse = 
                (gov.va.med.nhin.adapter.announce.AnnouncePatientResponse) this.patientAnnounceWebserviceTemplate
                                                                               .marshalSendAndReceive(announcePatient);
        response.setResult(adapterResponse.getReturn());
        /* 
        response.setResult(1);
        */
        // Get the associated announcement
        final Long announcementId = announcePatientRequest
                .getAnnouncementId();
        // Store the announcement
        final Announcement announcement = this.announcer
                .getAnnouncementById(announcementId);
        // Get associated orgs if using ANNOUNCEMENT_ORGS table
        Collection<AnnouncementOrg> announceOrgs = null;
		System.out.println("PatientAnnouncerEndpoint.announce "  + announcement);
		if (announcement.getTargetOrganization() == null) {
            announceOrgs = pipInterface.getAnnouncementOrgsByAnnouncement(announcement);
        }
        if (NullChecker.isNotEmpty(announcement)) {
            announcement.setCompletedTs(new Date());
            announcement.setScheduledTs(null);
            /*
             * The community response will be added response by the NwHIN
             * adapter

             * communityResponse.setPRPAIN201306UV02(resultFromNhin);
             * response.getCommunityResponse().add(communityResponse);
             */
            // TODO: Verify that the code below works for multiple assigning
            // authorities & Request adapter to send proper response. The
            // communityResponse from NwHIN could be error.

            // If the response is zero, then its success
            /* Commented out for testing Batch Announcements - 10/01/2012 - Anand Srivastava */
            announcement.setAnnouncementResult(Integer
                    .toString(adapterResponse.getReturn()));
            announcement.setHieTransactionId(announcePatient.getHieTransactionId());

            this.announcer.storeAnnouncement(announcement);
            
            // Set result for each associated organization
            // Set them all to the overall result for now
            if (announceOrgs != null) {
                for (AnnouncementOrg org : announceOrgs) {
                    org.setAnnounceResult(Integer.toString(adapterResponse.getReturn()));
                    pipInterface.updateAnnouncementOrg(org);
                }
            }
        }
		// Construct the AnnouncePatient root element for the webservice
		return response;
	}

	@Override
	public void announceAsync(
			final AnnouncePatientRequest announcePatientRequest)
			throws PatientAnnouncerException {
		Assert.assertNotEmpty(announcePatientRequest,
				"AnnoucePatientRequest cannot be empty!");
		Assert.assertNotEmpty(
				announcePatientRequest.getAuthorizedOrganizations(),
				"Authorized organizations cannot be empty!");
		Assert.assertNotEmpty(announcePatientRequest.getPatient(),
				"Patient cannot be empty!");
		Assert.assertNotEmpty(announcePatientRequest.getPatient()
				.getPatientId(), "Patient ID cannot be empty!");
		Assert.assertNotEmpty(announcePatientRequest.getUser(),
				"User cannot be empty!");
		Assert.assertNotEmpty(announcePatientRequest.getUser().getUserId(),
				"User ID cannot be empty!");
		// TODO: Write to ANNOUNCEMENTS table
		final AnnouncePatientMessageCreator messageCreator = new AnnouncePatientMessageCreator(
				announcePatientRequest, this.patientAnnouncerJaxbHelper);
		this.jmsTemplate.send(messageCreator);
	}

    @Override
    public AnnouncePatientResponse announceAll(
            AnnouncePatientRequest announcePatientRequest)
            throws PatientAnnouncerException {

        Assert.assertNotEmpty(announcePatientRequest, "AnnoucePatientRequest cannot be empty!");
        Assert.assertNotEmpty(announcePatientRequest.getPatient(), "Patient cannot be empty!");
        Assert.assertNotEmpty(announcePatientRequest.getPatient().getPatientId(), "Patient ID cannot be empty!");
        Assert.assertNotEmpty(announcePatientRequest.getUser(), "User cannot be empty!");
        Assert.assertNotEmpty(announcePatientRequest.getUser().getUserId(), "User ID cannot be empty!");

        final Date createdTs = new Date();

        final AnnouncePatientResponse response = new AnnouncePatientResponse();
        final AnnouncePatient announcePatient = this.constructAnnouncePatientMessage(announcePatientRequest);
        final gov.va.med.nhin.adapter.announce.AnnouncePatientResponse adapterResponse =
                (gov.va.med.nhin.adapter.announce.AnnouncePatientResponse) this.patientAnnounceWebserviceTemplate.marshalSendAndReceive(announcePatient);
        response.setResult(adapterResponse.getReturn());

        // create a new announcement and persist it in the table.
        try {
            // Get consent directives for this patient
            List<String> patientIens = new ArrayList<String>();
            String patientIen = announcePatientRequest.getPatient().getPatientId();
            patientIens.add(patientIen);
            Collection<ConsentDirective> allConsentDirectives = this.pipInterface.getConsentHistory(patientIens);
            // Use the latest directive to store the announcement
            if (allConsentDirectives != null && !allConsentDirectives.isEmpty()) {
                final ConsentDirective cd = allConsentDirectives.iterator().next();
                final Announcement announcement = new Announcement();
                announcement.setAnnouncementResult(Integer.toString(adapterResponse.getReturn()));
                announcement.setCompletedTs(new Date());
                announcement.setCreatedTs(createdTs);
                announcement.setPatientConsentDir(cd);
                announcement.setPatientIen(patientIen);
                announcement.setUserId(announcePatientRequest.getUser().getUserId());
                announcement.setHieTransactionId(announcePatient.getHieTransactionId());
                announcer.storeAnnouncement(announcement);
                announcePatientRequest.setAnnouncementId(announcement.getAnnouncementId());
                
                // Insert orgs for announcement
                for (final AuthorizedOrganizationType authorizedOrgType : announcePatientRequest.getAuthorizedOrganizations()) {
                    Organization org = pipInterface.getOrganizationByOid(authorizedOrgType.getOrgOid());

                    String result = String.valueOf(adapterResponse.getReturn());
                    
                    AnnouncementOrg announceOrg = new AnnouncementOrg();
                    announceOrg.setAnnouncement(announcement);
                    announceOrg.setTargetOrganization(org);
                    announceOrg.setAnnounceResult(result);
                    pipInterface.storeAnnouncementOrg(announceOrg);
                }
            }
        } catch (final Exception ex) {
            String msg = ex.getMessage();
            System.out.println(msg);
        }

        return response;
    }

    /**
     * Finds all announcements for a particular user.
     * 
     * @param userId User ID
     * @return List of objects of type Announcement found for this user
     */  
    @Override
    public List<Announcement> getAnnouncementsByUserId(String userId) {
        return this.announcer.getAnnouncementsByUserId(userId);
    }
    
    /**
     * Finds all announcements for a list of consent directive ID numbers
     * 
     * @param consentDirectiveIds list of consent directive ID numbers
     * @return List of objects of type Announcement found for this patient
     */
    @Override
    public List<Announcement> getAnnouncementsByConsentDirectiveId(List<Long> consentDirectiveIds) {
        return this.announcer.getAnnouncementsByConsentDirectiveId(consentDirectiveIds);
    }
    
	private AnnouncePatient constructAnnouncePatientMessage(final AnnouncePatientRequest announcePatientRequest) {
		// Get the list of assigning authorities
		final List<String> assigningAuthorities = new ArrayList<String>();
		for (final AuthorizedOrganizationType authorizedOrgType : announcePatientRequest.getAuthorizedOrganizations()) {
			assigningAuthorities.add(authorizedOrgType.getOrgNumber());
		}
		// Create the announce patient message
		final AnnouncePatient announcePatient = new AnnouncePatient();
        // If authorized organizations are in request add to AnnouncePatient
        if (assigningAuthorities.size() > 0) {
            AnnouncePatient.Facilities allFacilities = new AnnouncePatient.Facilities();
            allFacilities.getFacility().addAll(assigningAuthorities);
            announcePatient.setFacilities(allFacilities);
        }
		announcePatient.setPatientId(announcePatientRequest.getPatient()
				.getPatientId());
		final UserInfoType userInfoType = new UserInfoType();
		userInfoType.setUserId(announcePatientRequest.getUser().getUserId());
		userInfoType
				.setFullName(announcePatientRequest.getUser().getFullName());
		userInfoType.setAuthenticationMethod(announcePatientRequest.getUser()
				.getAuthenticationMethod());
		userInfoType.setRole(announcePatientRequest.getUser().getRole());
		announcePatient.setUserInfo(userInfoType);
        String hieTransactionId = DigestUtils.md5Hex(DateFormatUtils.format(new Date(), "yyyy-MM-dd-HH-mm-ss-SSS"));
        hieTransactionId = hieTransactionId.substring(24);
        announcePatient.setHieTransactionId(hieTransactionId);
		return announcePatient;
	}

	/*
	 * @Required public void setAuditInterceptor( final
	 * PatientAnnouncerAuditInterceptor auditInterceptor) {
	 * this.auditInterceptor = auditInterceptor; }
	 */

	@Required
	public void setAnnouncer(final AnnouncerInterface announcer) {
		this.announcer = announcer;
	}

	@Required
	public void setJmsTemplate(final JmsTemplate jmsTemplate) {
		this.jmsTemplate = jmsTemplate;
	}

	@Required
	public void setPatientAnnouncerJaxbHelper(
			final JaxbUtil patientAnnouncerJaxbHelper) {
		this.patientAnnouncerJaxbHelper = patientAnnouncerJaxbHelper;
	}

	@Required
	public void setPatientAnnounceWebserviceTemplate(
			final WebServiceTemplate patientAnnounceWebserviceTemplate) {
		this.patientAnnounceWebserviceTemplate = patientAnnounceWebserviceTemplate;
	}

	@Required
	public void setPipInterface(final PIPInterface pipInterface) {
		this.pipInterface = pipInterface;
	}
}
