package gov.va.med.vler.nwhin.vap.migration;

import gov.va.med.nhin.vap.common.validation.NullChecker;
import gov.va.med.nhin.vap.pip.data.ConsentDirective;
import gov.va.med.nhin.vap.pip.data.PatientDocument;
import gov.va.med.nhin.vap.pip.data.PatientDocumentType;

import java.util.Collection;

import javax.persistence.EntityManager;
import javax.persistence.PersistenceContext;
import javax.persistence.Query;

import org.apache.commons.cli.CommandLine;
import org.apache.commons.cli.CommandLineParser;
import org.apache.commons.cli.HelpFormatter;
import org.apache.commons.cli.Option;
import org.apache.commons.cli.OptionBuilder;
import org.apache.commons.cli.Options;
import org.apache.commons.cli.ParseException;
import org.apache.commons.cli.PosixParser;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.springframework.beans.factory.InitializingBean;
import org.springframework.beans.factory.annotation.Required;
import org.springframework.context.support.ClassPathXmlApplicationContext;

/**
 * Create patient document from patient consent directive table.
 *
 * @author Asha Amritraj
 *
 */

public class PrivacyConsentDirectiveDocumentMigrator implements
		InitializingBean {

	private static final Log LOG = LogFactory
			.getLog(PrivacyConsentDirectiveDocumentMigrator.class);

	/**
	 * Main migration class - takes in a argument of start and end sequences.
	 *
	 * @param args
	 */
	public static void main(final String[] args) {

		String migrationHome = System.getenv("MIGRATION_HOME");
		if(NullChecker.isEmpty(migrationHome)) {
			System.out.println("MIGRATION_HOME environment variable must be set.");
			return;
		}

		OptionBuilder.withArgName("Starting Sequence");
		OptionBuilder.hasArg();
		OptionBuilder.withDescription("The beginning record to migrate");
		OptionBuilder.isRequired(true);
		// Add the arguments needed
		final Option startSequence = OptionBuilder.create("startSeq");
		OptionBuilder.withArgName("Ending Sequence");
		OptionBuilder.hasArg();
		OptionBuilder.withDescription("The last record to migrate");
		final Option endSequence = OptionBuilder.create("endSeq");
		// Create a group - so that start and end are required if one of them is
		// used
		// Add options
		final Options options = new Options();
		options.addOption(startSequence);
		options.addOption(endSequence);
		final CommandLineParser parser = new PosixParser();

		long startSeq = -1;
		long endSeq = -1;

		try {

			// parse the command line arguments
			final CommandLine line = parser.parse(options, args);
			if (line.hasOption("startSeq")) {
				final String startSeqStr = line.getOptionValue("startSeq");
				startSeq = Long.parseLong(startSeqStr);
			}
			if (line.hasOption("endSeq")) {
				final String endSeqStr = line.getOptionValue("endSeq");
				endSeq = Long.parseLong(endSeqStr);
			}

			if (startSeq <= 0) {
				throw new ParseException(
						"Starting Sequence is required and should be greater than 0");
			}

		} catch (final ParseException exp) {
			// oops, something went wrong
			PrivacyConsentDirectiveDocumentMigrator.LOG
					.error("Parsing failed.  Reason: " + exp.getMessage());
			// automatically generate the help statement
			final HelpFormatter formatter = new HelpFormatter();
			formatter
					.printHelp(
							"java -jar nhin_vap_migration-1.0.0.0.jar",
							"\nVAP Migration Utility",
							options,
							"\nExample: \n "
									+ "java -jar nhin_vap_migration-1.0.0.0.jar -startSeq 100 -endSeq 200 "
									+ "\n "
									+ "java -jar nhin_vap_migration-1.0.0.0.jar -startSeq 100 ");
			return;
		}

		PrivacyConsentDirectiveDocumentMigrator.LOG
				.info("Initializing Migration Utiltiy with arguments Start Sequence: "
						+ startSeq + " End Seqeuence: " + endSeq);

		final ClassPathXmlApplicationContext appContext = new ClassPathXmlApplicationContext(
				new String[] { "applicationContext.xml" });

		final PrivacyConsentDirectiveDocumentMigrator migrator = appContext
				.getBean("privacyConsentDirectiveDocumentMigrator",
						PrivacyConsentDirectiveDocumentMigrator.class);
		migrator.migrate(startSeq, endSeq);

		PrivacyConsentDirectiveDocumentMigrator.LOG.info("\nDONE!");
	}

	@PersistenceContext(name = "nwhin_vap_migration")
	private EntityManager entityManager;
	private String facilityId;

	private PatientDocumentType optInDocumentType;
	private PatientDocumentType optOutDocumentType;

	private PatientDocumentDAO patientDocumentDAO;

	private PrivacyConsentDirectiveDocumentCreator privacyConsentDirectiveDocumentCreator;

	private String userId;

	@Override
	public void afterPropertiesSet() throws Exception {
		// Load document types
		final Query patientDocumentTypesQuery = this.entityManager
				.createNamedQuery("PatientDocumentType.findAll");
		final Collection<PatientDocumentType> patientDocumentTypes = patientDocumentTypesQuery
				.getResultList();

		for (final PatientDocumentType documentType : patientDocumentTypes) {
			if (documentType.getDocumentName().equals("OPT-IN")) {
				this.optInDocumentType = documentType;
			} else if (documentType.getDocumentName().equals("OPT-OUT")) {
				this.optOutDocumentType = documentType;
			}
		}

		PrivacyConsentDirectiveDocumentMigrator.LOG
				.info("Loaded Patient Document Types: "
						+ this.optInDocumentType.getDocumentName() + " and "
						+ this.optOutDocumentType.getDocumentName());

	}

	private Collection<ConsentDirective> loadConsentDirectives(
			final long startSeq, final long endSeq) {
		String query = "SELECT c FROM ConsentDirective c";
		if (startSeq > 0) {
			query += " WHERE c.consentDirId >= " + startSeq;
		}
		if (endSeq > 0) {
			query += " AND c.consentDirId <= " + endSeq;
		}

		PrivacyConsentDirectiveDocumentMigrator.LOG
				.info("Loading Consent Directives - Executing query: " + query);

		final Query consentDirectiveQuery = this.entityManager
				.createQuery(query);
		final Collection<ConsentDirective> consentDirectives = consentDirectiveQuery
				.getResultList();
		return consentDirectives;
	}

	public void migrate(final long startSeq, final long endSeq) {
		// Load Consent Directives
		final Collection<ConsentDirective> consentDirectives = this
				.loadConsentDirectives(startSeq, endSeq);
		// Return if empty
		if (NullChecker.isEmpty(consentDirectives)) {
			PrivacyConsentDirectiveDocumentMigrator.LOG
					.info("Query returned zero resutls. No records to process!");
			return;
		}
		// Process
		final int recordSize = consentDirectives.size();
		PrivacyConsentDirectiveDocumentMigrator.LOG.info("Fetched "
				+ recordSize + " Consent Directives");

		int i = 0;
		int success = 0;
		int insertedOptins = 0;
		int insertedOptouts = 0;
		// Iterate through all consent directives and create documents
		for (final ConsentDirective cd : consentDirectives) {
			try {
				PrivacyConsentDirectiveDocumentMigrator.LOG
						.info("====================== Processing Record "
								+ cd.getConsentDirId() + ": " + ++i + " of "
								+ recordSize + " =======================");
				this.printConsentDirective(cd);
				PrivacyConsentDirectiveDocumentMigrator.LOG
						.info("Querying for OPT-IN documents ...");

				Query patientDocumentQuery = this.entityManager
						.createNamedQuery("PatientDocument.findByConsentDirAndType");
				patientDocumentQuery.setParameter("consentDirective", cd);
				patientDocumentQuery.setParameter("patientDocumentType",
						this.optInDocumentType);
				Collection<PatientDocument> patientDocuments = patientDocumentQuery
						.getResultList();

				if (NullChecker.isEmpty(patientDocuments)) {
					PrivacyConsentDirectiveDocumentMigrator.LOG
							.info("Found no OPT-IN Documents - Creating one...");
					// create OPT-IN
					final PatientDocument patientDocument = new PatientDocument();
					patientDocument.setConsentDirective(cd);
					patientDocument
							.setPatientDocumentType(this.optInDocumentType);
					final String cdaR2Xml = this.privacyConsentDirectiveDocumentCreator
							.optIn(cd.getPatientIen(), null, null, null,
									this.userId, this.facilityId,
									cd.getOptinDate());
					patientDocument.setDocument(cdaR2Xml);
					PrivacyConsentDirectiveDocumentMigrator.LOG
							.info("CDA R2 XML:\n" + cdaR2Xml);
					this.patientDocumentDAO.storePatientDocument(
							patientDocument, cd.getConsentDirId(), i,
							recordSize);
					insertedOptins++;
				} else {
					PrivacyConsentDirectiveDocumentMigrator.LOG
							.info("Found OPT-IN Document already in the database - Skip creating one...");
				}

				if (NullChecker.isNotEmpty(cd.getOptoutDate())) {
					PrivacyConsentDirectiveDocumentMigrator.LOG
							.info("Patient was opted-out, so checking for OPT-OUT document...");
					patientDocumentQuery = this.entityManager
							.createNamedQuery("PatientDocument.findByConsentDirAndType");
					patientDocumentQuery.setParameter("consentDirective", cd);
					patientDocumentQuery.setParameter("patientDocumentType",
							this.optOutDocumentType);
					patientDocuments = patientDocumentQuery.getResultList();
					if (NullChecker.isEmpty(patientDocuments)) {
						PrivacyConsentDirectiveDocumentMigrator.LOG
								.info("Found no OPT-OUT Documents - Creating one...");
						// create OPT-OUT
						final PatientDocument patientDocument = new PatientDocument();
						patientDocument.setConsentDirective(cd);
						patientDocument
								.setPatientDocumentType(this.optOutDocumentType);
						final String cdaR2Xml = this.privacyConsentDirectiveDocumentCreator
								.optOut(cd.getConsentDirId().toString(),
										cd.getPatientIen(), null, null, null,
										this.userId, this.facilityId,
										cd.getOptoutDate());
						PrivacyConsentDirectiveDocumentMigrator.LOG
								.info("CDA R2 XML:\n" + cdaR2Xml);
						patientDocument.setDocument(cdaR2Xml);
						this.patientDocumentDAO.storePatientDocument(
								patientDocument, cd.getConsentDirId(), i,
								recordSize);
						insertedOptouts++;
					} else {
						PrivacyConsentDirectiveDocumentMigrator.LOG
								.info("Found OPT-OUT Document already in the database - Skip creating one...");
					}
				}
				success++;
			} catch (final Exception ex) {
				PrivacyConsentDirectiveDocumentMigrator.LOG
						.error("Error Occured when Processing Record "
								+ cd.getConsentDirId() + ": " + i + " of "
								+ +recordSize);
				PrivacyConsentDirectiveDocumentMigrator.LOG
						.error("Exception Message: " + ex.getMessage());
				ex.printStackTrace();
			}
		}
		PrivacyConsentDirectiveDocumentMigrator.LOG
				.info("\n******************************************************\n"
						+ "Successfully processed "
						+ success
						+ " records of "
						+ recordSize
						+ "\n"
						+ "Inserted Opt-ins: "
						+ insertedOptins
						+ "\n"
						+ "Inserted Opt-outs: "
						+ insertedOptouts
						+ "\n******************************************************");
	}

	private void printConsentDirective(final ConsentDirective cd) {
		PrivacyConsentDirectiveDocumentMigrator.LOG
				.info("Consent Directive ID: " + cd.getConsentDirId());
		PrivacyConsentDirectiveDocumentMigrator.LOG.info("Consent Type: "
				+ (cd.getConsentType() != null ? cd.getConsentType().getName()
						: "Unknown"));
		PrivacyConsentDirectiveDocumentMigrator.LOG
				.info("Patient IEN: "
						+ (cd.getPatientIen() != null ? cd.getPatientIen()
								: "Unknown"));
		PrivacyConsentDirectiveDocumentMigrator.LOG
				.info("Created Date: "
						+ (cd.getOptinTS() != null ? cd.getOptinTS()
								: "Unknown"));
		PrivacyConsentDirectiveDocumentMigrator.LOG.info("Opt-In Date: "
				+ (cd.getOptinDate() != null ? cd.getOptinDate() : "Unknown"));
		PrivacyConsentDirectiveDocumentMigrator.LOG
				.info("Opt-Out Date: "
						+ (cd.getOptoutDate() != null ? cd.getOptoutDate()
								: "Unknown"));

		PrivacyConsentDirectiveDocumentMigrator.LOG.info("Expiration Date: "
				+ (cd.getExpirationDate() != null ? cd.getExpirationDate()
						: "Unknown"));

		PrivacyConsentDirectiveDocumentMigrator.LOG
				.info("Optout Reason: "
						+ (cd.getOptoutReason() != null ? cd.getOptoutReason()
								: "N/A"));
	}

	@Required
	public void setFacilityId(final String facilityId) {
		this.facilityId = facilityId;
	}

	@Required
	public void setPatientDocumentDAO(
			final PatientDocumentDAO patientDocumentDAO) {
		this.patientDocumentDAO = patientDocumentDAO;
	}

	public void setPrivacyConsentDirectiveDocumentCreator(
			final PrivacyConsentDirectiveDocumentCreator privacyConsentDirectiveDocumentCreator) {
		this.privacyConsentDirectiveDocumentCreator = privacyConsentDirectiveDocumentCreator;
	}

	@Required
	public void setUserId(final String userId) {
		this.userId = userId;
	}
}
