package gov.va.med.vler.nwhin.vap.migration;

import gov.va.med.nhin.vap.common.date.hl7.HL7DateUtil;
import gov.va.med.nhin.vap.common.jaxb.JaxbUtil;
import gov.va.med.nhin.vap.common.transformer.Transformer;
import gov.va.med.nhin.vap.common.transformer.TransformerException;
import gov.va.med.nhin.vap.common.transformer.xml.XMLToString;
import gov.va.med.nhin.vap.common.uuid.UUIDUtil;
import gov.va.med.nhin.vap.common.validation.Assert;
import gov.va.med.nhin.vap.common.validation.NullChecker;
import gov.va.med.nhin.vap.privacy.data.ConsentDirectiveData;
import gov.va.med.nhin.vap.privacy.data.ScannedConsentDirectiveDocumentBase64Text;
import gov.va.med.nhin.vap.service.pdq.PatientDemographics;

import java.util.Calendar;
import java.util.Date;
import java.util.List;

import javax.xml.bind.JAXBException;

import org.apache.commons.fileupload.FileItem;
import org.springframework.beans.factory.annotation.Required;
import org.w3c.dom.Document;

public class PrivacyConsentDirectiveDocumentCreator {

	/**
	 * To convert from ConsentDirectiveData to XML for transformation into the
	 * CDA XML.
	 */
	private JaxbUtil dataJaxbHelper;
	/**
	 * Transformer to transform from the value object to CDA R2 XML document.
	 */
	private Transformer<Document, Document> dataToConsentDirectiveDocument;
	/**
	 * Home community.
	 */
	private String homeCommunityId;
	/**
	 * XML to String Transformer.
	 */
	private XMLToString xmlToString;

	/**
	 * Construct the patient demographics data for the CDA R2 XML.
	 */
	private ConsentDirectiveData constructConsentDirectiveDocumentDemographics(
			final ConsentDirectiveData data, final String icn,
			final PatientDemographics demographics,
			final List<FileItem> fileItems) {
		Assert.assertNotEmpty(data, "ConsentDirectiveData cannot be NULL!");

		// Set the demographics information
		data.setIcn(icn);
		if (NullChecker.isNotEmpty(demographics)) {
			data.setPatientRoleCity(demographics.getResidenceCity());
			data.setPatientRoleState(demographics.getResidenceState());
			data.setPatientRoleGivenName(demographics.getFirstName());
			data.setPatientRoleFamilyName(demographics.getLastName());
			data.setPatientRoleFamilyNameAlias(demographics.getAlias1());
			data.setPatientRoleEthnicGroupCodeDisplayText(demographics
					.getEthnicityDescription());
			data.setPatientRoleSsn(demographics.getSsn());
			data.setPatientRoleGenderCode(demographics.getGender());
			data.setPatientRoleGenderDisplayText(demographics
					.getGenderDescription());
			data.setPatientRoleMaritalStatusCode(demographics
					.getMaritalStatus());
			data.setPatientRoleMaritalStatusDisplayText(demographics
					.getMaritalStatusDescription());
			data.setPatientRolePostalCode(demographics.getResidenceZip4());
			data.setPatientRolePrefix(demographics.getPrefix());
			data.setPatientRoleStreetAddressLine(demographics
					.getStreetAddressLine1());
			data.setPatientRoleSuffix(demographics.getSuffix());
			data.setPatientRoleTelecom(demographics.getResidencePhoneNumber());
			data.setPatientRoleProviderOrganizationName(demographics
					.getFacilityName());
			data.setPatientRoleProviderOrganizationNumber(demographics
					.getFacilityNumber());
			data.setPatientRoleProviderOrganizationOid(this.homeCommunityId);
			if (NullChecker.isNotEmpty(demographics.getDob())) {
				try {
					// Need to use the HL7 date, because the CDA R2 needs it
					data.setPatientRoleDob(HL7DateUtil
							.yyyyMMddhhmmssZ(demographics.getDob()));
				} catch (final java.text.ParseException ex) {
					throw new RuntimeException(ex);
				}
			}
		}
		// For each file item, create the media type
		if (NullChecker.isNotEmpty(fileItems)) {
			for (final FileItem fileItem : fileItems) {
				if (fileItem.getSize() > 0) {
					final ScannedConsentDirectiveDocumentBase64Text scannedConsentDirectiveDocumentBase64Text = new ScannedConsentDirectiveDocumentBase64Text();

					final byte[] bytes = fileItem.get();
					// JAXB automatically decodes the document when making it
					// into XML
					// String encodedString = new sun.misc.BASE64Encoder()
					// .encode(bytes);
					// byte[] encodedBytes = encodedString.getBytes();
					final String contentType = fileItem.getContentType();
					// Set the type and representation
					scannedConsentDirectiveDocumentBase64Text
							.setMediaType(contentType);
					scannedConsentDirectiveDocumentBase64Text
							.setRepresentation("B64");
					scannedConsentDirectiveDocumentBase64Text
							.setDocument(bytes);
					// add the document to ConsentDirectiveData
					data.getScannedConsentDirectiveDocumentBase64Text().add(
							scannedConsentDirectiveDocumentBase64Text);
				}
			}
		}
		// Set a unique id per document
		data.setDocId(UUIDUtil.generateUUID());
		// return response
		return data;

	}

	/**
	 * Convert from the ConsentDirectiveData to the CDA R2 XML Privacy consent
	 * directive document and then convert that to string.
	 */
	private String makeConsentDirectiveDocumentString(
			final ConsentDirectiveData data) {
		try {
			// Convert the ConsentDirectiveData to XML document
			final Document consentDirectiveDataDoc = this.dataJaxbHelper
					.marshal(data);
			// Convert ConsentDirectiveData XML to CDA R2 XML
			final Document consentDirectiveDocument = this.dataToConsentDirectiveDocument
					.transform(consentDirectiveDataDoc);
			// Convert CDA R2 XML to string
			final String consentDirectiveDocumentString = this.xmlToString
					.transform(consentDirectiveDocument);
			return consentDirectiveDocumentString;

		} catch (final TransformerException ex) {
			throw new RuntimeException(ex);
		} catch (final JAXBException ex) {
			throw new RuntimeException(ex);
		}
	}

	/**
	 * Opt-in Patients!
	 */
	public String optIn(final String icn, final String[] organizationNumbers,
			final PatientDemographics demographics,
			final List<FileItem> fileItems, final String userId,
			final String userFacilityStationId, final Date signatureDate) {
		Assert.assertNotEmpty(icn, "ICN cannot be empty!");
		Assert.assertNotEmpty(signatureDate, "Signature Date cannot be empty!");
		// Get the excluded organizations
		if (NullChecker.isNotEmpty(organizationNumbers)) {
			// Handle Exclusions - TODO
		}
		// Create a ConsentDirectiveData object
		ConsentDirectiveData data = new ConsentDirectiveData();
		data = this.constructConsentDirectiveDocumentDemographics(data, icn,
				demographics, fileItems);
		// Set author
		data.setAuthorPersonOid(userId);
		// data.setAuthorPersonOrgOid(this.homeCommunity.getOrgOid());
		data.setAuthorPersonOrgOid(userFacilityStationId);
		// Set the status to active
		data.setComponentStatusCode("active");
		try {
			final String signatureDateString = HL7DateUtil
					.yyyyMMddhhmmssZ(signatureDate);
			// Set the effective date
			data.setEffectiveDateTime(HL7DateUtil.yyyyMMddhhmmssZ(new Date()));
			// Create the begin and end date
			data.setDocumentationBeginTime(signatureDateString);
			final Calendar dateCal = Calendar.getInstance();
			dateCal.setTime(signatureDate);
			// End date should be 10 years after begin date
			dateCal.add(Calendar.YEAR, 10);
			final Date expirationDate = dateCal.getTime();
			data.setDocumentationEndTime(HL7DateUtil
					.yyyyMMddhhmmssZ(expirationDate));
		} catch (final java.text.ParseException ex) {
			throw new RuntimeException(ex);
		}
		// Make the attachment into bytes
		final String consentDirectiveDocumentBytes = this
				.makeConsentDirectiveDocumentString(data);
		return consentDirectiveDocumentBytes;
	}

	/**
	 * Opt out the patient.
	 */
	public String optOut(final String consentDirectiveId, final String icn,
			final PatientDemographics demographics,
			final List<FileItem> fileItems, final String optoutReason,
			final String userId, final String userFacilityStationId,
			final Date signatureDate) {
		Assert.assertNotEmpty(icn, "ICN cannot be empty!");
		Assert.assertNotEmpty(userId, "User id cannot be empty!");
		Assert.assertNotEmpty(signatureDate, "Signature Date cannot be empty!");
		// Create a consent directive data object
		ConsentDirectiveData data = new ConsentDirectiveData();
		// Create Demographics
		data = this.constructConsentDirectiveDocumentDemographics(data, icn,
				demographics, fileItems);
		// Set the status code to aborted
		data.setComponentStatusCode("aborted");
		try {
			final String signatureDateString = HL7DateUtil
					.yyyyMMddhhmmssZ(signatureDate);
			// Set the effective date
			data.setEffectiveDateTime(HL7DateUtil.yyyyMMddhhmmssZ(new Date()));
			// Create the begin and end date
			data.setDocumentationBeginTime(signatureDateString);
			// Start and end time are required
			data.setDocumentationEndTime(signatureDateString);
		} catch (final java.text.ParseException ex) {
			throw new RuntimeException(ex);
		}
		// Set the consent directive
		data.setPreviousConsentDirectiveId(consentDirectiveId);
		data.setOptoutReason(optoutReason);
		data.setAuthorPersonOid(userId);
		data.setAuthorPersonOrgOid(userFacilityStationId);
		// Convert the PDF into byte string
		final String consentDirectiveDocumentBytes = this
				.makeConsentDirectiveDocumentString(data);

		return consentDirectiveDocumentBytes;
	}

	@Required
	public void setDataJaxbHelper(final JaxbUtil dataJaxbHelper) {
		this.dataJaxbHelper = dataJaxbHelper;
	}

	@Required
	public void setDataToConsentDirectiveDocument(
			final Transformer<Document, Document> dataToConsentDirectiveDocument) {
		this.dataToConsentDirectiveDocument = dataToConsentDirectiveDocument;
	}

	@Required
	public void setHomeCommunityId(final String homeCommunityId) {
		this.homeCommunityId = homeCommunityId;
	}

	@Required
	public void setXmlToString(final XMLToString xmlToString) {
		this.xmlToString = xmlToString;
	}
}
