package gov.va.nvap.common.transformer.xsl;

import gov.va.nvap.common.validation.Assert;
import gov.va.nvap.common.validation.NullChecker;
import gov.va.nvap.common.xsl.TransformerFactory;

import java.io.IOException;
import java.util.HashMap;
import java.util.Iterator;
import java.util.Map;

import javax.xml.transform.ErrorListener;
import javax.xml.transform.Result;
import javax.xml.transform.Source;
import javax.xml.transform.Templates;
import javax.xml.transform.TransformerException;

import org.springframework.beans.factory.annotation.Required;
import org.springframework.context.ApplicationContext;
import org.springframework.context.ApplicationContextAware;
import org.springframework.core.io.Resource;
import org.w3c.dom.DOMException;

/**
 * The abstract class for transforming using XSL.
 * 
 * @author Asha Amritraj
 */
public abstract class AbstractXSLTransformer implements
		ApplicationContextAware, XSLTransformer {

	public static void throwTransformerException(final String message)
			throws TransformerException {
		throw new TransformerException(message);
	}

	private ErrorListener errorListener = null;
	private final Map<String, Object> parameters = new HashMap<String, Object>();
	private TransformerFactory transformerFactory;

	public TransformerFactory getTransformerFactory() {
		return this.transformerFactory;
	}

	private void setAllParameters(final javax.xml.transform.Transformer trans,
			final Map<String, Object> theParameters) {
		if (NullChecker.isNotEmpty(theParameters)) {
			final Iterator<Map.Entry<String, Object>> ci = theParameters
					.entrySet().iterator();
			while (ci.hasNext()) {
				final Map.Entry<String, Object> entry = ci.next();
				trans.setParameter(entry.getKey(), entry.getValue());
			}
		}
	}

	@Override
	public final void setApplicationContext(
			final ApplicationContext applicationContext) {
		this.parameters.put("applicationContext", applicationContext);
	}

	public final void setErrorListener(final ErrorListener listener) {
		this.errorListener = listener;
	}

	public final void setParameter(final String name, final Object value) {
		this.parameters.put(name, value);
	}

	public final void setParameters(
			final Map<String, ? extends Object> theParameters) {
		this.parameters.putAll(theParameters);
	}

	@Required
	public void setTransformerFactory(
			final TransformerFactory transformerFactory) {
		this.transformerFactory = transformerFactory;
	}

	protected final Result transform(final Resource resource,
			final Source sourceMessage) throws TransformerException {
		return this.transform(resource, sourceMessage, null);
	}

	protected final Result transform(final Resource resource,
			final Source sourceMessage, final Result targetMessage)
			throws TransformerException {
		return this.transform(resource, sourceMessage, targetMessage, null);
	}

	protected final Result transform(final Resource resource,
			final Source sourceMessage, final Result targetMessage,
			final Map<String, Object> dynamicParameters)
			throws TransformerException {
		Assert.assertNotEmpty(resource, "Resource cannot be null!");
		try {

			final Templates templates = this.transformerFactory
					.newTemplates(resource.getURL());
			final javax.xml.transform.Transformer trans = templates
					.newTransformer();
			if (this.errorListener != null) {
				trans.setErrorListener(this.errorListener);
			}
			this.setAllParameters(trans, this.parameters);
			this.setAllParameters(trans, dynamicParameters);
			trans.transform(sourceMessage, targetMessage);
			return targetMessage;
		} catch (final javax.xml.transform.TransformerException ex) {
			throw new TransformerException(ex);
		} catch (final IOException ex) {
			throw new TransformerException(ex);
		} catch (DOMException ex) {
            throw new TransformerException(ex);
        }
	}
}
