package gov.va.nvap.common.transceiver.client;

import gov.va.nvap.common.validation.Assert;
import gov.va.nvap.common.validation.NullChecker;

import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.net.InetSocketAddress;
import java.net.Socket;
import java.net.SocketTimeoutException;
import java.net.UnknownHostException;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.springframework.beans.factory.annotation.Required;

/**
 * SocketTransceiver.
 *
 * @author Asha Amritraj
 */
public abstract class SocketTransceiver implements Transceiver<String>
{
    private static final int DEFAULT_BUFFER = 512;
    private static final int DEFAULT_TIMEOUT = 10000;
    private static final Log LOG = LogFactory.getLog(SocketTransceiver.class);
    private long endTime;
    private String host;
    private int port;
    private int connectTimeout;
	private int readTimeout;
    private int retries;

    protected SocketTransceiver()
    {
    }

    protected final Socket connect(int retriesUsed) {
		final long startTime = System.currentTimeMillis();
		this.setEndTime(startTime + this.readTimeout);
        final Socket socket = new Socket();
        try {
            final InetSocketAddress isoc = new InetSocketAddress(this.host, this.port);
            socket.setSoTimeout(this.readTimeout);
			socket.connect(isoc, this.connectTimeout);
        }
        catch (final UnknownHostException e) {
            final String msg = "Unable to contact remote host at address "
                    + this.host + " and port " + this.port;
            throw new RuntimeException(msg, e);
        }
        // catch SocketTimeoutException for retries while connecting
        catch(final SocketTimeoutException e) {
            if(retriesUsed < retries) {
                this.connect(retriesUsed+1);
            }
            else {
                final String msg = "Exhausted " + retriesUsed + " retries, unable to contact remote host at address "
                        + this.host + " and port " + this.port;
                LOG.warn(msg);
                throw new RuntimeException(msg, e);
            }
        }
        //catches other IOExceptions, like ConnectionException (Connection refused)
        catch (final IOException e) {
            final String msg = "Unable to contact remote host at address "
                    + this.host + " and port " + this.port;
            throw new RuntimeException(msg, e);
        }
        return socket;
    }

    private void disconnect(final Socket socket)
    {
        if (!NullChecker.isEmpty(socket)) {
            try {
                if (socket.isConnected() && !socket.isClosed()) {
                    final InputStream is = socket.getInputStream();
                    final OutputStream os = socket.getOutputStream();

                    if (NullChecker.isNotEmpty(is)) {
                        is.close();
                    }
                    if (NullChecker.isNotEmpty(os)) {
                        os.close();
                    }
                    if (NullChecker.isNotEmpty(socket)) {
                        socket.close();
                    }
                }
            }
            catch (final IOException e) {
                // Do Nothing.
                SocketTransceiver.LOG.warn("Exception in disconnect", e);
            }
        }
    }

    protected abstract String formatIncomingMessage(String payload);

    protected abstract String formatOutgoingMessage(String payload);

    protected abstract String getEndMarker();

    protected abstract int getEndMarkerPosition();

    private InputStream getInputStream(final Socket socket) throws IOException
    {
        Assert.assertNotEmpty(socket, "Socket cannot be null!");
        Assert.assertFalse(socket.isClosed(), "Trancseiver is not connected.");
        return socket.getInputStream();
    }

    private OutputStream getOutputStream(final Socket socket) throws IOException
    {
        Assert.assertNotEmpty(socket, "Socket cannot be null!");
        Assert.assertFalse(socket.isClosed(), "Trancseiver is not connected.");
        return socket.getOutputStream();
    }

    private String receive(final Socket socket, final String endMarker) throws IOException, SocketTimeoutException
    {
        Assert.assertNotEmpty(socket, "Socket cannot be null!");
        Assert.assertFalse(socket.isClosed(), "Trancseiver is not connected.");

        final StringBuffer buffer = new StringBuffer();
        // final InputStreamReader reader = new InputStreamReader(this
        // .getInputStream(socket));
        int i = 0;
        final byte[] arr = new byte[SocketTransceiver.DEFAULT_BUFFER];
        InputStream in = getInputStream(socket);
            while ((i != -1)
                    && (buffer.indexOf(endMarker) < this.getEndMarkerPosition())) {
                i = in.read(arr, 0, arr.length);
                if (i > 0) {
                    for (int j = 0; j < i; j++) {
                        buffer.append((char) arr[j]);
                    }
                }
                Assert.assertFalse(System.currentTimeMillis() > this.endTime,
                        "Receive operation timed out.");
            }
        return buffer.toString();
    }

    private void send(final Socket socket, final String data) throws IOException
    {
        Assert.assertNotEmpty(socket, "Socket cannot be null!");
        Assert.assertFalse(socket.isClosed(), "Trancseiver is not connected.");
        OutputStream out = getOutputStream(socket);
        out.write(data.getBytes());
        out.flush();
        Assert.assertTrue(System.currentTimeMillis() <= this.endTime,
                "Send operation timed out.");
    }

    private void setEndTime(final long value)
    {
        this.endTime = value;
    }

    @Required
    public final void setHost(final String theHost)
    {
        this.host = theHost;
    }

    @Required
    public final void setPort(final int thePort)
    {
        this.port = thePort;
    }

    @Required
	public final void setReadTimeout(final int theReadTimeout) {
		this.readTimeout = theReadTimeout;
	}

        @Required
	public final void setConnectTimeout(final int theConnectTimeout) {
		this.connectTimeout = theConnectTimeout;
	}

        @Required
	public final void setRetries(final int theRetries) {
		this.retries = theRetries;
	}

    @Override
    public final String transceive(final String payload)
    {
        String ret = null;
        Socket socket = null;
        try {
            // Connect
            socket = this.connect(0);
            Assert.assertNotEmpty(socket, "Socket cannot be null!");
            // Send - format outgoing
            this.send(socket, this.formatOutgoingMessage(payload));
            // Receive and format incoming
            ret = this.formatIncomingMessage(this.receive(socket, this.getEndMarker()));
        }
        catch (IOException ioe) {

        }
        finally {
            this.disconnect(socket);
        }
        return ret;
    }
}
