package gov.va.nvap.common.splitter.regex;

import gov.va.nvap.common.splitter.Splitter;
import gov.va.nvap.common.splitter.SplitterException;
import gov.va.nvap.common.validation.NullChecker;
import gov.va.nvap.common.date.DateUtil;

import java.util.ArrayList;
import java.util.Date;
import java.util.HashMap;
import java.util.Map;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

import org.springframework.beans.factory.annotation.Required;

/**
 * Split the string into key/value pair based on position.
 * 
 * @author Asha Amritraj
 */
public class RegExToMapSplitter implements
		Splitter<String, Map<String, String>> {

	private Pattern expression;
	private Pattern facilityIDExpression;
	private Pattern correlationDateExpression;
	private int matchGroupKeyPosition;
	private int matchGroupValuePosition;

	@Required
	public void setExpression(final String expression) {
		this.expression = Pattern.compile(expression, Pattern.DOTALL
				| Pattern.MULTILINE);
	}
	
	@Required
	public void setfacilityIDExpression(final String expression) {
		this.facilityIDExpression = Pattern.compile(expression, Pattern.DOTALL
				| Pattern.MULTILINE);
	}
	
	@Required
	public void setcorrelationDateExpression(final String expression) {
		this.correlationDateExpression = Pattern.compile(expression, Pattern.DOTALL
				| Pattern.MULTILINE);
	}

	@Required
	public void setMatchGroupKeyPosition(final int matchGroupKeyPosition) {
		this.matchGroupKeyPosition = matchGroupKeyPosition;
	}

	@Required
	public void setMatchGroupValuePosition(final int matchGroupValuePosition) {
		this.matchGroupValuePosition = matchGroupValuePosition;
	}

	@Override
	public Map<String, String> split(final String src) throws SplitterException {
		if (NullChecker.isEmpty(src)) {
			return null;
		}
		final Map<String, String> propertyMap = new HashMap<String, String>();

		final Matcher matcher = this.expression.matcher(src);
		while (matcher.find()) {
			final String key = matcher.group(this.matchGroupKeyPosition);
			final String value = matcher.group(this.matchGroupValuePosition);
			if (NullChecker.isNotEmpty(key) && NullChecker.isNotEmpty(value)) {
				propertyMap.put(key, value);
			}
		}
		return propertyMap;
	}
	
	@Override
	public Map<String, String> splitAndFilter(final String src) throws SplitterException {
		if (NullChecker.isEmpty(src)) {
			return null;
		}
		final Map<String, String> propertyMap = new HashMap<String, String>();
		
		final Matcher matcher = this.expression.matcher(src);
		
		// loop through each facility
		while (matcher.find()) {
			final String key = matcher.group(this.matchGroupKeyPosition);
			String facilityResponse = matcher.group(this.matchGroupValuePosition);
			
			//get the index of the beginning of each separate response
			ArrayList<Integer> indexList = new ArrayList<Integer>();
			final Matcher facilityMatcher = this.facilityIDExpression.matcher(facilityResponse);
			
			while(facilityMatcher.find()) {
				indexList.add(facilityMatcher.start() - key.length() - 1);
			}
			
			//split the response into separate responses for each update
			if(indexList.size() > 1)
			{
				String latestResponse = null;
				Date latestDate = null;
				
				for (int i = 0; i < indexList.size(); i++)
				{
					String currentResponse = null;
					if(i == indexList.size() - 1)
					{
						currentResponse = facilityResponse.substring(indexList.get(i), facilityResponse.length());
					}
					else
					{
						currentResponse = facilityResponse.substring(indexList.get(i), indexList.get(i+1));
					}
					
					// find the correlation date in each separate response
					final Matcher correlationDateMatcher = this.correlationDateExpression.matcher(currentResponse);
					Date currentCorrelationDate = null;
					if(correlationDateMatcher.find()){
						String dateString = correlationDateMatcher.group(this.matchGroupValuePosition);//.replace('@', ' ');
						
						currentCorrelationDate = DateUtil.dateFromString(dateString);
					}
					
					// compare correlation date and store the latest correlation
					// date with the separate response
					if(latestDate == null)
					{
						latestDate = currentCorrelationDate;
						latestResponse = currentResponse;
					}
					else
					{
						if(latestDate.before(currentCorrelationDate))
						{
							latestDate = currentCorrelationDate;
							latestResponse = currentResponse;
						}
					}
				}
				
				// add the response with the latest correlation date to the propertyMap
				final String value = latestResponse;
				if (NullChecker.isNotEmpty(key) && NullChecker.isNotEmpty(value)) {
					propertyMap.put(key, value);
				}
			}
			else
			{
				final String value = matcher.group(this.matchGroupValuePosition);
				if (NullChecker.isNotEmpty(key) && NullChecker.isNotEmpty(value)) {
					propertyMap.put(key, value);
				}
			}

		}
		return propertyMap;
	}
}
