package gov.va.ccd.consol.component.transform;

import gov.va.ccd.components.transform.SubstanceAdminTransformer;
import gov.va.ccd.components.transform.Transformer;
import gov.va.ccd.service.util.EffTimeUtils;
import gov.va.ccd.service.util.Utils;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang3.StringEscapeUtils;
import org.apache.commons.lang3.StringUtils;
import org.eclipse.emf.common.util.EList;
import org.openhealthtools.mdht.uml.cda.AssignedAuthor;
import org.openhealthtools.mdht.uml.cda.AssignedEntity;
import org.openhealthtools.mdht.uml.cda.Author;
import org.openhealthtools.mdht.uml.cda.Consumable;
import org.openhealthtools.mdht.uml.cda.Criterion;
import org.openhealthtools.mdht.uml.cda.EncompassingEncounter;
import org.openhealthtools.mdht.uml.cda.Encounter;
import org.openhealthtools.mdht.uml.cda.EntryRelationship;
import org.openhealthtools.mdht.uml.cda.HealthCareFacility;
import org.openhealthtools.mdht.uml.cda.Informant12;
import org.openhealthtools.mdht.uml.cda.Location;
import org.openhealthtools.mdht.uml.cda.ManufacturedProduct;
import org.openhealthtools.mdht.uml.cda.Material;
import org.openhealthtools.mdht.uml.cda.Observation;
import org.openhealthtools.mdht.uml.cda.Organization;
import org.openhealthtools.mdht.uml.cda.Participant2;
import org.openhealthtools.mdht.uml.cda.ParticipantRole;
import org.openhealthtools.mdht.uml.cda.Performer2;
import org.openhealthtools.mdht.uml.cda.Person;
import org.openhealthtools.mdht.uml.cda.Place;
import org.openhealthtools.mdht.uml.cda.PlayingEntity;
import org.openhealthtools.mdht.uml.cda.Precondition;
import org.openhealthtools.mdht.uml.cda.Procedure;
import org.openhealthtools.mdht.uml.cda.Product;
import org.openhealthtools.mdht.uml.cda.StrucDocText;
import org.openhealthtools.mdht.uml.cda.SubstanceAdministration;
import org.openhealthtools.mdht.uml.cda.Supply;
import org.openhealthtools.mdht.uml.cda.ccd.EncounterLocation;
import org.openhealthtools.mdht.uml.hl7.datatypes.AD;
import org.openhealthtools.mdht.uml.hl7.datatypes.ADXP;
import org.openhealthtools.mdht.uml.hl7.datatypes.ANY;
import org.openhealthtools.mdht.uml.hl7.datatypes.CD;
import org.openhealthtools.mdht.uml.hl7.datatypes.CE;
import org.openhealthtools.mdht.uml.hl7.datatypes.CR;
import org.openhealthtools.mdht.uml.hl7.datatypes.ED;
import org.openhealthtools.mdht.uml.hl7.datatypes.EN;
import org.openhealthtools.mdht.uml.hl7.datatypes.ENXP;
import org.openhealthtools.mdht.uml.hl7.datatypes.II;
import org.openhealthtools.mdht.uml.hl7.datatypes.IVL_PQ;
import org.openhealthtools.mdht.uml.hl7.datatypes.IVL_TS;
import org.openhealthtools.mdht.uml.hl7.datatypes.IVXB_TS;
import org.openhealthtools.mdht.uml.hl7.datatypes.ON;
import org.openhealthtools.mdht.uml.hl7.datatypes.PN;
import org.openhealthtools.mdht.uml.hl7.datatypes.PQ;
import org.openhealthtools.mdht.uml.hl7.datatypes.PQR;
import org.openhealthtools.mdht.uml.hl7.datatypes.SXCM_TS;
import org.openhealthtools.mdht.uml.hl7.datatypes.TS;
import org.openhealthtools.mdht.uml.hl7.vocab.ActClass;
import org.openhealthtools.mdht.uml.hl7.vocab.ActClassObservation;
import org.openhealthtools.mdht.uml.hl7.vocab.NullFlavor;
import org.openhealthtools.mdht.uml.hl7.vocab.ParticipationType;
import org.openhealthtools.mdht.uml.hl7.vocab.RoleClassRoot;
import org.openhealthtools.mdht.uml.hl7.vocab.x_ActMoodDocumentObservation;
import org.openhealthtools.mdht.uml.hl7.vocab.x_ActRelationshipEntryRelationship;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class TransformerConsolCore
{
	private static final Logger logger = LoggerFactory.getLogger(TransformerConsolCore.class);

	public static void mapEncompassingEncouner(final Map<String, Object> results, final EncompassingEncounter encounter)
	{
		if(encounter != null)
		{
			mapIds(results, encounter.getIds());
			mapCode(results, "", encounter.getCode(), null);
			mapEffectiveTime(results, encounter.getEffectiveTime());
			mapWrappedCode(results, "discharge", encounter.getDischargeDispositionCode(), null);
			mapLocation(results, encounter.getLocation());
		}
	}

	public static void mapLocation(final Map<String, Object> results, final Location loc)
	{
		if(loc != null)
		{
			Map<String, Object> facility = new HashMap<String, Object>();
			mapHealthCareFacility(facility, loc.getHealthCareFacility());
			if(!facility.isEmpty())
			{
				results.put("facility", facility);
			}
		}
	}

	public static void mapHealthCareFacility(final Map<String, Object> results, final HealthCareFacility hcf)
	{
		if(hcf != null)
		{
			mapIds(results, hcf.getIds());
			mapCode(results, "", hcf.getCode(), null);
			mapPlace(results, hcf.getLocation());
		}
	}

	public static void mapPlace(final Map<String, Object> results, final Place place)
	{
		if(place != null)
		{
			mapString(results, "name", createName(place.getName()));
			mapString(results, "address", createAddress(place.getAddr()));
		}
	}

	public static void mapEncounterItem(final Map<String, Object> results, final Encounter encounter, final StrucDocText narrative)
	{
		// Default Items
		// id 16.01
		mapIds(results, encounter.getIds());
		// code 16.02-16.03
		mapCode(results, "", encounter.getCode(), narrative);
		// text
		mapOriginalText(results, "", encounter.getText(), narrative);
		// time 16.04
		mapEffectiveTime(results, encounter.getEffectiveTime());

		// TODO: DischargeDisposition. 16.09

		// priority code 16.07
		mapWrappedCode(results, "priority", encounter.getPriorityCode(), narrative);
		// statusCode
		mapWrappedCode(results, "status", encounter.getStatusCode(), narrative);

		// 16.05
		mapPerformers(results, "", encounter.getPerformers());
		// author
		mapAuthors(results, "", encounter.getAuthors());

		// Find other sources
		mapSources(results, encounter.getAuthors(), encounter.getPerformers(), encounter.getParticipants(), encounter.getInformants());

		// participant
		mapEncounterParticipants(results, "", encounter.getParticipants(), narrative);
	}

	public static void mapSources(final Map<String, Object> results, final EList<Author> authors, final EList<Performer2> performers, final EList<Participant2> participants, final EList<Informant12> informants)
	{
		if(!results.containsKey("source"))
		{

			String source = Utils.getSource(authors, performers, participants, informants);

			if(StringUtils.isNotBlank(source))
			{
				results.put("source", source);
			}
		}
	}

	public static void mapProcedureItem(final Map<String, Object> results, final Procedure procedure, final StrucDocText narrative)
	{
		// Default Items
		// id
		mapIds(results, procedure.getIds());
		// code
		mapCode(results, "", procedure.getCode(), narrative);

		// For VA specific qualifiers
		List<Map<String, Object>> qualifiers = new ArrayList<Map<String, Object>>();

		if(ActClass.PROC.equals(procedure.getClassCode()))
		{
			CD code = procedure.getCode();
			if(code != null)
			{
				for(CR qual : code.getQualifiers())
				{
					Map<String, Object> qualifier = new HashMap<String, Object>();
					CD qualVal = qual.getValue();
					mapCode(qualifier, "", qualVal, narrative);
					if(!qualifier.isEmpty())
					{
						qualifiers.add(qualifier);
					}
				}
			}
		}
		if(CollectionUtils.isNotEmpty(qualifiers))
		{
			results.put("qualifiers", qualifiers);
		}

		// text
		mapOriginalText(results, "", procedure.getText(), narrative);
		// time
		mapEffectiveTime(results, procedure.getEffectiveTime());
		// priority code
		mapWrappedCode(results, "priority", procedure.getPriorityCode(), narrative);
		// statusCode
		mapWrappedCode(results, "status", procedure.getStatusCode(), narrative);

		// Approach Sties.
		List<Map<String, Object>> sites = new ArrayList<Map<String, Object>>();

		for(CD code : procedure.getApproachSiteCodes())
		{
			Map<String, Object> site = new HashMap<String, Object>();
			mapCode(site, "", code, narrative);
			if(!site.isEmpty())
			{
				sites.add(site);
			}
		}

		if(CollectionUtils.isNotEmpty(sites))
		{
			results.put("sites", sites);
		}

		// target Sties.
		List<Map<String, Object>> targets = new ArrayList<Map<String, Object>>();

		for(CD code : procedure.getTargetSiteCodes())
		{
			Map<String, Object> target = new HashMap<String, Object>();
			mapCode(target, "", code, narrative);
			if(!target.isEmpty())
			{
				targets.add(target);
			}
		}

		if(CollectionUtils.isNotEmpty(targets))
		{
			results.put("targets", targets);
		}

		// author
		mapAuthors(results, "", procedure.getAuthors());
		// participant
		mapParticipants(results, "", procedure.getParticipants(), narrative);
		// Performers
		mapPerformers(results, "", procedure.getPerformers());
		// Find other sources
		mapSources(results, procedure.getAuthors(), procedure.getPerformers(), procedure.getParticipants(), procedure.getInformants());
	}

	public static void mapSubstanceItem(final Map<String, Object> results, final SubstanceAdministration subAdm, final StrucDocText narrative)
	{
		// Default Items
		// id
		if(subAdm.getNegationInd() != null)
		{
			if(!subAdm.getNegationInd())
			{
				mapString(results, "refused", "0");
			}
		}

		if(!results.containsKey("refused"))
		{
			mapString(results, "refused", "1");
		}

		mapIds(results, subAdm.getIds());
		// code
		mapCode(results, "", subAdm.getCode(), narrative);
		// time
		if(CollectionUtils.isNotEmpty(subAdm.getEffectiveTimes()))
		{
			mapEffectiveTime(results, subAdm.getEffectiveTimes().get(0));
		}
		// text
		mapOriginalText(results, "", subAdm.getText(), narrative);
		// RouteCode
		mapWrappedCode(results, "routeCode", subAdm.getRouteCode(), narrative);
		// Site Code
		mapWrappedCodes(results, "siteCodes", subAdm.getApproachSiteCodes(), narrative);
		if(subAdm.getRepeatNumber() != null && subAdm.getRepeatNumber().getValue() != null)
		{
			mapString(results, "series", subAdm.getRepeatNumber().getValue().toString());
		}
		// Administration Unit Code
		mapWrappedCode(results, "adminUnit", subAdm.getAdministrationUnitCode(), narrative);
		// Dose Quantity
		mapPQValue(results, "doseQuantity", subAdm.getDoseQuantity(), narrative);
		// Dose Restriction
		SubstanceAdminTransformer.mapDoseRestriction(subAdm, results);

		mapReasons(results, subAdm.getEntryRelationships(), narrative);

		// Criterion Break out later.
		List<Map<String, Object>> crits = new ArrayList<Map<String, Object>>();
		for(Precondition cond : subAdm.getPreconditions())
		{
			Map<String, Object> vals = new HashMap<String, Object>();
			Criterion crit = cond.getCriterion();
			mapCode(vals, "", crit.getCode(), narrative);
			mapOriginalText(vals, "", crit.getText(), narrative);
			if(!vals.isEmpty())
			{
				crits.add(vals);
			}
		}
		if(CollectionUtils.isNotEmpty(crits))
		{
			results.put("criterias", crits);
		}

		SubstanceAdminTransformer.mapSubstanceEffectiveTimes(subAdm, results);

		if(subAdm.getConsumable() != null)
		{
			Consumable cons = subAdm.getConsumable();
			if(cons.getManufacturedProduct() != null)
			{
				ManufacturedProduct manProd = cons.getManufacturedProduct();
				if(manProd.getManufacturedMaterial() != null)
				{
					Material material = manProd.getManufacturedMaterial();
					mapWrappedCode(results, "material", material.getCode(), narrative);
					String name = createName(material.getName());
					if(StringUtils.isNotBlank(name))
					{
						mapString(results, "brandName", name);
					}
					if(material.getLotNumberText() != null)
					{
						mapWrappedOriginalText(results, "lot", material.getLotNumberText(), narrative);
					}
				}
				if(manProd.getManufacturerOrganization() != null)
				{
					Organization org = manProd.getManufacturerOrganization();
					mapRepresentedOrg(results, "man", org);
				}
			}
		}
		SubstanceAdminTransformer.mapEntryRels(results, subAdm, narrative);
		// author
		mapAuthors(results, "", subAdm.getAuthors());
		// Reactions
		mapReactions(results, subAdm.getEntryRelationships(), narrative);

		mapReasons(results, subAdm.getEntryRelationships(), narrative);
		// participant
		mapParticipants(results, "", subAdm.getParticipants(), narrative);
		// Find other sources
		mapSources(results, subAdm.getAuthors(), subAdm.getPerformers(), subAdm.getParticipants(), subAdm.getInformants());
	}

	public static void mapReactions(final Map<String, Object> results, final Observation obs, final StrucDocText narrative)
	{
		if(CollectionUtils.isNotEmpty(obs.getEntryRelationships()))
		{
			mapReactions(results, obs.getEntryRelationships(), narrative);
		}
	}

	public static void mapReactions(final Map<String, Object> results, final EList<EntryRelationship> rels, final StrucDocText narrative)
	{
		if(CollectionUtils.isNotEmpty(rels))
		{

			List<Map<String, Object>> reactions = new ArrayList<Map<String, Object>>();
			for(EntryRelationship rel : rels)
			{
				if(rel.getInversionInd() != null && x_ActRelationshipEntryRelationship.MFST.equals(rel.getTypeCode()) && rel.getInversionInd())
				{
					// Get Reactions
					if(rel.getObservation() != null)
					{
						Observation relObs = rel.getObservation();
						Map<String, Object> reactionMap = new HashMap<String, Object>();
						TransformerConsolCore.mapEDValue(reactionMap, "text", relObs.getText(), narrative);
						mapSeverityObservation(reactionMap, relObs.getEntryRelationships(), narrative);
						for(ANY any : relObs.getValues())
						{
							if(any instanceof CD)
							{
								CD reactionCode = (CD) any;
								TransformerConsolCore.mapCode(reactionMap, "", reactionCode, narrative);
							}
						}
						if(!reactionMap.isEmpty())
						{
							reactions.add(reactionMap);
						}
					}
				}
				else
				{
					mapSeverityObservation(results, rel, narrative);
				}
			}
			if(CollectionUtils.isNotEmpty(reactions))
			{
				results.put("reactions", reactions);
			}
		}
	}

	public static void mapSeverityObservation(final Map<String, Object> results, final EList<EntryRelationship> entryRels, final StrucDocText narrative)
	{
		if(CollectionUtils.isEmpty(entryRels))
		{
			return;
		}

		for(EntryRelationship rel : entryRels)
		{
			mapSeverityObservation(results, rel, narrative);
		}

	}

	public static void mapSeverityObservation(final Map<String, Object> results, final EntryRelationship entryRel, final StrucDocText narrative)
	{
		if(entryRel == null || entryRel.getInversionInd() == null || !(x_ActRelationshipEntryRelationship.SUBJ.equals(entryRel.getTypeCode()) && entryRel.getInversionInd()))
		{
			return;
		}

		Observation obs = entryRel.getObservation();
		if(obs != null && ActClassObservation.OBS.equals(obs.getClassCode()) && x_ActMoodDocumentObservation.EVN.equals(obs.getMoodCode()) && Utils.containsTemplateId(obs.getTemplateIds(), "2.16.840.1.113883.10.20.22.4.8"))
		{
			Map<String, Object> ret = new HashMap<String, Object>();

			if(CollectionUtils.isNotEmpty(obs.getValues()))
			{
				for(ANY any : obs.getValues())
				{
					if(any instanceof CD)
					{
						CD code = (CD) any;
						TransformerConsolCore.mapCode(ret, "", code, narrative);
					}
				}
			}

			if(obs.getText() != null)
			{
				TransformerConsolCore.mapEDValue(ret, "", obs.getText(), narrative);
			}

			if(CollectionUtils.isNotEmpty(obs.getInterpretationCodes()))
			{
				List<Map<String, Object>> interps = new ArrayList<Map<String, Object>>();
				for(CE code : obs.getInterpretationCodes())
				{
					Map<String, Object> interp = new HashMap<String, Object>();
					TransformerConsolCore.mapCode(interp, "", code, narrative);
					if(!interp.isEmpty())
					{
						interps.add(interp);
					}
				}
				if(CollectionUtils.isNotEmpty(interps))
				{
					ret.put("interps", interps);
				}
			}

			if(!ret.isEmpty())
			{
				results.put("severity", ret);
			}
		}
	}

	public static void mapSupplyItem(final Map<String, Object> results, final Supply supply, final StrucDocText narrative)
	{
		// Default Items
		// id
		mapIds(results, supply.getIds());
		// code
		mapCode(results, "", supply.getCode(), narrative);
		// text
		mapOriginalText(results, "", supply.getText(), narrative);
		// time
		List<Map<String, Object>> times = new ArrayList<Map<String, Object>>();

		for(SXCM_TS time : supply.getEffectiveTimes())
		{
			Map<String, Object> map = new HashMap<String, Object>();
			mapEffectiveTime(map, time);
			if(!map.isEmpty())
			{
				times.add(map);
			}
		}

		if(CollectionUtils.isNotEmpty(times))
		{
			results.put("times", times);
		}

		// statusCode
		mapWrappedCode(results, "status", supply.getStatusCode(), narrative);
		// author
		mapAuthors(results, "", supply.getAuthors());
		// participant
		mapParticipants(results, "", supply.getParticipants(), narrative);

		// Product
		mapProduct(results, supply.getProduct(), narrative);
		// Find other sources
		mapSources(results, supply.getAuthors(), supply.getPerformers(), supply.getParticipants(), supply.getInformants());
	}

	public static void mapObservationItem(final Map<String, Object> results, final Observation observation, final StrucDocText narrative)
	{
		// Default Items
		// id
		mapIds(results, observation.getIds());
		// code
		mapCode(results, "", observation.getCode(), narrative);
		// text
		mapOriginalText(results, "", observation.getText(), narrative);

		// time
		mapEffectiveTime(results, observation.getEffectiveTime());

		// statusCode
		mapWrappedCode(results, "status", observation.getStatusCode(), narrative);
		// author
		mapAuthors(results, "", observation.getAuthors());

		// Map Values
		mapValues(results, observation.getValues(), narrative);

		// participant
		mapParticipants(results, "", observation.getParticipants(), narrative);
		// Find other sources
		mapSources(results, observation.getAuthors(), observation.getPerformers(), observation.getParticipants(), observation.getInformants());
	}

	/**
	 * Will create a List of maps which can contain any of the following PQ, CD,
	 * or ST types.
	 * 
	 * @param results
	 * @param values
	 * @param narrative
	 * @see #mapPQValue(Map, String, PQ, StrucDocText)
	 * @see #mapCode(Map, String, CD, StrucDocText)
	 * @see #mapOriginalText(Map, String, ED, StrucDocText)
	 */
	public static void mapValues(final Map<String, Object> results, final EList<ANY> values, final StrucDocText narrative)
	{
		if(CollectionUtils.isNotEmpty(values))
		{
			List<Map<String, Object>> ret = new ArrayList<Map<String, Object>>();
			for(ANY any : values)
			{
				Map<String, Object> map = new HashMap<String, Object>();
				if(any instanceof PQ)
				{
					mapPQValue(map, "", (PQ) any, narrative);
				}
				else if(any instanceof CD)
				{
					mapCode(map, "", (CD) any, narrative);
				}
				else if(any instanceof ED)
				{
					mapOriginalText(map, "", (ED) any, narrative);
				}
				if(!map.isEmpty())
				{
					ret.add(map);
				}
			}
			if(CollectionUtils.isNotEmpty(ret))
			{
				results.put("values", ret);
			}
		}
	}

	public static void mapObservationReason(final Map<String, Object> results, final List<EntryRelationship> rels, final StrucDocText narrative)
	{
		for(EntryRelationship rel : rels)
		{
			if(x_ActRelationshipEntryRelationship.RSON.equals(rel.getTypeCode()))
			{
				if(rel.getObservation() != null)
				{
					Map<String, Object> reason = new HashMap<String, Object>();
					Transformer.mapObservationItem(reason, rel.getObservation(), narrative);
					if(!reason.isEmpty())
					{
						results.put("reason", reason);
					}
				}
			}
		}
	}

	public static void mapReasons(final Map<String, Object> results, final List<EntryRelationship> rels, final StrucDocText narrative)
	{
		for(EntryRelationship rel : rels)
		{
			if(x_ActRelationshipEntryRelationship.RSON.equals(rel.getTypeCode()))
			{
				logger.debug("Found the reason.");
				if(rel.getEncounter() != null)
				{
					Map<String, Object> reason = new HashMap<String, Object>();
					Transformer.mapEncounterItem(reason, rel.getEncounter(), narrative);
					if(!reason.isEmpty())
					{
						results.put("reason", reason);
					}
				}
				else if(rel.getAct() != null)
				{
					Map<String, Object> reason = new HashMap<String, Object>();
					Transformer.mapActItem(reason, rel.getAct(), narrative);
					if(!reason.isEmpty())
					{
						results.put("reason", reason);
					}
				}
				else if(rel.getObservation() != null)
				{
					Map<String, Object> reason = new HashMap<String, Object>();
					Transformer.mapObservationItem(reason, rel.getObservation(), narrative);
					if(!reason.isEmpty())
					{
						results.put("reason", reason);
					}
				}
			}
		}
	}

	public static void mapManProduct(final Map<String, Object> results, final ManufacturedProduct manufacturedProduct, StrucDocText narrative)
	{
		if(manufacturedProduct != null)
		{
			Material mat = manufacturedProduct.getManufacturedMaterial();
			Organization org = manufacturedProduct.getManufacturerOrganization();
			Map<String, Object> matMap = new HashMap<String, Object>();
			if(mat != null)
			{
				mapWrappedCode(matMap, "materialCode", mat.getCode(), narrative);
				mapString(matMap, "materialName", createName(mat.getName()));
			}

			if(org != null)
			{
				mapRepresentedOrg(matMap, "", org);
			}
			if(!matMap.isEmpty())
			{
				results.put("material", matMap);
			}
		}

	}

	public static void mapProduct(final Map<String, Object> results, final Product product, StrucDocText narrative)
	{
		if(product != null)
		{
			mapManProduct(results, product.getManufacturedProduct(), narrative);
		}
	}

	public static void mapIds(final Map<String, Object> results, List<II> ids)
	{
		List<Map<String, Object>> retIds = new ArrayList<Map<String, Object>>();
		for(II id : ids)
		{
			Map<String, Object> retId = new HashMap<String, Object>();
			mapId(retId, id);
			if(!retId.isEmpty())
			{
				retIds.add(retId);
			}
		}
		if(CollectionUtils.isNotEmpty(retIds))
		{
			results.put("ids", retIds);
		}
	}

	public static void mapId(final Map<String, Object> results, II id)
	{
		mapString(results, "idRoot", id.getRoot());
		mapString(results, "idExtension", id.getExtension());
		mapNullFlavor(results, "idNullFlavor", id.getNullFlavor());
	}

	/**
	 * <PRE>
	 * Will added CD data elements to results:
	 * <ul>
	 * <li> Code: keyPrefix + <b>code</b></li>
	 * <li> Code System: keyPrefix + <b>codeSystem</b></li>
	 * <li> Code DisplayName: keyPrefix + <b>codeDisplayName</b></li>
	 * <li> Code Null Flavor passes keyPrefix + <b>codeNullFlavor</b> to {@link #mapNullFlavor(Map, String, NullFlavor)}</li>
	 * <li> Code Original Text passes keyPrefix + <b>code</b> to {@link #mapOriginalText(Map, String, ED, StrucDocText)}</li>
	 * <li> Adds a list of translations to <b>translations</b> list items are mapped using {@link #mapCode(Map, String, CD, StrucDocText)} (this)</li>   
	 * </ul>
	 * </PRE>
	 * 
	 * @param results
	 * @param keyPrefix
	 * @param code
	 * @param narrative
	 */
	public static void mapCode(final Map<String, Object> results, final String keyPrefix, final CD code, final StrucDocText narrative)
	{
		if(code != null)
		{
			mapString(results, keyPrefix + "code", code.getCode());
			mapString(results, keyPrefix + "codeSystem", code.getCodeSystem());
			mapString(results, keyPrefix + "codeDisplayName", code.getDisplayName());
			mapNullFlavor(results, keyPrefix + "codeNullFlavor", code.getNullFlavor());
			if(code.getOriginalText() != null)
			{
				mapOriginalText(results, keyPrefix + "code", code.getOriginalText(), narrative);
			}

			List<Map<String, Object>> trans = new ArrayList<Map<String, Object>>();
			for(CD tran : code.getTranslations())
			{
				Map<String, Object> tranMap = new HashMap<String, Object>();
				mapCode(tranMap, "", tran, narrative);
				if(!tranMap.isEmpty())
				{
					trans.add(tranMap);
				}
			}
			if(results != null && !trans.isEmpty())
			{
				results.put("translations", trans);
			}
		}
	}

	public static void mapWrappedCodes(final Map<String, Object> results, final String keyName, final EList<CD> codes, final StrucDocText narrative)
	{
		List<Map<String, Object>> ret = new ArrayList<Map<String, Object>>();
		if(CollectionUtils.isNotEmpty(codes))
		{
			for(CD code : codes)
			{
				Map<String, Object> wrapper = new HashMap<String, Object>();
				mapCode(wrapper, "", code, narrative);
				if(!wrapper.isEmpty())
				{
					ret.add(wrapper);
				}
			}
		}
		if(CollectionUtils.isNotEmpty(ret))
		{
			results.put(keyName, ret);
		}
	}

	public static void mapWrappedCode(final Map<String, Object> results, final String keyName, final CD code, final StrucDocText narrative)
	{
		Map<String, Object> wrapper = new HashMap<String, Object>();
		mapCode(wrapper, "", code, narrative);
		if(!wrapper.isEmpty())
		{
			results.put(keyName, wrapper);
		}
	}

	public static void mapIVL_PQValue(final Map<String, Object> results, final String keyPrefix, final IVL_PQ value, final StrucDocText narrative)
	{
		if(value != null && results != null)
		{
			mapPQValue(results, "", value, narrative);
			mapPQValue(results, "low", value.getLow(), narrative);
			mapPQValue(results, "high", value.getHigh(), narrative);
		}
	}

	public static void mapEDValue(final Map<String, Object> results, final String keyPrefix, final ED value, final StrucDocText narrative)
	{
		if(value != null && results != null)
		{
			// First look for text
			if(StringUtils.isNotBlank(value.getText()))
			{
				mapString(results, keyPrefix + "value", value.getText());
			}
			else if(value.getReference() != null)
			{
				// Check for a reference.
				Map<String, Object> valRef = Utils.getTagMapFromId(narrative, value.getReference().getValue());

				if(!valRef.isEmpty())
				{
					results.put(keyPrefix + "references", valRef);
				}
			}
		}
	}

	/**
	 * <PRE>
	 * Will add PQ values to results:
	 * <ul>
	 * <li> Translation Values: keyPrefix + <b>tranValue</b></li>
	 * <li> Code Values using {@link #mapCode(Map, String, CD, StrucDocText)} </li>
	 * <li> Value: keyPrefix + <b>value</b></li>
	 * <li> Unit (if it is not == 1 which MDHT default):keyPrefix + <b>unit</b></li>
	 * </ul>
	 * </PRE>
	 * 
	 * @param results
	 * @param keyPrefix
	 * @param value
	 * @param narrative
	 */
	public static void mapPQValue(final Map<String, Object> results, final String keyPrefix, final PQ value, final StrucDocText narrative)
	{
		if(value != null && results != null)
		{
			// Add possible translations. For now assumes just one.
			// TODO: If needed update to account for others.
			for(PQR tran : value.getTranslations())
			{
				if(tran.getValue() != null)
				{
					mapString(results, keyPrefix + "tranValue", tran.getValue().toString());
				}
				else
				{
					mapCode(results, keyPrefix, tran, narrative);
				}
			}

			if(value.getValue() != null)
			{
				mapString(results, keyPrefix + "value", value.getValue().toString());
			}
			if(!"1".equals(value.getUnit()))
			{
				mapString(results, keyPrefix + "unit", value.getUnit());
			}
		}
	}

	public static void mapObsValueOrCode(final Map<String, Object> results, final String keyPrefix, final Observation obs, final StrucDocText narrative)
	{
		if(obs != null)
		{
			if(CollectionUtils.isNotEmpty(obs.getValues()))
			{
				for(ANY val : obs.getValues())
				{
					if(val instanceof CD)
					{
						// CD is one of the main types that
						// contains a display name.
						CD cd = (CD) val;
						Transformer.mapWrappedCode(results, keyPrefix, cd, narrative);
					}
				}
			}
			else if(obs.getCode() != null)
			{
				Transformer.mapWrappedCode(results, keyPrefix, obs.getCode(), narrative);
			}
		}
	}

	/**
	 * If there is a text it will be mapped to keyPrefix + OriginalText. If
	 * there is a reference that will be mapped to keyPrefix + OriginalTextMap
	 * from the map created by
	 * {@link Utils#getTagMapFromId(StrucDocText, String)}.
	 * 
	 * @param results
	 * @param keyPrefix
	 * @param ot
	 * @param narrative
	 */
	public static void mapOriginalText(final Map<String, Object> results, final String keyPrefix, final ED ot, final StrucDocText narrative)
	{
		if(ot == null)
			return;
		if(StringUtils.isNotBlank(ot.getText()))
		{
			mapString(results, keyPrefix + "OriginalText", ot.getText());
		}
		else if(ot.getReference() != null && StringUtils.isNotBlank(ot.getReference().getValue()))
		{
			Map<String, Object> otRef = Utils.getTagMapFromId(narrative, ot.getReference().getValue());
			if(!otRef.isEmpty())
			{
				results.put(keyPrefix + "OriginalTextMap", otRef);
			}
		}
	}

	public static void mapWrappedOriginalText(final Map<String, Object> results, final String keyPrefix, final ED ot, final StrucDocText narrative)
	{
		Map<String, Object> map = new HashMap<String, Object>();
		mapOriginalText(map, "", ot, narrative);
		if(!map.isEmpty())
		{
			results.put(keyPrefix, map);
		}
	}

	/**
	 * Checks to see if this participant is from a location (LOC/SDLOC).
	 * 
	 * @param part
	 * @return
	 */
	public static boolean isLocation(final Participant2 part)
	{
		if(part != null && ParticipationType.LOC.equals(part.getTypeCode()))
		{
			ParticipantRole role = part.getParticipantRole();
			if(role != null && RoleClassRoot.SDLOC.equals(role.getClassCode()))
			{
				return true;
			}
		}

		return false;
	}

	/**
	 * Checks to see if this participant is a medication vehicle.
	 * 
	 * @param part
	 * @return
	 */
	public static boolean isVehicle(final Participant2 part)
	{
		if(part != null)
		{
			if(part.getParticipantRole() != null)
			{
				ParticipantRole role = part.getParticipantRole();
				if(role.getCode() != null)
				{
					CD code = role.getCode();
					if("412307009".equals(code.getCode()))
					{
						return true;
					}
				}
			}
		}

		return false;
	}

	/**
	 * Checks to see if this participant is from administration (ORG).
	 * 
	 * @param part
	 * @return
	 */
	public static boolean isAdministration(final Participant2 part)
	{
		if(part != null && ParticipationType.ORG.equals(part.getTypeCode()))
		{
			return true;
		}

		return false;
	}

	public static void mapParticipants(final Map<String, Object> results, final String keyPrefix, final EList<Participant2> participants, StrucDocText narrative)
	{
		List<Map<String, Object>> parts = new ArrayList<Map<String, Object>>();
		for(Participant2 participant : participants)
		{
			Map<String, Object> map = new HashMap<String, Object>();

			if(participant != null)
			{
				if(isLocation(participant))
				{
					mapParticipant(map, "", participant, narrative);
					if(!map.isEmpty())
					{
						results.put("facilityLocation", map);
					}
				}
				else if(isAdministration(participant))
				{
					mapParticipant(map, "", participant, narrative);
					if(!map.isEmpty())
					{
						results.put("admissionLocation", map);
					}
				}
				if(isVehicle(participant))
				{
					mapParticipant(map, "", participant, narrative);
					if(!map.isEmpty())
					{
						results.put("vehicle", map);
					}

				}
				else
				{
					mapParticipant(map, "", participant, narrative);

					if(!map.isEmpty())
					{
						parts.add(map);
					}
				}
			}
		}
		if(CollectionUtils.isNotEmpty(parts))
		{
			results.put("participants", parts);
		}
	}

	/**
	 * Maps participant info.
	 * 
	 * @param results
	 * @param keyPrefix
	 * @param participant
	 * @param narrative
	 * @see #mapEffectiveTime(Map, IVL_TS)
	 * @see #mapParticipantRole(Map, String, ParticipantRole, StrucDocText)
	 */
	public static void mapParticipant(final Map<String, Object> results, final String keyPrefix, final Participant2 participant, StrucDocText narrative)
	{
		if(participant != null)
		{
			mapEffectiveTime(results, participant.getTime());
			mapParticipantRole(results, keyPrefix, participant.getParticipantRole(), narrative);
		}
	}

	/**
	 * Currently only maps the sub playing entity
	 * {@link #mapPlayingEntity(Map, String, PlayingEntity, StrucDocText)}
	 * 
	 * @param results
	 * @param keyPrefix
	 * @param role
	 * @param narrative
	 * @see #mapPlayingEntity(Map, String, PlayingEntity, StrucDocText)
	 */
	public static void mapParticipantRole(final Map<String, Object> results, final String keyPrefix, final ParticipantRole role, StrucDocText narrative)
	{
		if(role != null)
		{
			mapPlayingEntity(results, keyPrefix, role.getPlayingEntity(), narrative);
		}
	}

	/**
	 * Adds a map with the key <b>playingEntity</b> to results.
	 * 
	 * playingNames (List<String>) code
	 * 
	 * @param results
	 * @param keyPrefix
	 * @param entity
	 * @param narrative
	 */
	public static void mapPlayingEntity(final Map<String, Object> results, final String keyPrefix, final PlayingEntity entity, StrucDocText narrative)
	{
		Map<String, Object> ret = new HashMap<String, Object>();
		mapCode(ret, "", entity.getCode(), narrative);
		List<String> names = new ArrayList<String>();
		if(entity != null)
		{
			for(PN name : entity.getNames())
			{
				String strName = createName(name);
				if(StringUtils.isNotBlank(strName))
				{
					names.add(strName);
				}
			}
			if(CollectionUtils.isNotEmpty(names))
			{
				ret.put("names", names);
			}

			if(!ret.isEmpty())
			{
				results.put("playingEntity", ret);
			}
		}
	}

	public static void mapRepresentedOrgs(final Map<String, Object> results, final String keyPrefix, final EList<Organization> orgs)
	{
		List<Map<String, Object>> list = new ArrayList<Map<String, Object>>();
		for(Organization org : orgs)
		{
			Map<String, Object> map = new HashMap<String, Object>();
			mapRepresentedOrg(map, keyPrefix, org);
			if(!map.isEmpty())
			{
				list.add(map);
			}
		}
		results.put("orgs", list);
	}

	public static void mapRepresentedOrg(final Map<String, Object> results, final String keyPrefix, final Organization org)
	{
		if(org != null)
		{
			for(ON orgname : org.getNames())
			{
				mapString(results, keyPrefix + "source", orgname.getText());
			}
		}
	}

	public static void mapPerformers(final Map<String, Object> results, final String keyPrefix, final EList<Performer2> performers)
	{
		List<Map<String, Object>> list = new ArrayList<Map<String, Object>>();

		for(Performer2 performer : performers)
		{
			Map<String, Object> map = new HashMap<String, Object>();
			mapAssignedEntity(map, keyPrefix, performer.getAssignedEntity());

			if(!map.isEmpty())
			{
				list.add(map);
			}
		}
		if(CollectionUtils.isNotEmpty(list))
		{
			results.put("performers", list);
		}
	}

	public static void mapAssignedEntity(final Map<String, Object> results, final String keyPrefix, final AssignedEntity entity)
	{
		if(entity != null)
		{
			mapAssignedPerson(results, keyPrefix, entity.getAssignedPerson());
			if(CollectionUtils.isNotEmpty(entity.getAddrs()))
			{
				List<String> addrs = new ArrayList<String>();
				for(AD address : entity.getAddrs())
				{
					addrs.add(createAddress(address));
				}
				if(CollectionUtils.isNotEmpty(addrs))
				{
					results.put("entityAddr", addrs);
				}
			}
			mapRepresentedOrgs(results, keyPrefix, entity.getRepresentedOrganizations());
		}
	}

	public static void mapAuthors(final Map<String, Object> results, final String keyPrefix, final EList<Author> authors)
	{
		for(Author author : authors)
		{
			Map<String, Object> map = new HashMap<String, Object>();
			mapEffectiveTime(map, author.getTime());
			if(!map.isEmpty())
			{
				results.put("authorTime", map);
			}
			mapAssignedAuthor(results, keyPrefix, author.getAssignedAuthor());
		}
	}

	public static void mapAssignedAuthor(final Map<String, Object> results, final String keyPrefix, final AssignedAuthor author)
	{

		if(author != null && author.getAssignedPerson() != null)
		{
			Person person = author.getAssignedPerson();
			StringBuilder sb = new StringBuilder();
			for(PN name : person.getNames())
			{
				String retName = createName(name);
				if(StringUtils.isNotBlank(retName))
				{
					sb.append(retName).append(",");
				}
			}
			if(sb.length() > 0)
			{
				results.put("assignedPerson", sb.substring(0, sb.length() - 1));
			}
		}
		if(author != null)
		{
			mapRepresentedOrg(results, "", author.getRepresentedOrganization());
		}
	}

	public static void mapAssignedPerson(final Map<String, Object> results, final String keyPrefix, final Person person)
	{
		if(person != null)
		{
			for(PN name : person.getNames())
			{
				mapString(results, keyPrefix + "assignedPerson", createName(name));
			}

			mapNullFlavor(results, keyPrefix + "assignedPersonNullFlavor", person.getNullFlavor());
		}
	}

	public static void mapEffectiveTime(final Map<String, Object> results, final SXCM_TS time)
	{
		if(time != null)
		{
			mapString(results, "timeValue", time.getValue());
			mapNullFlavor(results, "timeNullFlavor", time.getNullFlavor());
		}
	}

	public static void mapEffectiveTime(final Map<String, Object> results, final TS time)
	{
		if(time != null)
		{
			mapString(results, "timeValue", time.getValue());
			mapNullFlavor(results, "timeNullFlavor", time.getNullFlavor());
		}
	}

	public static void mapEffectiveTime(final Map<String, Object> results, final IVL_TS time)
	{
		if(time != null)
		{
			mapString(results, "timeValue", time.getValue());
			results.put("convertedTime", EffTimeUtils.convertDate(time.getValue()));
			// MDHT will default to returning ASKU even if null not set. So need
			// to check.
			if(time.isSetNullFlavor())
			{
				mapNullFlavor(results, "timeNullFlavor", time.getNullFlavor());
			}
			mapTimeLow(results, time.getLow());
			mapTimeHigh(results, time.getHigh());
		}
	}

	public static void mapEncounterParticipants(final Map<String, Object> results, final String keyPrefix, final EList<Participant2> participants, final StrucDocText narrativeText)
	{
		if(participants != null)
		{
			for(Participant2 participant : participants)
			{
				Map<String, Object> map = new HashMap<String, Object>();
				if(participant instanceof EncounterLocation)
				{
					// Facility Location information.
					Transformer.mapParticipant(map, "", participant, narrativeText);
					if(!map.isEmpty())
					{
						results.put("facilityLocation", map);
					}
				}
				else if(ParticipationType.ORG.equals(participant.getTypeCode()))
				{
					// Admission Facility information
					Transformer.mapParticipant(map, "", participant, narrativeText);
					if(!map.isEmpty())
					{
						results.put("admissionLocation", map);
					}
				}
			}
		}
	}

	public static void mapTimeLow(final Map<String, Object> results, final IVXB_TS time)
	{
		if(time != null)
		{
			mapString(results, "timeLow", time.getValue());
			if(time.isSetNullFlavor())
			{
				mapNullFlavor(results, "timeLowNullFlavor", time.getNullFlavor());
			}
		}
	}

	public static void mapTimeHigh(final Map<String, Object> results, final IVXB_TS time)
	{
		if(time != null)
		{
			mapString(results, "timeHigh", time.getValue());
			if(time.isSetNullFlavor())
			{
				mapNullFlavor(results, "timeHighNullFlavor", time.getNullFlavor());
			}
		}
	}

	public static void mapNullFlavor(final Map<String, Object> map, final String key, final NullFlavor value)
	{
		if((value != null) && StringUtils.isNotBlank(key))
		{
			mapString(map, key, value.getLiteral());
		}
	}

	public static String createAddress(final AD address)
	{
		StringBuilder sb = new StringBuilder();
		for(ADXP street : address.getStreetAddressLines())
		{
			if(StringUtils.isNotBlank(street.getText()))
			{
				sb.append(street.getText()).append(" ");
			}
		}

		for(ADXP city : address.getCities())
		{
			if(StringUtils.isNotBlank(city.getText()))
			{
				sb.append(city.getText()).append(", ");
			}
		}

		for(ADXP state : address.getStates())
		{
			if(StringUtils.isNotBlank(state.getText()))
			{
				sb.append(state.getText()).append(" ");
			}
		}

		for(ADXP zip : address.getPostalCodes())
		{
			if(StringUtils.isNotBlank(zip.getText()))
			{
				sb.append(zip.getText());
			}
		}

		return sb.toString();
	}

	public static String loopForName(final EList<ENXP> names)
	{
		for(ENXP name : names)
		{
			if(StringUtils.isNotBlank(name.getText()))
			{
				return name.getText();
			}
		}

		return null;
	}

	public static String createNamesON(final EList<ON> names)
	{
		StringBuilder sb = new StringBuilder();
		for(ON name : names)
		{
			String nameVal = createName(name);
			if(StringUtils.isNotBlank(nameVal))
			{
				sb.append(nameVal).append(" ");
			}
		}

		return sb.toString();
	}

	public static String createNames(final EList<PN> names)
	{
		StringBuilder sb = new StringBuilder();
		for(PN name : names)
		{
			String nameVal = createName(name);
			if(StringUtils.isNotBlank(nameVal))
			{
				sb.append(nameVal).append(" ");
			}
		}

		return sb.toString();
	}

	public static String createName(final EN name)
	{
		if(name == null)
		{
			return null;
		}

		StringBuilder sb = new StringBuilder();
		if(StringUtils.isNotBlank(name.getText()))
		{
			return name.getText();
		}

		if(StringUtils.isNotBlank(loopForName(name.getPrefixes())))
		{
			sb.append(loopForName(name.getPrefixes()));
			if(StringUtils.isNotBlank(loopForName(name.getGivens())))
			{
				sb.append(" ");
			}
		}

		if(StringUtils.isNotBlank(loopForName(name.getGivens())))
		{
			sb.append(loopForName(name.getGivens()));
			if(StringUtils.isNotBlank(loopForName(name.getFamilies())))
			{
				sb.append(" ");
			}
		}

		if(StringUtils.isNotBlank(loopForName(name.getFamilies())))
		{
			sb.append(loopForName(name.getFamilies()));
			if(StringUtils.isNotBlank(loopForName(name.getSuffixes())))
			{
				sb.append(" ");
			}
		}

		if(StringUtils.isNotBlank(loopForName(name.getSuffixes())))
		{
			sb.append(loopForName(name.getSuffixes()));
		}
		return sb.toString();
	}

	/**
	 * Handles null checking and escaping of values, for entering items into a
	 * map.
	 * 
	 * @param map
	 * @param key
	 * @param value
	 */
	public static void mapString(final Map<String, Object> map, final String key, final String value)
	{
		if(StringUtils.isNotBlank(value) && StringUtils.isNotBlank(key))
		{
			String tValue = StringEscapeUtils.unescapeXml(value);
			if(map.containsKey(key))
			{
				map.put(key, map.get(key) + ", " + StringEscapeUtils.escapeXml(tValue));
			}
			else
			{
				map.put(key, StringEscapeUtils.escapeXml(tValue));
			}
		}
	}

	/**
	 * An old way of handling content which needs to be encapsulated in a
	 * content block.
	 * 
	 * @param map
	 * @param key
	 * @param id
	 * @param value
	 */
	@Deprecated
	public static void mapReference(final Map<String, Object> map, final String key, final String id, final String value)
	{
		if(StringUtils.isNotBlank(key) && StringUtils.isNotBlank(id) && StringUtils.isNotBlank(value))
		{
			mapString(map, key, "<content ID=\"" + StringEscapeUtils.escapeXml(id) + "\">" + StringEscapeUtils.escapeXml(value) + "</content>");
		}
		
		logger.debug("This class is deprecated please update the caller.");
	}
}
