package gov.va.ccd.consol.component.impl;

import freemarker.template.Configuration;
import freemarker.template.Template;
import freemarker.template.TemplateException;
import gov.va.ccd.components.transform.Transformer;
import gov.va.ccd.consol.AbstractConsolComponent;
import gov.va.ccd.mapper.TemplateMapper;
import gov.va.ccd.service.util.Utils;
import gov.va.ccd.service.value.object.SectionValueObject;

import java.io.IOException;
import java.io.StringWriter;
import java.util.ArrayList;
import java.util.Collections;
import java.util.Comparator;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang3.StringUtils;
import org.joda.time.DateTimeComparator;
import org.openhealthtools.mdht.uml.cda.Act;
import org.openhealthtools.mdht.uml.cda.Component4;
import org.openhealthtools.mdht.uml.cda.Entry;
import org.openhealthtools.mdht.uml.cda.EntryRelationship;
import org.openhealthtools.mdht.uml.cda.Observation;
import org.openhealthtools.mdht.uml.cda.ObservationRange;
import org.openhealthtools.mdht.uml.cda.Organizer;
import org.openhealthtools.mdht.uml.cda.Procedure;
import org.openhealthtools.mdht.uml.cda.ReferenceRange;
import org.openhealthtools.mdht.uml.cda.consol.ContinuityOfCareDocument;
import org.openhealthtools.mdht.uml.cda.consol.ResultsSection;
import org.openhealthtools.mdht.uml.hl7.datatypes.ANY;
import org.openhealthtools.mdht.uml.hl7.datatypes.CD;
import org.openhealthtools.mdht.uml.hl7.datatypes.CE;
import org.openhealthtools.mdht.uml.hl7.datatypes.ED;
import org.openhealthtools.mdht.uml.hl7.datatypes.IVL_PQ;
import org.openhealthtools.mdht.uml.hl7.datatypes.PQ;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class ResultConsolComponent extends AbstractConsolComponent<ResultsSection>
{
	private static final Logger logger = LoggerFactory.getLogger(ResultConsolComponent.class);

	public ResultConsolComponent(ContinuityOfCareDocument ccd)
	{
		super(ccd.getResultsSection(), ccd);
	}

	@Override
	protected String generateXml()
	{
		if(values.isEmpty())
			return StringUtils.EMPTY;

		try
		{
			Configuration cfg = TemplateMapper.getInstance().getCfg();
			Template temp = cfg.getTemplate("Results.xml");
			/* Merge data-model with template */

			StringWriter out = new StringWriter();
			temp.process(values, out);
			return out.toString();
		}
		catch(IOException e)
		{
			logger.error("Error occurred", e);
		}
		catch(TemplateException e)
		{
			logger.error("Error occurred", e);
		}
		return StringUtils.EMPTY;
	}

	@Override
	protected void createObjects()
	{
		List<Map<String, Object>> panels = new ArrayList<Map<String, Object>>();

		for(Entry entry : section.getEntries())
		{

			// First shceck to see if this contains a Procedure block.
			if(entry.getProcedure() != null)
			{
				Map<String, Object> map = new HashMap<String, Object>();
				// Process any information in the procedure.
				Procedure proc = entry.getProcedure();
				Transformer.mapCode(map, "", proc.getCode(), narrativeText);
				// All values need to be handled seperatly.
				if(!map.isEmpty())
				{
					values.put("procedure", map);
				}
			}

			// Start Processing panels.
			if(entry.getOrganizer() != null)
			{
				panels.add(processOrganizer(entry.getOrganizer()));
			}
		}
		if(CollectionUtils.isNotEmpty(panels))
		{
			Collections.sort(panels, new Comparator<Map<String, Object>>()
			{

				public int compare(Map<String, Object> arg0, Map<String, Object> arg1)
				{
					DateTimeComparator comp = DateTimeComparator.getDateOnlyInstance();
					int val = comp.compare(arg0.get("convertedTime"), arg1.get("convertedTime"));
					if(val > 0)
					{
						return -1;
					}
					else if(val < 0)
					{
						return 1;
					}
					return val;
				}

			});

			values.put("panels", panels);
		}
	}

	private Map<String, Object> processOrganizer(final Organizer org)
	{
		if(org == null)
			return Collections.emptyMap();

		Map<String, Object> orgMap = new HashMap<String, Object>();

		Map<String, Object> organizer = new HashMap<String, Object>();
		Transformer.mapCode(organizer, "", org.getCode(), narrativeText);
		if(!organizer.isEmpty())
		{
			orgMap.put("panelName", organizer);
		}

		Transformer.mapEffectiveTime(orgMap, org.getEffectiveTime());

		Transformer.mapWrappedCode(orgMap, "", org.getStatusCode(), narrativeText);

		// Map Source info.
		String source = Utils.getSourceFromOrganizer(org);
		if(StringUtils.isNotBlank(source))
		{
			Transformer.mapString(orgMap, "source", source);
		}

		List<Map<String, Object>> tests = new ArrayList<Map<String, Object>>();

		for(Component4 comp : org.getComponents())
		{
			Observation obs = comp.getObservation();

			if(obs != null)
			{
				Map<String, Object> test = new HashMap<String, Object>();

				Map<String, Object> testName = new HashMap<String, Object>();
				Transformer.mapCode(testName, "", obs.getCode(), narrativeText);
				if(!testName.isEmpty())
				{
					test.put("testName", testName);
				}

				Map<String, Object> obsText = new HashMap<String, Object>();
				Transformer.mapOriginalText(obsText, "", obs.getText(), narrativeText);
				if(!obsText.isEmpty())
				{
					test.put("obsText", obsText);
				}

				Transformer.mapWrappedCode(test, "status", obs.getStatusCode(), narrativeText);

				if(!orgMap.containsKey("timeValue"))
				{
					Transformer.mapEffectiveTime(orgMap, obs.getEffectiveTime());
				}
				Transformer.mapEffectiveTime(test, obs.getEffectiveTime());
				// Find result value. May break out later if needed.
				List<Map<String, Object>> results = new ArrayList<Map<String, Object>>();

				for(ANY any : obs.getValues())
				{
					Map<String, Object> result = new HashMap<String, Object>();
					// Some values can be found in a TXT value. Others can be
					// found in a PQ value. Others will probably come later.
					if(any instanceof PQ)
					{
						PQ value = (PQ) any;
						Transformer.mapPQValue(result, "", value, narrativeText);
					}
					else if(any instanceof ED)
					{
						ED value = (ED) any;
						Transformer.mapEDValue(result, "", value, narrativeText);
					}
					else if(any instanceof CD)
					{
						CD value = (CD) any;
						Transformer.mapCode(result, "", value, narrativeText);
						result.put("value", "ValueIsCode");
					}
					if(!result.isEmpty())
					{
						results.add(result);
					}
				}
				if(CollectionUtils.isNotEmpty(results))
				{
					test.put("results", results);
				}

				List<Map<String, Object>> interps = new ArrayList<Map<String, Object>>();
				for(CE code : obs.getInterpretationCodes())
				{
					Map<String, Object> interp = new HashMap<String, Object>();
					Transformer.mapCode(interp, "", code, narrativeText);
					if(!interp.isEmpty())
					{
						interps.add(interp);
					}
				}
				if(CollectionUtils.isNotEmpty(interps))
				{
					test.put("interp", interps);
				}

				List<Map<String, Object>> comments = new ArrayList<Map<String, Object>>();
				// See if there are any comments associated with this element.
				for(EntryRelationship rel : obs.getEntryRelationships())
				{
					Act act = rel.getAct();
					if(act != null)
					{
						Map<String, Object> comment = new HashMap<String, Object>();
						// Make sure this is a comment.
						if("48767-8".equalsIgnoreCase(act.getCode().getCode()))
						{
							Transformer.mapOriginalText(comment, "", act.getText(), narrativeText);
							Transformer.mapCode(comment, "", act.getStatusCode(), narrativeText);
						}
						if(!comment.isEmpty())
						{
							comments.add(comment);
						}
					}
				}
				if(CollectionUtils.isNotEmpty(comments))
				{
					test.put("comments", comments);
				}

				// Start reference range.
				List<Map<String, Object>> ranges = new ArrayList<Map<String, Object>>();
				for(ReferenceRange refrange : obs.getReferenceRanges())
				{
					Map<String, Object> range = new HashMap<String, Object>();
					if(refrange.getObservationRange() != null)
					{
						// Range can be either in original text or in a value
						// component.
						ObservationRange obsRan = refrange.getObservationRange();
						Transformer.mapOriginalText(range, "", refrange.getObservationRange().getText(), narrativeText);
						if(obsRan.getValue() != null)
						{
							if(obsRan.getValue() instanceof IVL_PQ)
							{
								Transformer.mapIVL_PQValue(range, "", (IVL_PQ) obsRan.getValue(), narrativeText);
							}
						}
						if(!range.isEmpty())
						{
							ranges.add(range);
						}
					}
				}
				if(CollectionUtils.isNotEmpty(ranges))
				{
					test.put("ranges", ranges);
				}

				// Map Source info.
				String testSource = Utils.getSourceFromAuthors(obs.getAuthors());
				if(StringUtils.isBlank(testSource))
				{
					testSource = Utils.getSourceFromPerformer(obs.getPerformers());
				}
				if(StringUtils.isBlank(testSource))
				{
					testSource = Utils.getSourceFromScoping(obs.getParticipants());
				}

				if(StringUtils.isNotBlank(testSource))
				{
					Transformer.mapString(test, "source", testSource);
				}

				if(!test.isEmpty())
				{
					tests.add(test);
				}
			}
		}
		if(CollectionUtils.isNotEmpty(tests))
		{
			orgMap.put("tests", tests);
		}
		return orgMap;
	}

	@Deprecated
	public List<SectionValueObject> getSectionData()
	{
		throw new UnsupportedOperationException("Get Section data is for legacy code.");
	}

}
