package gov.va.ccd.components.impl;

import java.io.IOException;
import java.io.StringWriter;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang3.StringEscapeUtils;
import org.apache.commons.lang3.StringUtils;
import org.eclipse.emf.common.util.EList;
import org.openhealthtools.mdht.uml.cda.Act;
import org.openhealthtools.mdht.uml.cda.AssignedEntity;
import org.openhealthtools.mdht.uml.cda.Entry;
import org.openhealthtools.mdht.uml.cda.EntryRelationship;
import org.openhealthtools.mdht.uml.cda.Organization;
import org.openhealthtools.mdht.uml.cda.Participant2;
import org.openhealthtools.mdht.uml.cda.ParticipantRole;
import org.openhealthtools.mdht.uml.cda.Performer2;
import org.openhealthtools.mdht.uml.cda.Person;
import org.openhealthtools.mdht.uml.cda.PlayingEntity;
import org.openhealthtools.mdht.uml.cda.ccd.ContinuityOfCareDocument;
import org.openhealthtools.mdht.uml.cda.ccd.PayersSection;
import org.openhealthtools.mdht.uml.hl7.datatypes.AD;
import org.openhealthtools.mdht.uml.hl7.datatypes.ADXP;
import org.openhealthtools.mdht.uml.hl7.datatypes.CD;
import org.openhealthtools.mdht.uml.hl7.datatypes.ENXP;
import org.openhealthtools.mdht.uml.hl7.datatypes.II;
import org.openhealthtools.mdht.uml.hl7.datatypes.ON;
import org.openhealthtools.mdht.uml.hl7.datatypes.PN;
import org.openhealthtools.mdht.uml.hl7.datatypes.TEL;
import org.openhealthtools.mdht.uml.hl7.vocab.x_ActClassDocumentEntryAct;
import org.openhealthtools.mdht.uml.hl7.vocab.x_ActRelationshipEntryRelationship;
import org.openhealthtools.mdht.uml.hl7.vocab.x_DocumentActMood;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import freemarker.template.Configuration;
import freemarker.template.Template;
import freemarker.template.TemplateException;
import gov.va.ccd.components.enums.PayerFields;
import gov.va.ccd.mapper.TemplateMapper;

/**
 * @author ChmaraC
 * 
 */
public class PayersComponent extends AbstractComponent<PayersSection>
{
	private static final Logger LOG = LoggerFactory.getLogger(PayersComponent.class);

	public PayersComponent(final ContinuityOfCareDocument ccdDocument)
	{
		super(ccdDocument.getPayersSection(), ccdDocument);
	}

	@Override
	protected String generateXml()
	{
		if(values.isEmpty())
			return StringUtils.EMPTY;

		try
		{
			Configuration cfg = TemplateMapper.getInstance().getCfg();
			Template temp = cfg.getTemplate("Payers.xml");
			/* Merge data-model with template */

			StringWriter out = new StringWriter();
			temp.process(values, out);
			return out.toString();
		}
		catch(IOException e)
		{
			// CCR 177986
			LOG.error("Error occurred {}:", e);
		}
		catch(TemplateException e)
		{
			// CCR 177986
			LOG.error("Error occurred {}:", e);
		}
		return StringUtils.EMPTY;
	}

	@Override
	protected void createObjects()
	{
		if(section == null)
			return;
		List<Map<String, Object>> entries = new ArrayList<Map<String, Object>>();
		for(Entry entry : section.getEntries())
		{

			for(EntryRelationship entryRelationship : entry.getAct().getEntryRelationships())
			{
				Map<String, Object> map = new HashMap<String, Object>();
				populateActInformation(map, entryRelationship.getAct());

				entries.add(map);
			}
		}

		if(CollectionUtils.isNotEmpty(entries))
		{
			values.put("values", entries);
		}
	}

	private void populateActInformation(final Map<String, Object> map, final Act act)
	{
		for(II id : act.getIds())
		{
			populateString(map, PayerFields.GROUP_NUMBER.getKey(), id.getExtension());
		}
		populateInsuranceType(map, act);
		populateParticipantInfo(map, act);
		populatePerformerInfo(map, act);
		populatePlanEntryRel(map, act);
		populateGuarInfo(map, act);
	}

	private void populateGuarInfo(Map<String, Object> map, Act act)
	{
		for(Performer2 performer : act.getPerformers())
		{
			AssignedEntity assigned = performer.getAssignedEntity();

			if(assigned == null)
			{
				return;
			}

			if(assigned.getCode() != null && "GUAR".equals(assigned.getCode().getCode()))
			{

				if(performer.getTime() != null)
				{
					populateString(map, PayerFields.GUAR__START.getKey(), performer.getTime().getValue());
				}

				for(AD addr : assigned.getAddrs())
				{
					populateString(map, PayerFields.GUAR_ADDR.getKey(), createAddress(addr));
				}

				for(TEL phone : assigned.getTelecoms())
				{
					populateString(map, PayerFields.GUAR_PHONE.getKey(), phone.getValue());
				}

				Person entity = assigned.getAssignedPerson();
				if(entity != null)
				{
					for(PN name : entity.getNames())
					{
						populateString(map, PayerFields.GUAR_NAME.getKey(), createName(name));
					}
				}

				for(Organization org : assigned.getRepresentedOrganizations())
				{
					for(ON orgname : org.getNames())
					{
						populateString(map, PayerFields.GUAR_NAME.getKey(), orgname.getText());
					}
				}
			}
		}

	}

	private void populatePlanEntryRel(Map<String, Object> map, Act act)
	{
		for(EntryRelationship entryrel : act.getEntryRelationships())
		{
			if(entryrel.getTypeCode() == x_ActRelationshipEntryRelationship.REFR)
			{
				Act relact = entryrel.getAct();
				if(relact.getClassCode() == x_ActClassDocumentEntryAct.ACT && relact.getMoodCode() == x_DocumentActMood.DEF)
				{
					populateString(map, PayerFields.PLAN_NAME.getKey(), relact.getText().getText());
				}
			}
		}

	}

	private void populateInsuranceType(final Map<String, Object> map, final Act act)
	{
		/**
		 * Business Rules for Insurance type. 1) displayname 2) originalText 3)
		 * translation
		 */
		if(StringUtils.isNotBlank(act.getCode().getDisplayName()))
		{
			populateString(map, PayerFields.INSURANCE_TYPE.getKey(), act.getCode().getDisplayName());
		}
		else if(act.getCode().getOriginalText() != null && StringUtils.isNotBlank(act.getCode().getOriginalText().getText()))
		{
			populateString(map, PayerFields.INSURANCE_TYPE.getKey(), act.getCode().getOriginalText().getText());
		}
		else if(CollectionUtils.isNotEmpty(act.getCode().getTranslations()))
		{
			for(CD ed : act.getCode().getTranslations())
			{
				if(StringUtils.isNotBlank(ed.getOriginalText().getText()))
				{
					populateString(map, PayerFields.INSURANCE_TYPE.getKey(), ed.getOriginalText().getText());
				}
			}
		}
	}

	private void populateParticipantInfo(final Map<String, Object> map, final Act act)
	{
		for(Participant2 participant : act.getParticipants())
		{
			ParticipantRole role = participant.getParticipantRole();
			PlayingEntity entity = role.getPlayingEntity();

			switch(participant.getTypeCode())
			{
			case COV:
				// Process the covered patient.
				if(participant.getTime() != null && participant.getTime().getLow() != null && StringUtils.isNotBlank(participant.getTime().getLow().getValue()))
				{
					populateString(map, PayerFields.COVERED_START.getKey(), participant.getTime().getLow().getValue());
				}
				if(participant.getTime() != null && participant.getTime().getHigh() != null && StringUtils.isNotBlank(participant.getTime().getHigh().getValue()))
				{
					populateString(map, PayerFields.COVERED_END.getKey(), participant.getTime().getHigh().getValue());
				}

				for(II id : role.getIds())
				{
					populateString(map, PayerFields.COVERED_ID.getKey(), id.getExtension());
				}

				if(role.getCode() != null)
				{
					populateString(map, PayerFields.COVERED_RELATIONSHIP.getKey(), role.getCode().getDisplayName());
				}
				for(AD address : role.getAddrs())
				{
					populateString(map, PayerFields.COVERED_ADDR.getKey(), createAddress(address));
				}

				for(TEL tel : role.getTelecoms())
				{
					populateString(map, PayerFields.COVERED_PHONE.getKey(), tel.getValue());
				}

				if(entity != null)
				{
					for(PN name : entity.getNames())
					{
						populateString(map, PayerFields.COVERED_NAME.getKey(), createName(name));
					}

					if(entity.getSDTCBirthTime() != null)
					{
						populateString(map, PayerFields.COVERED_DOB.getKey(), entity.getSDTCBirthTime().getValue());
					}

					if(entity.getCode() != null)
					{
						populateString(map, PayerFields.RESPONSIBLE_PARTY_TYPE.getKey(), entity.getCode().getDisplayName());
					}
				}
				break;
			case HLD:
				// Process the holder of the policy.
				for(II id : role.getIds())
				{
					populateString(map, PayerFields.SUBSCRIBER_ID.getKey(), id.getExtension());
				}

				for(AD address : role.getAddrs())
				{
					populateString(map, PayerFields.SUBSCRIBER_ADDR.getKey(), createAddress(address));
				}

				for(TEL tel : role.getTelecoms())
				{
					populateString(map, PayerFields.SUBSCRIBER_PHONE.getKey(), tel.getValue());
				}

				for(PN name : entity.getNames())
				{
					populateString(map, PayerFields.SUBSCRIBER_NAME.getKey(), createName(name));
				}

				if(entity.getSDTCBirthTime() != null)
				{
					populateString(map, PayerFields.SUBSCRIBER_DOB.getKey(), entity.getSDTCBirthTime().getValue());
				}

				break;
			default:
				break;

			}
		}
	}

	private void populatePerformerInfo(final Map<String, Object> map, final Act act)
	{
		for(Performer2 performer : act.getPerformers())
		{

			AssignedEntity assigned = performer.getAssignedEntity();
			for(AD addr : assigned.getAddrs())
			{
				if(StringUtils.isNotBlank(createAddress(addr)))
					populateString(map, PayerFields.SOURCE_ADDR.getKey(), createAddress(addr));
			}

			for(TEL tel : assigned.getTelecoms())
			{
				populateString(map, PayerFields.SOURCE_PHONE.getKey(), tel.getValue());
			}

			for(Organization ro : assigned.getRepresentedOrganizations())
			{
				// cda:id AKA Group Number.
				for(II id : ro.getIds())
				{
					if(StringUtils.isNotBlank(id.getExtension()))
					{
						populateString(map, PayerFields.GROUP_NUMBER.getKey(), id.getExtension());
					}
				}
				for(ON name : ro.getNames())
				{
					if(StringUtils.isNotBlank(name.getText()))
					{
						populateString(map, PayerFields.SOURCE_NAME.getKey(), name.getText());
					}
				}
			}
		}
	}

	private String createAddress(final AD address)
	{
		StringBuilder sb = new StringBuilder();
		for(ADXP street : address.getStreetAddressLines())
		{
			if(StringUtils.isNotBlank(street.getText()))
			{
				sb.append(street.getText()).append(" ");
			}
		}

		for(ADXP city : address.getCities())
		{
			if(StringUtils.isNotBlank(city.getText()))
			{
				sb.append(city.getText()).append(", ");
			}
		}

		for(ADXP state : address.getStates())
		{
			if(StringUtils.isNotBlank(state.getText()))
			{
				sb.append(state.getText()).append(" ");
			}
		}

		for(ADXP zip : address.getPostalCodes())
		{
			if(StringUtils.isNotBlank(zip.getText()))
			{
				sb.append(zip.getText());
			}
		}

		return sb.toString();
	}

	private String loopForName(final EList<ENXP> names)
	{
		for(ENXP name : names)
		{
			if(StringUtils.isNotBlank(name.getText()))
			{
				return name.getText();
			}
		}

		return null;
	}

	private String createName(final PN name)
	{
		StringBuilder sb = new StringBuilder();

		if(StringUtils.isNotBlank(loopForName(name.getPrefixes())))
		{
			sb.append(loopForName(name.getPrefixes())).append(" ");
		}

		if(StringUtils.isNotBlank(loopForName(name.getGivens())))
		{
			sb.append(loopForName(name.getGivens())).append(" ");
		}

		if(StringUtils.isNotBlank(loopForName(name.getFamilies())))
		{
			sb.append(loopForName(name.getFamilies())).append(" ");
		}

		if(StringUtils.isNotBlank(loopForName(name.getSuffixes())))
		{
			sb.append(loopForName(name.getSuffixes())).append(" ");
		}
		return sb.toString();
	}

	private void populateString(final Map<String, Object> map, final String key, final String str)
	{
		if(StringUtils.isBlank(key))
			return;
		if(StringUtils.isNotBlank(str))
		{
			map.put(key, StringEscapeUtils.escapeHtml4(str));
		}
	}
}
