package gov.va.soapui;

import org.apache.ws.security.WSSConfig;
import org.apache.ws.security.WSSecurityException;
import org.apache.ws.security.message.WSSecBase;
import org.apache.ws.security.message.WSSecHeader;
import org.apache.ws.security.message.WSSecSAMLToken;
import org.apache.ws.security.saml.ext.AssertionWrapper;
import org.apache.ws.security.util.WSSecurityUtil;

import org.w3c.dom.Document;
import org.w3c.dom.Element;

/**
 * Builds a WS SAML Assertion and inserts it into the SOAP Envelope. Refer to
 * the WS specification, SAML Token profile. Overrides for use with CONNECT.
 */
public class CONNECTWSSecSAMLToken extends WSSecBase {

    private static org.apache.commons.logging.Log log = 
        org.apache.commons.logging.LogFactory.getLog(WSSecSAMLToken.class);
    
    private Document document = null;
    
    private CONNECTAssertionWrapper saml = null;
    
    private Element samlElement;

    public CONNECTWSSecSAMLToken() {
        super();
    }
    public CONNECTWSSecSAMLToken(WSSConfig config) {
        super(config);
    }
    
    /**
     * Creates a SAML token.
     * 
     * The method prepares and initializes a WSSec UsernameToken structure after
     * the relevant information was set. A Before calling
     * <code>prepare()</code> all parameters such as user, password,
     * passwordType etc. must be set. A complete <code>UsernameToken</code> is
     * constructed.
     * 
     * @param doc
     *            The SOAP envelope as W3C document
     */
    public void prepare(Document doc, CONNECTAssertionWrapper assertion) {
        document = doc;
        saml = assertion;
    }

    /**
     * Prepends the SAML Assertion to the elements already in the
     * Security header.
     * 
     * The method can be called any time after <code>prepare()</code>.
     * This allows to insert the SAML assertion at any position in the
     * Security header.
     * 
     * @param secHeader
     *            The security header that holds the Signature element.
     */
    public void prependToHeader(WSSecHeader secHeader) {
        try {
            Element element = getElement();
            if (element != null) {
                WSSecurityUtil.prependChildElement(secHeader.getSecurityHeader(), element);
            }
        } catch (WSSecurityException ex) {
            throw new RuntimeException(ex.toString(), ex);
        }
    }
    
    public Element getElement() throws WSSecurityException {
        if (samlElement != null) {
            return samlElement;
        }
        if (saml == null) {
            return null;
        }
        samlElement = saml.toDOM(document);
        return samlElement;
    }
    
    /**
     * Get the id generated during <code>prepare()</code>.
     * 
     * Returns the the value of wsu:Id attribute of this Timestamp. 
     * 
     * @return Return the wsu:Id of this token or null if <code>prepareToken()</code>
     * was not called before.
     */
    public String getId() {
        if (saml == null) {
            return null;
        }
        return saml.getId();
    }
    
    /**
     * Adds a new <code>SAMLAssertion</code> to a soap envelope.
     * <p/>
     * A complete <code>SAMLAssertion</code> is added to the
     * <code>wsse:Security</code> header.
     *
     * @param doc      The SOAP envelope as W3C document
     * @param assertion TODO
     * @return Document with UsernameToken added
     */
    public Document build(Document doc, CONNECTAssertionWrapper assertion, WSSecHeader secHeader) {
        log.debug("Begin add SAMLAssertion token...");
        
        prepare(doc, assertion);
        prependToHeader(secHeader);
        
        return doc;
    }
}
