package gov.va.med.nhin.adapter.web.audit;

import gov.va.med.nhin.adapter.audit.BulkExportLib;
import gov.va.med.nhin.adapter.propertylookup.PropertyLookup;
import gov.va.med.nhin.adapter.propertylookup.PropertyLookupLocal;
import java.io.File;
import java.io.IOException;
import java.io.OutputStream;
import java.sql.SQLException;
import java.text.ParseException;
import java.util.Date;
import javax.annotation.Resource;
import javax.ejb.EJB;
import javax.servlet.ServletException;
import javax.servlet.http.HttpServlet;
import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;
import javax.sql.DataSource;
import org.apache.commons.lang3.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import weblogic.security.internal.SerializedSystemIni;
import weblogic.security.internal.encryption.ClearOrEncryptedService;
import weblogic.security.internal.encryption.EncryptionService;

/**
 * ReST interface to download a copy (in zip form) of documents that were
 * captured in the audit log as being downloaded through adapter (disclosures
 * and receipts).
 *
 * @author cbarber
 */
public class BulkDownloadDocsServlet extends HttpServlet
{
   private static final Logger LOGGER = LoggerFactory.getLogger(BulkDownloadDocsServlet.class);
 
   public static final String PATIENT_IDENTIFIER = "patientIdentifier";
   public static final String START_PARAM = "start";
   public static final String END_PARAM = "end";
   public static final String CREATOR_ORG_PARAM = "creator";
   private static final String VAP_ID_PROP = "vap.identifier";
   private static final String RQST_HDR_VAP_ID =  "VAP_ID";

   @Resource(name = "jdbc/adapter")
   DataSource ds;

   @EJB(beanInterface = PropertyLookupLocal.class, beanName = "PropertyFileLookup")
   private PropertyLookup propLookup;
   
   //***
   // Set by init()
   //***
   private String vapId;
   private ClearOrEncryptedService encSvc;

   @Override
   public void init() throws ServletException
   {
      super.init();
      vapId = propLookup.getProperty(VAP_ID_PROP);
      if(vapId == null)
      {
         throw new ServletException(
            "Missing " + VAP_ID_PROP + " property.");
      }

      // works off convention that the current working directory is the domain home
      String domainHome = new File("dummy").getAbsoluteFile().getParent();
      EncryptionService encryptSrv = SerializedSystemIni.getEncryptionService(domainHome);
      encSvc = new ClearOrEncryptedService(encryptSrv);
   }
    
   @Override
   protected void doGet(HttpServletRequest req, HttpServletResponse resp) throws ServletException, IOException
   {
      if(!checkClientIdentity(req))
      {
          resp.sendError(HttpServletResponse.SC_UNAUTHORIZED);
          return;
      }

	  String patientIdentifier = req.getParameter(PATIENT_IDENTIFIER);
      String startAudit = req.getParameter(START_PARAM);
      String endAudit = req.getParameter(END_PARAM);
      String creatorOrgId = req.getParameter(CREATOR_ORG_PARAM);
      
      if(StringUtils.isBlank(startAudit) || StringUtils.isBlank(endAudit))
      {
         resp.sendError(HttpServletResponse.SC_BAD_REQUEST,
            "Start and end audit dates must both be specified.");
         return;
      }
      
      Date startAuditDate, endAuditDate;
      try
      {
         startAuditDate = BulkExportLib.parseRangeDateStr(startAudit);
         endAuditDate = BulkExportLib.parseRangeDateStr(endAudit);
      }
      catch(ParseException ex)
      {
         resp.sendError(HttpServletResponse.SC_BAD_REQUEST, ex.getMessage());
         return;
      }

      resp.setContentType("application/zip");
      resp.setHeader("Content-Disposition", "attachment; filename=auditedDocsExport.zip");
      OutputStream out = resp.getOutputStream();
      BulkExportLib bulkExport = new BulkExportLib(ds, true /* doZip */);
      try
      {
         bulkExport.exportRetrievedDocs(
            startAuditDate, endAuditDate, creatorOrgId, patientIdentifier, null /* destinationRoot*/, out);
      }
      catch (SQLException ex)
      {
         LOGGER.error("Database error during bulk export.", ex);
         if(!resp.isCommitted())
         {
            resp.sendError(HttpServletResponse.SC_INTERNAL_SERVER_ERROR,
               "Unable to perform the document export due to a database error.");
         }
      }
   }
   
   private boolean checkClientIdentity(HttpServletRequest rqst)
   {
      String rqstVapId = rqst.getHeader(RQST_HDR_VAP_ID);
      String vapIdClear = encSvc.decrypt(vapId);
      return rqstVapId != null && rqstVapId.equals(vapIdClear);
   }
}
