package gov.va.med.nhin.adapter.audit;

import static org.junit.Assert.assertEquals;
import static org.junit.Assert.assertNotNull;

import java.io.BufferedWriter;
import java.io.File;
import java.io.FileInputStream;
import java.io.FileNotFoundException;
import java.io.FileOutputStream;
import java.io.FileWriter;
import java.io.IOException;
import java.io.InputStream;
import java.sql.SQLException;
import java.text.DateFormat;
import java.text.ParseException;
import java.text.SimpleDateFormat;
import java.util.Date;
import java.util.HashSet;
import java.util.Set;
import java.util.zip.ZipOutputStream;

import org.junit.BeforeClass;
import org.junit.Test;

public class BulkExportLibTest {
	public static final String IMPORT_FILE_NAME = "/src/test/resources/ImportSample.txt";

	public static final String EXPORT_FILE_NAME = "/src/test/resources/ExportSample.txt";
	
	public static final String ZIP_OUTPUT_FILE_NAME = "/src/test/resources/ImportZipSample.zip";

	@BeforeClass
	/**
	 * Test setup method
	 * 
	 * Method used to create test data and files for unit testing.
	 * 
	 */
	public static void setup() {
		System.out.println(
				"Executing BulkExportLibTest running test setup ................................................");

		File testSetupFile = null;

		try {
			// Get the current directory
			String current = new java.io.File(".").getCanonicalPath();

			System.out.println("Current dir:" + current);

			String currentDir = System.getProperty("user.dir");

			System.out.println("Current dir using System:" + currentDir);

			BufferedWriter bufferedWriter = null;

			FileWriter fileWriter = null;

			// Remove old export files
			testSetupFile = new File(currentDir + EXPORT_FILE_NAME);

			if (testSetupFile.exists())
			{
				System.out.println("Deleted export file "+testSetupFile.getName()+"? "+testSetupFile.delete());
			}


			// Remove old import files
			testSetupFile = new File(currentDir + IMPORT_FILE_NAME);

			if (testSetupFile.exists())
			{
				System.out.println("Deleted import file "+testSetupFile.getName()+"? "+testSetupFile.delete());
			}
			
			System.out.println("Creating new import file");
			
			DateFormat dateFormat = new SimpleDateFormat("yyyy/MM/dd HH:mm:ss");
			
	        Date date = new Date();
	         
			String content = "This is the content of the new import file generated by the BulkExportLibTest unit test on "+ dateFormat.format(date)+"\n";
		
			fileWriter = new FileWriter(currentDir + IMPORT_FILE_NAME);
			
			bufferedWriter = new BufferedWriter(fileWriter);
			
			bufferedWriter.write(content);
			
			bufferedWriter.flush();
			
			bufferedWriter.close();
			
			
			testSetupFile = new File(currentDir + ZIP_OUTPUT_FILE_NAME);

			if (testSetupFile.exists())
			{
				System.out.println("Deleted output zip file "+testSetupFile.getName()+"? "+testSetupFile.delete());
			}

		} catch (Exception e) {
			// if any error occurs
			e.printStackTrace();
		}

		System.out.println("Setup complete ................................................");

	}

	@Test
	/**
	 * testMapToDocumentTypeName Method
	 * 
	 * Method to test BulkExportLib static method mapToDocumentTypeName.
	 * 
	 */
	public void testMapToDocumentTypeName() {
		System.out.println("Executing TestMapToDocumentTypeName................................................");

		assertEquals("UNK-DOC-TYPE", BulkExportLib.mapToDocumentTypeName("", ""));

		assertEquals("C32", BulkExportLib.mapToDocumentTypeName("34133-9", "urn:ihe:pcc:xphr:2007"));

		assertEquals("CCDACCD",
				BulkExportLib.mapToDocumentTypeName("34133-9", "urn:hl7-org:sdwg:ccda-structuredBody:1.1"));

		assertEquals("CCDACCD", BulkExportLib.mapToDocumentTypeName("34133-9", "urn:hl7-org:sdwg:CCDA:1.1"));

		assertEquals("CCDACCD",
				BulkExportLib.mapToDocumentTypeName("34133-9", "urn:hl7-org:sdwg:ccda-structuredBody:2.1"));

		assertEquals("CCDAPN",
				BulkExportLib.mapToDocumentTypeName("11506-3", "urn:hl7-org:sdwg:ccda-structuredBody:1.1"));

		assertEquals("CCDAPN",
				BulkExportLib.mapToDocumentTypeName("11506-3", "urn:hl7-org:sdwg:ccda-structuredBody:2.1"));

        // spec said "multiple" LOINCs allowed, but didn't say null allowed
		assertEquals("UNK-DOC-TYPE", BulkExportLib.mapToDocumentTypeName(null, "urn:ihe:iti:xds-sd:pdf:2008"));

		assertEquals("UNK-DOC-TYPE", BulkExportLib.mapToDocumentTypeName(null, "urn:ihe:iti:xds-sd:text:2008"));

		assertEquals("C62PDF", BulkExportLib.mapToDocumentTypeName("any loinc", "urn:ihe:iti:xds-sd:pdf:2008"));

		assertEquals("C62TXT", BulkExportLib.mapToDocumentTypeName("any loinc", "urn:ihe:iti:xds-sd:text:2008"));

        System.out.println("TestMapToDocumentTypeName complete ................................................");
	}

	@Test
	/**
	 * testParseRangeDateStr Method
	 * 
	 * Method to test BulkExportLib static method parseRangeDateStr.
	 * 
	 */
	public void testParseRangeDateStr() {
		System.out.println("Executing TestParseRangeDateStr ................................................");

		try {
			assertNotNull(BulkExportLib.parseRangeDateStr("2017-01-27"));
		} catch (ParseException e) {
			// TODO Auto-generated catch block
			e.printStackTrace();
		}

		System.out.println("TestParseRangeDateStr complete ................................................");
	}

	@Test
	/**
	 * testExportFile Method
	 * 
	 * Method to test BulkExportLib static method exportFile.
	 * 
	 */
	public void testExportFile() {
		System.out.println("Executing TestExportFile ................................................");
		
		String currentDir = System.getProperty("user.dir");

		File importFile = new File(currentDir+IMPORT_FILE_NAME);

		File exportFile = new File(currentDir+EXPORT_FILE_NAME);

		InputStream importStream;

		try {
			if (importFile.exists() && !exportFile.exists())
			{
				System.out.println(importFile.getName() +" does exist and " + exportFile.getName()+" does not exist. Starting export.");
				
				importStream = new FileInputStream(importFile);
				
				BulkExportLib.exportFile(exportFile, "", importStream);
				
				System.out.println("File export was successful. " + exportFile.getName()+" was created? " + exportFile.exists());
			}
			else
			{
				System.out.println("An error has occurred. Unable the perform export test. Please verify the import file was created and the export was removed during test initialization.");
			}

		} catch (FileNotFoundException e1) {
			// TODO Auto-generated catch block
			e1.printStackTrace();
		} catch (IOException | SQLException e) {
			// TODO Auto-generated catch block
			e.printStackTrace();
		}

		System.out.println("TestExportFile complete ................................................");
	}
	
	@Test
	/**
	 * testExportZipFile Method
	 * 
	 * Method to test BulkExportLib static method exportFile with Zip file.
	 * 
	 */
	public void testExportZipFile()
	{
		System.out.println("Executing TestExportZipFile ................................................");
		
		String currentDir = System.getProperty("user.dir");

		File importFile = new File(currentDir+IMPORT_FILE_NAME);

		File zipOutputFile = new File(currentDir+ZIP_OUTPUT_FILE_NAME);

		InputStream importStream;

		try {
	        ZipOutputStream zipOutputStream = new ZipOutputStream(new FileOutputStream(new File(currentDir, ZIP_OUTPUT_FILE_NAME)));
	        
			if (importFile.exists())
			{
				System.out.println(importFile.getName() +" does exist and " + zipOutputFile.getName()+" does not exist. Starting export.");
				
				importStream = new FileInputStream(importFile);
				
				Set<String> pathSet = new HashSet<String>();
				
				pathSet.add(currentDir);

				BulkExportLib.exportFile(zipOutputStream,pathSet, "", importStream);
				
				System.out.println("File export was successful. " + zipOutputFile.getName()+" was created? " + zipOutputFile.exists());
			}
			else
			{
				System.out.println("An error has occurred. Unable the perform export test. Please verify the import file was created and the export was removed during test initialization.");
			}

		} catch (FileNotFoundException e1) {
			// TODO Auto-generated catch block
			e1.printStackTrace();
		} catch (IOException | SQLException e) {
			// TODO Auto-generated catch block
			e.printStackTrace();
		}
		
		System.out.println("TestExportZipFile complete ................................................");
	}
}
