package gov.va.med.nhin.adapter.audit;

import java.io.Console;
import java.io.File;
import java.io.FileInputStream;
import java.io.IOException;
import java.io.StringWriter;
import java.util.Properties;

import org.apache.commons.io.IOUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import gov.va.med.nhin.adapter.utils.DatabaseUtils;
import oracle.jdbc.pool.OracleDataSource;

public class BulkExportMain
{
	public static final String NL = System.getProperty("line.separator");
	public static final String USAGE = "Usage:" + NL +
			BulkExportMain.class.getName() + " {properties file}"; 
   public static final String PROP_DATABASE_URL = "database.url";
   public static final String PROP_DATABASE_USER = "database.user";
   public static final String PROP_ZIP_OUTPUT = "zip.output";

   private static final Logger LOGGER = LoggerFactory.getLogger(BulkExportMain.class);

   public static void main(String[] args)
   {
   	if(args.length != 1)
   	{
   		LOGGER.error(USAGE);
   		return;
   	}
   	
   	String dbPassword;
   	Console console = System.console();
   	try
   	{
	   	if(console == null)
	   	{
	   		// read from standard in
	   		StringWriter inputHolder = new StringWriter();
	   		if(System.in.available() > 0)
	   		{
	   			IOUtils.copy(System.in, inputHolder);
	   		}
   			dbPassword = inputHolder.toString();
   			if(dbPassword.endsWith(NL))
   			{
   				dbPassword = dbPassword.substring(0, dbPassword.length() - NL.length());
   			}

   			if(dbPassword.length() == 0)
	   		{
	   			LOGGER.error("The database password must be piped on stdin for non-interactive console usage.");
	   			return;
	   		}
	   	}
	   	else
	   	{
	   		char[] dbPasswdChars = console.readPassword("Database Password: ");
	   		if(dbPasswdChars == null || dbPasswdChars.length == 0)
	   		{
	   			LOGGER.error("No password given; exiting.");
	   			return;
	   		}
	   		dbPassword = new String(dbPasswdChars);
	   	}
   	}
   	catch(IOException ex)
   	{
   		LOGGER.error("Reading of DB Password failed.", ex);
   		return;
   	}
   	
   	processDqExportProperties(new File(args[0]), dbPassword);
   }

   /**
    * Sample file contents:
    * start.audit.datetime = 2016-11-01T00:00
    * end.audit.datetime = 2016-12-01T00:00
    * export.root.dir = /opt/dqExport
    * @param exportPropsFile
    * @param dqExportInfo
    */
   public static void processDqExportProperties(File exportPropsFile, String dbPassword)
   {
   	FileInputStream fis = null;
   	Properties exportProps = new Properties();
   	try
   	{
   		fis = new FileInputStream(exportPropsFile);
   		exportProps.load(fis);
   	}
   	catch(Exception ex)
   	{
   		LOGGER.error("Reading of data quality bulk export properties file failed.", ex);
   		return;
   	}
   	finally
   	{
   		DatabaseUtils.closeIO(fis, LOGGER);
   	}
   	
	   String dbUrl = exportProps.getProperty(PROP_DATABASE_URL);
	   String dbUser = exportProps.getProperty(PROP_DATABASE_USER);
	   String zipOutStr = exportProps.getProperty(PROP_ZIP_OUTPUT);
	   boolean zipOut = (zipOutStr != null && zipOutStr.equalsIgnoreCase("true"));
		   
   	BulkExportLib exportLib;
   	try
   	{
   	   OracleDataSource orclDataSrc = new OracleDataSource();
		   orclDataSrc.setURL(dbUrl);
		   orclDataSrc.setUser(dbUser);
		   orclDataSrc.setPassword(dbPassword);
		   exportLib = new BulkExportLib(orclDataSrc, zipOut);
   	}
   	catch(Exception ex)
   	{
   		LOGGER.error("Creation of OracleDataSource failed.", ex);
   		return;
   	}

   	try
   	{
   		exportLib.exportRetrievedDocs(exportProps);
   	}
   	catch(Exception ex)
   	{
   		LOGGER.error("The export operation failed.", ex);
   	}
   }
}
