package gov.va.med.nhin.adapter.audit;

import java.io.BufferedOutputStream;
import java.io.File;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.sql.Blob;
import java.sql.Connection;
import java.sql.PreparedStatement;
import java.sql.ResultSet;
import java.sql.SQLException;
import java.sql.Timestamp;
import java.text.ParseException;
import java.text.SimpleDateFormat;
import java.util.Date;
import java.util.HashSet;
import java.util.Properties;
import java.util.Set;
import java.util.zip.ZipEntry;
import java.util.zip.ZipOutputStream;

import javax.sql.DataSource;

import org.apache.commons.io.FileUtils;
import org.apache.commons.io.IOUtils;
import org.apache.commons.lang3.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import gov.va.med.nhin.adapter.utils.DatabaseUtils;
import java.io.OutputStream;


public class BulkExportLib  
{
	public static final String YEAR_MONTH_DAY = "yyyyMMdd";
	public static final String DISCLOSURE_ACTION = "Retrieve Document";
   public static final String RETRIEVE_FROM_PARTNER_ACTION = "RetrieveDocumentOut";

   public static final String ORG_URI_SCHEME = "urn:oid:";
   // this is the real org ID, but in SQA environments, .1 or .2 can be
   // appended to make a unique community ID for testing
   public static final String VA_ORG_NUMBER_PREFIX = "2.16.840.1.113883.4.349";
   public static final String VA_LONG_NAME = "DEPARTMENT OF VETERANS AFFAIRS";

   public static final String PROP_START_AUDIT_DATETIME = "start.audit.datetime";
   public static final String PROP_END_AUDIT_DATETIME = "end.audit.datetime";
   public static final String PROP_EXPORT_ROOT_DIR = "export.root.dir";
   public static final String PROP_CREATOR_ORG_ID = "creator.org.id";

   /**
    * The main query.
    * <table border="1" cellpadding="3px" cellspacing="0">
    *  <tr style="font-weight: bold">
    *   <td>Action Type</td>
    *   <td>Logical Action</td>
    *   <td>ORGANIZATION_ID</td>
    *   <td>REMOTE_ORGANIZATION_ID</td>
    *   <td>Doc Creator</td>
    *  </tr>
    * <tr>
    *   <td>RetrieveDocumentOut</td>
    *   <td>VA Downloads *from* Partner (Receipt)</td>
    *   <td>Dept. of Vet Affairs</td>
    *   <td>Partner</td>
    *   <td>REMOTE_ORGANIZATION_ID</td>
    *  </tr>
    *  <tr>
    *   <td>Retrieve Document</td>
    *   <td>Partner Downloads *from* VA (Disclosure)</td>
    *   <td>Partner (Maybe Internal VA)</td>
    *   <td>NULL</td>
    *   <td>Dept. of Vet Affairs</td>
    *  </tr>
    * </table>
    */
   private static final String EXPORT_SQL =
      "SELECT ACTION_NAME, ORGANIZATION_ID /* all same for RetrieveDocumentOut (same initiator, DoVA) */, " +
      "REMOTE_ORGANIZATION_ID /* use for RetrieveDocumentOut, null for Retrieve Document */, " +
      "REMOTE_ORG_FAC.FACILITY_NAME AS REMOTE_ORG_FACILITY_NAME, " +
      "AUDIT_TIME, DOCUMENTS.DOCUMENT_ID INTERNAL_DOC_ID, CLASS_CODE, FORMAT_CODE, RAW_DATA " +
      "FROM AUDITS LEFT JOIN DOCUMENTS " +
      "ON DOCUMENTS.DOCUMENT_UNIQUE_ID = AUDITS.DOCUMENT_ID " +
      "LEFT JOIN FACILITIES REMOTE_ORG_FAC " +
      "ON AUDITS.REMOTE_ORGANIZATION_ID = REMOTE_ORG_FAC.FULL_HOME_COMMUNITY_ID " +      
      "WHERE AUDIT_TIME >= ? AND AUDIT_TIME < ? " +
      "AND ACTION_NAME IN ('RetrieveDocumentOut', 'Retrieve Document') " +
      "AND (1 = ? /* will be 1 iff creatorOrgId is null */ OR " +
      "(ACTION_NAME = 'RetrieveDocumentOut' AND REMOTE_ORGANIZATION_ID = ?) OR " +
      "(ACTION_NAME = 'Retrieve Document' AND 1 = ?)) " +
      "AND (1 = ? /* will be 1 iff patient_id is null */ OR " +
      "(AUDITS.PATIENT_ID = ?)) " + // will be 1 iff they want documents created by VA
      "ORDER BY AUDIT_TIME";
   
   private static final Logger LOGGER = LoggerFactory.getLogger(BulkExportLib.class);

   private final Date exportDate = new Date();
   private final DataSource dataSource;
   private final boolean doZip;

   public BulkExportLib(DataSource dataSource)
   {
      this.dataSource = dataSource;
      doZip = true;
   }

   public BulkExportLib(DataSource dataSource, boolean doZip)
   {
      this.dataSource = dataSource;
      this.doZip = doZip;
   }

   /**
    * Exports a copy of all documents that were either disclosed ("Retrieve Document" audit record)
    * or retrieved from a partner ("RetrieveDocumentOut" audit record).
    * @param startAuditedTime the beginning audited date + time to include, inclusive
    * @param endAuditedTime the ending audited date + time to include, exclusive
    * @param creatorOrgId hierarchal organization number, like {@link #VA_ORG_NUMBER}, without <code>urn:oid:</code> prefix
    * @param destination the root folder into which to export the document tree;
    * ignored when destinationStream is specified
    * @param destinationStream an open output stream to which the contents of a zip
    * file holding all of the documents will be written. This stream will be closed
    * at the end of this method. If destinationStream is null, destination
    * must be specified instead.
    * @throws SQLException 
    * @throws IOException 
    */
   public void exportRetrievedDocs(Date startAuditedTime, Date endAuditedTime,
           String creatorOrgId, String patientIdentifier, File destinationRoot, OutputStream destinationStream)
      throws IOException, SQLException
   {
      Connection conn = null;
      PreparedStatement stmt = null;
      ZipOutputStream zipOutStream = null;
      Set<String> pathsWritten = null;
      
      boolean doStream = destinationStream != null;
      boolean doZipNow = doStream || doZip;
      try
      {
      	conn = dataSource.getConnection();
         // minimize locking via forward-only + read only ResultSet
         stmt = conn.prepareStatement(EXPORT_SQL, ResultSet.TYPE_FORWARD_ONLY, ResultSet.CONCUR_READ_ONLY);
         stmt.setTimestamp(1, new Timestamp(startAuditedTime.getTime()));
         stmt.setTimestamp(2, new Timestamp(endAuditedTime.getTime()));
         stmt.setInt(3, StringUtils.isBlank(creatorOrgId) ? 1 : 0);
         stmt.setString(4, ORG_URI_SCHEME + creatorOrgId);
         boolean isFilterForVACreator = creatorOrgId != null &&
            creatorOrgId.startsWith(VA_ORG_NUMBER_PREFIX);
         stmt.setInt(5, isFilterForVACreator ? 1 : 0);
         stmt.setInt(6, StringUtils.isBlank(patientIdentifier) ? 1 : 0);
         stmt.setString(7, patientIdentifier);             
         
         ResultSet rst = stmt.executeQuery();
         
         // hold off on creating the zip file until the query runs OK
         if(doZipNow)
         {
            if(doStream)
            {
               zipOutStream = new ZipOutputStream(new BufferedOutputStream(destinationStream));
            }
            else
            {
               FileUtils.forceMkdir(destinationRoot);
               SimpleDateFormat sdf = new SimpleDateFormat(YEAR_MONTH_DAY);
               File zipFile = new File(destinationRoot, sdf.format(exportDate) + ".zip");
               zipOutStream = new ZipOutputStream(new BufferedOutputStream(new FileOutputStream(zipFile)));
            }
            pathsWritten = new HashSet<>();
         }

         while(rst.next())
         {
            String actionName = rst.getString("ACTION_NAME");
            Timestamp auditDateTime = rst.getTimestamp("AUDIT_TIME");
            long internalDocId = rst.getLong("INTERNAL_DOC_ID");
            String docClassCode = rst.getString("CLASS_CODE");
            String formatCode = rst.getString("FORMAT_CODE"); 
            Blob rawData = rst.getBlob("RAW_DATA");
            
            String docCreatorOrgId;
            String docCreatorName;
            if(DISCLOSURE_ACTION.equals(actionName))
            {
                // Doesn't really matter as of now, because docCreatorName
                // is not null. But just being defensively consistent.
            	docCreatorOrgId = ORG_URI_SCHEME + VA_ORG_NUMBER_PREFIX;
            	docCreatorName = VA_LONG_NAME;
            }
            else
            {
            	docCreatorOrgId = rst.getString("REMOTE_ORGANIZATION_ID");
            	docCreatorName = rst.getString("REMOTE_ORG_FACILITY_NAME");
            }
            
            // make sure docCreatorName is populated
            if(docCreatorName == null) {
            	docCreatorName = docCreatorOrgId == null ? "UNKNOWN ORG" : docCreatorOrgId;
            }
            String documentType = mapToDocumentTypeName(docClassCode, formatCode);
            String fileSubPath = buildFileSubPath(auditDateTime, docCreatorName, "" + internalDocId, documentType);

            if(doZipNow)
            {
            	exportFile(zipOutStream, pathsWritten, fileSubPath, rawData.getBinaryStream());
            }
            else
            {
            	exportFile(destinationRoot, fileSubPath, rawData.getBinaryStream());
            }
         }
         if(doZipNow)
         {
         	zipOutStream.flush();
         }
      }
      finally
      {
         DatabaseUtils.closeStatement(stmt, LOGGER);
         DatabaseUtils.closeConnection(conn, LOGGER);
         DatabaseUtils.closeIO(zipOutStream, LOGGER);
      }
   }

   public static Date parseRangeDateStr(String rangeDateStr) throws ParseException
   {
   	SimpleDateFormat dateOnly = new SimpleDateFormat("yyyy-MM-dd");
	   SimpleDateFormat dateTime = new SimpleDateFormat("yyyy-MM-dd'T'HH:mm");

	   Date rangeDate;
      try
      {
         if(rangeDateStr.contains("T"))
         {
            rangeDate = dateTime.parse(rangeDateStr);
         }
         else
         {
            rangeDate = dateOnly.parse(rangeDateStr);
         }
      }
      catch(ParseException ex)
      {
         throw new ParseException("Start date and end date must be of the " +
           "format 'yyyy-MM-dd' or 'yyyy-MM-dd'T'HH:mm'.", ex.getErrorOffset());
      }
      return rangeDate;
   }

   /**
    * Wrapper around {@link #exportRetrievedDocs(Date, Date, File, OutputStream)} that takes parameters as Strings.
    * @param startAudit
    * @param endAudit
    * @param creatorOrgId hierarchal organization number, like {@link #VA_ORG_NUMBER}, without <code>urn:oid:</code> prefix
    * @param destinationRoot
    * @throws ParseException
    * @throws SQLException 
    * @throws IOException 
    */
   public void exportRetrievedDocs(String startAudit, String endAudit, String creatorOrgId, String destinationRoot) throws ParseException, IOException, SQLException
   {
   	if(StringUtils.isBlank(startAudit) || StringUtils.isBlank(endAudit))
	   {
	   	throw new ParseException("Start and end audit dates must both be specified.", 0);
	   }
	   Date startAuditDate = parseRangeDateStr(startAudit);
	   Date endAuditDate = parseRangeDateStr(endAudit);

	   exportRetrievedDocs(startAuditDate, endAuditDate, creatorOrgId, null, new File(destinationRoot), null /* destinationStream */);
   }

   /**
    * Wrapper around {@link #exportRetrievedDocs(String, String, String)} that takes
    * parameters inside a properties file.
    * @param exportProps
    * @throws ParseException
    * @throws SQLException 
    * @throws IOException 
    */
   public void exportRetrievedDocs(Properties exportProps) throws ParseException, IOException, SQLException
   {
	   String startAuditDateTime = exportProps.getProperty(PROP_START_AUDIT_DATETIME);
	   String endAuditDateTime = exportProps.getProperty(PROP_END_AUDIT_DATETIME);
	   String exportRootDir = exportProps.getProperty(PROP_EXPORT_ROOT_DIR);
       String creatorOrgId = exportProps.getProperty(PROP_CREATOR_ORG_ID);
	   exportRetrievedDocs(startAuditDateTime, endAuditDateTime, creatorOrgId, exportRootDir);
   }
   
   /**
    * Maps to friendly name that business likes.
    * @param docClassCode LOINC
    * @param formatCode from FORMAT_CODE in Documents table
    * @return
    */
   static String mapToDocumentTypeName(String docClassCode, String formatCode)
   {
   	if(StringUtils.isBlank(docClassCode))
   	{
   		return "UNK-DOC-TYPE";
   	}
   	
   	// obviate null checks below
   	if(formatCode == null) {
   		formatCode = "";
   	}
   	
   	if(docClassCode.equals("34133-9"))
   	{
   		if(formatCode.equals("urn:ihe:pcc:xphr:2007"))
   		{
   			return "C32";
   		}
   		
   		if(
   			formatCode.equals("urn:hl7-org:sdwg:ccda-structuredBody:1.1") ||
   			formatCode.equals("urn:hl7-org:sdwg:CCDA:1.1") ||
   			formatCode.equals("urn:hl7-org:sdwg:ccda-structuredBody:2.1"))
   		{
   			return "CCDACCD";
   		}
   	}
   	
   	if(docClassCode.equals("11506-3"))
   	{
   		if(
   			formatCode.equals("urn:hl7-org:sdwg:ccda-structuredBody:1.1") ||
   			formatCode.equals("urn:hl7-org:sdwg:ccda-structuredBody:2.1"))
   		{
   			return "CCDAPN";
   		}
   	}
   	
		if(formatCode.equals("urn:ihe:iti:xds-sd:pdf:2008"))
		{
			return "C62PDF";
		}

		if(formatCode.equals("urn:ihe:iti:xds-sd:text:2008"))
		{
			return "C62TXT";
		}
		
		// no mapping...default to LOINC
		return docClassCode;
   }

   static void exportFile(File destinationRoot, String fileSubPath, InputStream input) throws IOException, SQLException
   {
   	File targetFile = new File(destinationRoot, fileSubPath);
   	if(targetFile.exists())
      {
         return;
      }
      
      FileUtils.forceMkdir(targetFile.getParentFile());
      FileOutputStream fileOut = null;
      
      try
      {
         fileOut = new FileOutputStream(targetFile);
         IOUtils.copy(input, fileOut);
         fileOut.flush();
      }
      finally
      {
         DatabaseUtils.closeIO(fileOut, LOGGER);
      }
   }

   static void exportFile(ZipOutputStream zipOut, Set<String> pathsWritten, String fileSubPath, InputStream input)
   		throws IOException, SQLException
   {
   	if(pathsWritten.contains(fileSubPath))
		{
			return;
		}
   	pathsWritten.add(fileSubPath);
   	
   	ZipEntry entry = new ZipEntry(fileSubPath);
   	zipOut.putNextEntry(entry);
      IOUtils.copy(input, zipOut);
   }

   /**
    * Naming Convention for Downloaded Patient Files
    * The parent folder will the date the file was generated.
    * The sub parent folders will be separated by document creator.
    * Naming convention for specific files: <index><type><dateStored>.xml
    * E.g. 0001_C32_20161201.xml
	 * Date = date generated or received
    */
   String buildFileSubPath(Date auditDateTime, String docCreatorName, String index, String docType)
   {
      SimpleDateFormat sdf = new SimpleDateFormat(YEAR_MONTH_DAY);
   	StringBuilder path = new StringBuilder(sdf.format(exportDate));
   	
      docCreatorName = docCreatorName.replace("urn:oid:", "oid_");
      docCreatorName = cleanFileName(docCreatorName);
      path.append("/").append(docCreatorName);
      
   	docType = cleanFileName(docType);
      String auditDateStr = sdf.format(auditDateTime);
      String fileName = index + "_" + docType + "_" + auditDateStr + ".xml";
      path.append("/").append(fileName);
      
      return path.toString();
   }
   
   static final String cleanFileName(String fileName)
   {
   	char[] cleaned = null;
   	int fileLen = fileName.length();
   	for(int i = 0; i < fileLen; i++)
   	{
   		char charAt = fileName.charAt(i);
   		boolean ok =
   				('A' <= charAt && charAt <= 'Z') ||
   				('a' <= charAt && charAt <= 'z') ||
   				('0' <= charAt && charAt <= '9') ||
   				charAt == '-' ||
   				charAt == '_' ||
   				charAt == '.' ||
   				charAt == ' ';
   		
   		if(ok) {
   			continue;
   		}
   		
   		// lazy create copy...most names will be fine in reality
   		if(cleaned == null) {
   			cleaned = fileName.toCharArray();
   		}
   		cleaned[i] = '-';
   	}
   	
   	if(cleaned == null) {
   		return fileName;
   	} else {
   		return new String(cleaned);
   	}
   }
}
