package gov.va.med.nhin.adapter.xmlutils;

import gov.va.med.nhin.adapter.datamanager.SmartHashMap;
import gov.va.med.nhin.adapter.utils.NullChecker;
import gov.va.med.nhin.adapter.utils.Utils;
import gov.va.med.nhin.adapter.utils.XMLUtils;
import gov.va.med.nhin.adapter.xmlutils.config.ActionType;
import gov.va.med.nhin.adapter.xmlutils.config.ActionsType;
import gov.va.med.nhin.adapter.xmlutils.config.MappingType;
import gov.va.med.nhin.adapter.xmlutils.config.MappingsType;
import gov.va.med.nhin.adapter.xmlutils.config.NamespaceMappingType;
import gov.va.med.nhin.adapter.xmlutils.config.TemplateType;

import java.io.File;
import java.io.StringReader;
import java.lang.reflect.InvocationTargetException;
import java.util.Collection;
import java.util.HashMap;
import java.util.List;

import javax.xml.parsers.DocumentBuilder;
import javax.xml.xpath.XPath;
import javax.xml.xpath.XPathConstants;
import javax.xml.xpath.XPathExpression;
import javax.xml.xpath.XPathExpressionException;
import javax.xml.xpath.XPathFactory;

import org.apache.commons.beanutils.PropertyUtils;
import org.apache.commons.lang3.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.w3c.dom.Document;
import org.w3c.dom.Element;
import org.w3c.dom.NamedNodeMap;
import org.w3c.dom.Node;
import org.w3c.dom.NodeList;
import org.xml.sax.InputSource;

/**
 * Instances of XMLTemplate are used to create new XML documents from templates
 * using a mapping file to map properties of a POJO or HashMap to elements and
 * attributes in the template. The mapping file, given at the time an
 * XMLTemplate is constructed, is an XML file which gives the name of the
 * template, any namespace mappings which are required to properly resolve
 * elements in the template, and a collection of mappings that bind properties
 * to xpaths.
 *
 * @author David Vazquez
 */
public class XMLTemplate
{
    private static final Logger logger = LoggerFactory.getLogger(XMLTemplate.class.getName());

    private Document template;

    private TemplateType config;

    private XMLNamespaceContext namespaceContext = new XMLNamespaceContext();

    /**
     * Construct an XMLTemplate object. An XMLUtilsException will be thrown if
     * the mapping file or template file cannot be parsed.
     *
     * @param mapFilename path of the file that contains the mapping
     * configuration.
     * @throws xmltemplate.XMLUtilsException if the mapping or template file
     * cannot be parsed.
     */
    XMLTemplate(String configFilename) throws XMLUtilsException
    {
        loadConfig(configFilename);
    }

    private void loadConfig(String configFilename) throws XMLUtilsException
    {
        try {
            config = XMLUtils.load(configFilename, TemplateType.class);

            // Load template normally, if exception look for template in the
            // same path
            // as the config file - This is to support running in local stand
            // alone mode
            try {
                template = XMLUtils.parse(config.getSourceFile().getPath());
            }
            catch (IllegalArgumentException iae) {
                File file = new File(configFilename);
                template = XMLUtils.parse(file.getParent() + File.separator + config.getSourceFile().getPath());
            }
            namespaceContext = new XMLNamespaceContext();
            if (!NullChecker.isNullOrEmpty(config.getSourceFile().getNamespaceMappings())) {
                for (NamespaceMappingType nsMapping : config.getSourceFile().getNamespaceMappings().getNamespaceMapping()) {
                    namespaceContext.addMapping(nsMapping.getPrefix(), nsMapping.getUri());
                }
            }
        }
        catch (Throwable t) {
            throw new XMLUtilsException("An error occurred when constructing XML template.", t);
        }
    }

    /**
     * Creates a new XML document using the file specified in the mapping file
     * as a template. Attribute and element values are filled in using values
     * from the given bean based on the mappings given in the mapping file.
     *
     * @param fromBean POJO or HashMap whose properties are set to the values
     * which will be used to fill in the template.
     * @return a new XML document.
     * @throws XMLUtilsException if a required property is not
     * present in the given bean or and invalid xpath was specified in the
     * mapping file.
     */
    public Document fillIn(Object fromBean)
    {
        Document ret = null;
        
        logger.info("Entering");
        
        indexByAttribute.clear();

        try {
            // Mechanism used in older version of Saxon to find Saxon JAXP implementation doesn't work
            // in Java 8.  The following is used to ensure that we get Saxon's JAXP implentation.
            System.setProperty("javax.xml.xpath.XPathFactory:" + net.sf.saxon.lib.NamespaceConstant.OBJECT_MODEL_SAXON, "net.sf.saxon.xpath.XPathFactoryImpl");
            XPathFactory xpathFactory = XPathFactory.newInstance(net.sf.saxon.lib.NamespaceConstant.OBJECT_MODEL_SAXON);
            // end
            XPath xpathTemplate = xpathFactory.newXPath();
            XMLXPathEvaluationCache xpathEvaluationCache = new XMLXPathEvaluationCache();
            ret = (Document) template.cloneNode(true);
            xpathTemplate.setNamespaceContext(namespaceContext);
            fillIn(ret, ret, fromBean, config.getMappings(), xpathTemplate, xpathEvaluationCache);
            return ret;
        }
        catch (XMLUtilsException xte) {
            throw new XMLUtilsException("XML Util Exception: ", xte);
        }
        catch (Throwable t) {
            throw new XMLUtilsException("An error occurred when filling in a template.", t);
        }
        finally {
            logger.info("Exitting");
        }
    }

    private void fillIn(Document document, Node root, Object fromBean, MappingsType mappings, XPath xpathTemplate, XMLXPathEvaluationCache xpathEvaluationCache)
    {
        fillIn(document, root, fromBean, mappings, xpathTemplate, xpathEvaluationCache, 0);
    }

    @SuppressWarnings("unchecked")
    private void fillIn(Document document, Node root, Object fromBean, MappingsType mappings, XPath xpathTemplate, XMLXPathEvaluationCache xpathEvaluationCache, int index)
    {
        if (mappings != null) {
            for (MappingType mapping : mappings.getMapping()) {
                if (equalsProperty(mapping.getEqualsProperty(), mapping.getEqualsValue(), mapping.getNotEqualsProperty(), mapping.getNotEqualsValue(), fromBean)) {
                    try {
                        String propertyString = mapping.getProperty().getValue();
                        String xpathString = mapping.getXpath().getValue();
                        XPathExpression xpathExpression = xpathEvaluationCache.getExpression(xpathTemplate, xpathString);
                        Node node = (Node) xpathExpression.evaluate(root, XPathConstants.NODE);

                        if (node != null) {

                            Object value = null;
                            /*
							 * The cloning feature used by the mapping software
							 * to populate structures essentially requires all
							 * the values to be one list. The Plan of Care lists
							 * were from three different sources and creating
							 * one uber structure to support them all was not
							 * optimal
							 *
							 * Here we have fully qualified properties in the
							 * mapping file separated by a comma and process
							 * them to create a list of objects and there is no
							 * need for an uber structure
                             */
                            if (propertyString.contains(",")) {

                                String[] properties = propertyString.split(",");
                                for (String property : properties) {

                                    Object temp = getPropertyFromBean(fromBean, property);

                                    if (temp != null && temp instanceof List) {
                                        if (value == null) {
                                            value = temp;
                                        }
                                        else {
                                            ((List) value).addAll((Collection) temp);
                                        }

                                    }

                                }

                            }
                            else if (propertyString.contains("|")) {

                                String[] properties = propertyString.split("\\|");
                                for (String property : properties) {
                                    value = getPropertyFromBean(fromBean, property);
                                    if (value != null) {
                                        break;
                                    }
                                }

                            }
                            else {
                                value = getPropertyFromBean(fromBean, propertyString);
                            }

                            if (mapping.getXpath().isCopy()) {
                                Node newNode = node.cloneNode(true);
                                node.getParentNode().insertBefore(newNode, node.getNextSibling());
                                node = newNode;
                            }

                            if (!NullChecker.isNullOrEmpty(value)) {
                                fillIn(document, root, node, fromBean, value, mapping, mapping.getIfNotNull(), xpathTemplate, xpathEvaluationCache, index);
                            }
                            else {
                                if (mapping.getProperty().isOptional()) {
                                    XMLUtilsException xue = new XMLUtilsException("Optional property is missing.", propertyString, xpathString);
                                    logger.debug("Optional property is missing: ", xue);
                                }
                                else {
                                    XMLUtilsException xue = new XMLUtilsException("Required property is missing.", propertyString, xpathString);
                                    logger.debug("Required property is missing.", xue);
                                }

                                if (!NullChecker.isNullOrEmpty(mapping.getIfNull())) {
                                    fillIn(document, root, node, fromBean, value, mapping, mapping.getIfNull(), xpathTemplate, xpathEvaluationCache, index);
                                }
                            }
                        }
                        else {
                            XMLUtilsException xue = new XMLUtilsException("Given xpath did not resolve to a node. ---->>> " + root.toString(), propertyString, xpathString);
                            logger.debug("Given xpath did not resolve to a node ; ", xue);
                        }
                    }
                    catch (Throwable t) {
                        throw new XMLUtilsException("An error occurred when filling in a template.", t, mapping.getProperty().getValue(), mapping.getXpath().getValue());
                    }
                }
            }
        }
    }

    private Object getPropertyFromBean(Object bean, String property)
    {
        Object value = null;

        try {
            if (!NullChecker.isNullOrEmpty(property)) {
                if (property.contains("|")) {
                    String[] properties = property.split("|");
                    for (String prop : properties) {
                        value = PropertyUtils.getProperty(bean, prop);
                        if (!NullChecker.isNullOrEmpty(value)) {
                            break;
                        }
                    }
                }
                else {
                    value = PropertyUtils.getProperty(bean, property);
                }
            }
            else {
                value = bean;
            }
        }
        catch (Exception e) {
            value = null;
        }

        return value;
    }

    private void fillIn(Document document, Node root, Node node, Object fromBean, Object value, MappingType mapping, ActionsType actions, XPath xpathTemplate, XMLXPathEvaluationCache xpathEvaluationCache, int index) throws IllegalAccessException, InvocationTargetException, NoSuchMethodException, XPathExpressionException
    {
        for (ActionType action : actions.getAction()) {
            if (equalsProperty(mapping.getEqualsProperty(), mapping.getEqualsValue(), mapping.getNotEqualsProperty(), mapping.getNotEqualsValue(), fromBean)) {
                Node entryNode = getEntryNode(node, action, xpathTemplate, xpathEvaluationCache);
                Object entryValue = getValue(action, fromBean, value);

                if (entryNode != null) {
                    switch (action.getAction()) {
                        case SET_VALUE:
                        default:
                            if (!Utils.isCollection(entryValue)) {
                                entryNode.setTextContent(getValueAsString(action, entryValue, fromBean, index));
                            }
                            else {
                                Node siblingNode = entryNode.getNextSibling();
                                for (int i = 0; i < Utils.getCollectionLength(entryValue); ++i) {
                                    Object o = PropertyUtils.getIndexedProperty(entryValue, "", i);
                                    if (!NullChecker.isNullOrEmpty(o)) {
                                        Node newNode = entryNode.cloneNode(true);
                                        entryNode.getParentNode().insertBefore(newNode, siblingNode);
                                        newNode.setTextContent(o.toString());
                                        if (!NullChecker.isNullOrEmpty(action.getAttribute())) {
                                            NamedNodeMap attributes = newNode.getAttributes();
                                            Node attributeNode = attributes.getNamedItem(action.getAttribute());
                                            if (attributeNode == null) {
                                                attributeNode = entryNode.getOwnerDocument().createAttribute(action.getAttribute());
                                                attributes.setNamedItem(attributeNode);
                                            }
                                            String textContent = attributeNode.getTextContent();
                                            if (!NullChecker.isNullOrEmpty(textContent)) {
                                                textContent += "-" + (i + 1);
                                            }
                                            else {
                                                textContent = Integer.toString(i);
                                            }
                                            attributeNode.setTextContent(textContent);
                                        }
                                    }
                                }
                                entryNode.getParentNode().removeChild(entryNode);
                            }
                            break;

                        case SET_ATTRIBUTE:
                            {
                                NamedNodeMap attributes = entryNode.getAttributes();
                                Node attributeNode = attributes.getNamedItem(action.getAttribute());
                                if (attributeNode == null) {
                                    attributeNode = entryNode.getOwnerDocument().createAttribute(action.getAttribute());
                                    attributes.setNamedItem(attributeNode);
                                }
                                attributeNode.setTextContent(getValueAsString(action, entryValue, fromBean, index));
                            }
                            break;

                        case APPEND_ATTRIBUTE:
                            {
                                NamedNodeMap attributes = entryNode.getAttributes();
                                Node attributeNode = attributes.getNamedItem(action.getAttribute());
                                if (attributeNode == null) {
                                    attributeNode = entryNode.getOwnerDocument().createAttribute(action.getAttribute());
                                    attributes.setNamedItem(attributeNode);
                                    attributeNode.setTextContent(getValueAsString(action, entryValue, fromBean, index));
                                }
                                else {
                                    attributeNode.setTextContent(attributeNode.getTextContent() + getValueAsString(action, entryValue, fromBean, index));
                                }
                            }
                            break;

                        case APPEND_VALUE:
                            if (!Utils.isCollection(value)) {
                                String textContent = entryNode.getTextContent();
                                String appendText = getValueAsString(action, entryValue, fromBean, index);
                                if (!NullChecker.isNullOrEmpty(textContent)) {
                                    textContent += appendText;
                                }
                                else {
                                    textContent = appendText;
                                }
                                entryNode.setTextContent(textContent);
                            }
                            else {
                                logger.debug("Cannot use appendValue on a collection.");
                            }
                            break;

                        case APPEND_ELEMENT:

                            /*
						 * APPEND_ELEMENT is used to insert structured (tags)
						 * content into the document The primary driver was to
						 * support the ability to insert <br/> into the
						 * narrative block swm
                             */
                            String content = getValueAsString(action, entryValue, fromBean, index);

                            try {
                                DocumentBuilder documentBuilder = XMLParserUtils.getXMLDocBuilder();
                                // using content as outer tags - div is not
                                // acceptable by hl7
                                Document narrativeDocument = documentBuilder.parse(new InputSource(new StringReader("<content>" + content.toString() + "</content>")));
                                Node narrativeContent = narrativeDocument.getDocumentElement();
                                narrativeContent = (Element) document.importNode(narrativeContent, true);
                                entryNode.appendChild(narrativeContent);
                            }
                            catch (Exception e) {
                                throw new XMLUtilsException("An error occurred when processing append_element - Offending line: ", e);
                            }

                            break;

                        case EVALUATE_MAPPINGS:
                            if (!Utils.isCollection(entryValue)) {
                                fillIn(document, entryNode, entryValue, mapping.getMappings(), xpathTemplate, xpathEvaluationCache);
                            }
                            else {
                                Node parentNode = entryNode.getParentNode();
                                Node siblingNode = entryNode.getNextSibling();
                                for (int i = 0; i < Utils.getCollectionLength(entryValue); ++i) {
                                    Object o = PropertyUtils.getIndexedProperty(entryValue, "", i);
                                    Node newNode = entryNode.cloneNode(true);
                                    if (siblingNode != null) {
                                        parentNode.insertBefore(newNode, siblingNode);
                                    }
                                    else {
                                        parentNode.appendChild(newNode);
                                    }
                                    fillIn(document, newNode, o, mapping.getMappings(), xpathTemplate, xpathEvaluationCache, i);
                                }
                                parentNode.removeChild(entryNode);
                            }
                            break;
                        case EVALUATE_MAPPINGS_TO_PARENT:
                            if (!Utils.isCollection(entryValue)) {
                                fillIn(document, entryNode, entryValue, mapping.getMappings(), xpathTemplate, xpathEvaluationCache);
                            }
                            else {
                                for (int i = 0; i < Utils.getCollectionLength(entryValue); ++i) {
                                    Object o = PropertyUtils.getIndexedProperty(entryValue, "", i);
                                    fillIn(document, entryNode, o, mapping.getMappings(), xpathTemplate, xpathEvaluationCache, i);
                                }
                            }
                            break;
                        case REMOVE:
                            entryNode.getParentNode().removeChild(entryNode);
                            break;

                        case REMOVE_CHILDREN:
                            NodeList childNodes = entryNode.getChildNodes();
                            while (childNodes.getLength() > 0) {
                                entryNode.removeChild(childNodes.item(0));
                            }
                            break;

                        case REMOVE_ATTRIBUTES:
                            NamedNodeMap attributeNodes = entryNode.getAttributes();
                            while (attributeNodes.getLength() > 0) {
                                Node attribute = attributeNodes.item(0);
                                attributeNodes.removeNamedItem(attribute.getNodeName());
                            }
                            break;
                    }
                }
                else {
                    XMLUtilsException xue = new XMLUtilsException("Relative xpath (" + action.getRelative() + ") did not resolve to a node. --->" + root.getLocalName(), mapping.getProperty().getValue(), mapping.getXpath().getValue());
                    logger.debug("XMLUtilException occurred: ", xue);
                }
            }
        }
    }

    /**
     * Updated to include not equals and to look for bean value for all values
     * not just passed in string. Covers the following use cases:
     *
     * <PRE>
     * <ol>
     * 		<!-- Case for just equals property. -->
     * 		<li>(equals property != null and equals value != null) AND (not equals property == null and not equals value == null)</li>
     * 		<!-- Case for just not equals property. -->
     *  	<li>(equals property == null and equals value == null) AND (not equals property != null and not equals value != null)</li>
     *  	<!-- And condition for both -->
     *  	<li>(equals property == null and equals value == null) AND (not equals property != null and not equals value != null)</li>
     *
     * </ol>
     * </PRE>
     *
     * @param equalsProperty
     * @param equalsValue
     * @param notEqualsProperty
     * @param notEqualsValue
     * @param fromBean
     * @return
     */
    private boolean equalsProperty(final String equalsProperty, final String equalsValue, final String notEqualsProperty, final String notEqualsValue, final Object fromBean)
    {
        boolean ret = true;

        // First case
        if (StringUtils.isNotBlank(equalsProperty) && StringUtils.isBlank(notEqualsProperty)) {
            return singleEqualsTest(equalsProperty, equalsValue, fromBean);
        }

        // Second case
        if (StringUtils.isNotBlank(notEqualsProperty) && StringUtils.isBlank(equalsProperty)) {
            return !singleEqualsTest(notEqualsProperty, notEqualsValue, fromBean);
        }

        // Third Case
        if (StringUtils.isNotBlank(notEqualsProperty) && StringUtils.isNotBlank(equalsProperty)) {
            return singleEqualsTest(equalsProperty, equalsValue, fromBean) & !singleEqualsTest(notEqualsProperty, notEqualsValue, fromBean);
        }

        return ret;
    }

    private boolean singleEqualsTest(final String equalsProperty, final String equalsValue, final Object fromBean)
    {
        if (!NullChecker.isNullOrEmpty(equalsProperty)) {
            Object value = getPropertyFromBean(fromBean, equalsProperty);

            if (StringUtils.isEmpty(equalsValue) && value == null) {
                return true;
            }
            else if (value != null) {
                Object eqValue = getPropertyFromBean(fromBean, equalsValue);
                if (eqValue == null || eqValue == fromBean) {
                    // Not a property a static value or missing
                    return value != null && value.toString().equals(equalsValue);
                }
                else {
                    return value != null && value.toString().equals(eqValue.toString());
                }
            }
            else {
                return false;
            }
        }

        return true;
    }

    private Node getEntryNode(Node node, ActionType action, XPath xpathTemplate, XMLXPathEvaluationCache xpathEvaluationCache) throws XPathExpressionException
    {
        Node entryNode;

        if (!NullChecker.isNullOrEmpty(action.getRelative())) {
            XPathExpression xpathExpression = xpathEvaluationCache.getExpression(xpathTemplate, action.getRelative());
            entryNode = (Node) xpathExpression.evaluate(node, XPathConstants.NODE);
        }
        else {
            entryNode = node;
        }

        return entryNode;
    }

    private Object getValue(ActionType action, Object fromBean, Object value)
    {
        Object ret = action.getValue();

        if (NullChecker.isNullOrEmpty(ret)) {
            if (!NullChecker.isNullOrEmpty(action.getProperty())) {
                ret = getPropertyFromBean(fromBean, action.getProperty());
            }
            else {
                ret = value;
            }
        }

        return ret;
    }

    HashMap<String, Integer> indexByAttribute = new HashMap<String, Integer>();

    // Uses hash map to track indexes across loops
    // Updating to remove loop order requirement and external dependencies and
    // allow for more accurate tracking of values.
    private String getValueAsString(ActionType action, Object value, Object fromBean, int index)
    {
        String ret = value != null ? value.toString() : "";

        String key = ret + "_ICID";
        boolean reference = false;
        if (StringUtils.startsWith(key, "#")) {
            key = key.substring(1);
            reference = true;
        }
        if (action.isIndexed()) {
            // Make sure that we have a place to add these objects.
            if (fromBean instanceof SmartHashMap) {
                SmartHashMap map = (SmartHashMap) fromBean;
                if (map.containsKey(key)) {
                    // Our parent already contains a value for this object.
                    ret = (String) map.get(key);
                    if (reference) {
                        ret = "#" + ret;
                    }
                }
                else {
                    // The parent does not define the key. Create a new one.
                    if (!indexByAttribute.containsKey(key)) {
                        indexByAttribute.put(key, new Integer(1000));
                    }
                    Integer current = indexByAttribute.get(key);
                    indexByAttribute.put(key, ++current);
                    ret += current;
                    // Do not add a hash to the value.
                    if (reference) {
                        map.put(key, ret.substring(1));
                        if (StringUtils.isNotBlank(action.getPassIndexToChild())) {
                            passIndexToChild(fromBean, action.getPassIndexToChild(), key, ret.substring(1));
                        }
                    }
                    else {
                        map.put(key, ret);
                        if (StringUtils.isNotBlank(action.getPassIndexToChild())) {
                            passIndexToChild(fromBean, action.getPassIndexToChild(), key, ret);
                        }
                    }
                }
            }
            else {
                // Just in case there is not a valid parent.
                if (!indexByAttribute.containsKey(ret)) {
                    indexByAttribute.put(ret, new Integer(1000));
                }
                Integer current = indexByAttribute.get(ret);
                indexByAttribute.put(ret, ++current);
                ret += current;
            }
        }

        return ret;
    }

    private void passIndexToChild(final Object fromBean, final String childName, final String key, final String index)
    {
        if (fromBean instanceof SmartHashMap) {
            SmartHashMap map = (SmartHashMap) fromBean;
            Object child = map.get(childName);
            if (child != null) {
                if (child instanceof List) {
                    // There a few children to add this to.
                    @SuppressWarnings("unchecked")
                    List<SmartHashMap> cList = (List<SmartHashMap>) child;
                    for (SmartHashMap cMap : cList) {
                        cMap.put(key, index);
                    }
                }
                else // We are at the instance to add the value.
                if (child instanceof SmartHashMap) {
                    SmartHashMap cMap = (SmartHashMap) child;
                    cMap.put(key, index);
                }
            }
        }
    }

    TemplateType getConfig()
    {
        return config;
    }
}
