/*
 * To change this license header, choose License Headers in Project Properties.
 * To change this template file, choose Tools | Templates
 * and open the template in the editor.
 */
package gov.va.med.nhin.adapter.utils.soap.handler;

import gov.va.med.nhin.adapter.logging.EventAuditingFactory;
import gov.va.med.nhin.adapter.logging.EventAuditingFactoryImpl;
import gov.va.med.nhin.adapter.logging.LogConstants;
import gov.va.med.nhin.adapter.utils.NullChecker;
import java.io.ByteArrayOutputStream;
import java.util.Iterator;
import java.util.Set;
import java.util.UUID;
import javax.servlet.http.HttpServletRequest;
import javax.xml.namespace.QName;
import javax.xml.soap.SOAPException;
import javax.xml.soap.SOAPHeader;
import javax.xml.soap.SOAPHeaderElement;
import javax.xml.soap.SOAPMessage;
import javax.xml.transform.Source;
import javax.xml.transform.Transformer;
import javax.xml.transform.TransformerException;
import javax.xml.transform.TransformerFactory;
import javax.xml.transform.stream.StreamResult;
import javax.xml.ws.handler.MessageContext;
import javax.xml.ws.handler.soap.SOAPHandler;
import javax.xml.ws.handler.soap.SOAPMessageContext;
import org.hl7.fhir.dstu3.model.AuditEvent;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.slf4j.MDC;
import org.w3c.dom.Node;

/**
 *
 * @author david_vetsez
 */
public class SOAPLoggingHandler implements SOAPHandler<SOAPMessageContext> {

	private static final Logger logger = LoggerFactory.getLogger( SOAPLoggingHandler.class );
	private final ThreadLocal<String> txnid = new ThreadLocal<>();

	public SOAPLoggingHandler() {
	}

	public SOAPLoggingHandler( String transactionId ) {
		txnid.set( transactionId );
	}

	@Override
	public Set<QName> getHeaders() {
		return null;
	}

	@Override
	public void close( MessageContext context ) {
	}

	@Override
	public boolean handleFault( SOAPMessageContext context ) {
		return true;
	}

	private String findOrGenTransactionId() {
		String txn = null;
		if ( null == txnid.get() ) {
			txn = ( null == MDC.get( LogConstants.CORRELATION_MDCKEY )
					? UUID.randomUUID().toString()
					: String.class.cast( MDC.get( LogConstants.CORRELATION_MDCKEY ) ) );
		}
		else {
			txn = txnid.get();
		}

		return txn;
	}

	@Override
	public boolean handleMessage( SOAPMessageContext smc ) {
		Boolean isOutbound = (Boolean) smc.get( MessageContext.MESSAGE_OUTBOUND_PROPERTY );

		SOAPMessage message = smc.getMessage();
		try {
			String corrid = null;

			SOAPHeader headers = message.getSOAPHeader();
			Iterator<Node> it = headers.getChildElements( LogConstants.CORRELATION_SOAPPROP );
			if ( it.hasNext() ) {
				// we have a transaction id in the message headers, so use it
				Node n = it.next();
				corrid = n.getTextContent();
			}
			else {
				// no transaction id in the message headers...was one set in the ctor?
				corrid = findOrGenTransactionId();

				// set our transaction id in the message headers for anyone else to use
				SOAPHeaderElement txnheader
						= headers.addHeaderElement( LogConstants.CORRELATION_SOAPPROP );
				txnheader.addTextNode( corrid );
			}

			if ( !isOutbound ) {
				MDC.put( LogConstants.CORRELATION_MDCKEY, corrid );

				// log a bunch of stuff
				EventAuditingFactory<AuditEvent> fac
						= EventAuditingFactoryImpl.getFactory( AuditEvent.class );
				Object obj = smc.get( MessageContext.SERVLET_REQUEST );
				HttpServletRequest req = HttpServletRequest.class.cast( obj );
				AuditEvent ae = fac.newEvent( req, getClass() );
				fac.info( ae );
			}

		}
		catch ( SOAPException e ) {
			// don't care...it'll get set on the next call
		}

		if ( logger.isDebugEnabled() ) {
			String endpointAddress = (String) smc.get( "javax.xml.ws.service.endpoint.address" );
			if ( NullChecker.isNullOrEmpty( endpointAddress ) ) {
				endpointAddress = (String) smc.get( "com.sun.xml.ws.transport.http.servlet.requestURL" );
			}

			try {
				ByteArrayOutputStream baos = new ByteArrayOutputStream();
				Source source = message.getSOAPPart().getContent();
				Transformer transformer = TransformerFactory.newInstance().newTransformer();
				//transformer.setOutputProperty(OutputKeys.INDENT, "yes");
				//transformer.setOutputProperty("{http://xml.apache.org/xslt}indent-amount", "2");
				transformer.transform( source, new StreamResult( baos ) );
				logger.debug( "SOAP {} MESSAGE\nEndpoint Address: {}\nPayload: {}", isOutbound
						? "OUTBOUND" : "INBOUND", endpointAddress, new String( baos.toByteArray() ) );
			}
			catch ( SOAPException | TransformerException e ) {
				logger.debug( "Error occurred writing message to log.  Proceeding anyway." );
			}
		}

		if ( isOutbound ) {
			MDC.clear();
		}

		return true;
	}
}
