/*
 * To change this license header, choose License Headers in Project Properties.
 * To change this template file, choose Tools | Templates
 * and open the template in the editor.
 */
package gov.va.med.nhin.adapter.logging.messaging.fhir;

import ca.uhn.fhir.context.FhirContext;
import ca.uhn.fhir.parser.IParser;
import gov.hhs.fha.nhinc.common.nhinccommon.AssertionType;
import gov.hhs.fha.nhinc.common.nhinccommon.CeType;
import gov.hhs.fha.nhinc.common.nhinccommon.UserType;
import gov.va.med.nhin.adapter.logging.EventAuditingFactory;
import gov.va.med.nhin.adapter.logging.LogConstants;
import gov.va.med.nhin.adapter.logging.LogConstants.AuditingEvent;
import gov.va.med.nhin.adapter.logging.MessagingHelper;
import java.util.Date;
import javax.servlet.http.HttpServletRequest;
import org.hl7.fhir.dstu3.model.AuditEvent;
import org.hl7.fhir.dstu3.model.AuditEvent.AuditEventAgentComponent;
import org.hl7.fhir.dstu3.model.AuditEvent.AuditEventAgentNetworkComponent;
import org.hl7.fhir.dstu3.model.AuditEvent.AuditEventEntityComponent;
import org.hl7.fhir.dstu3.model.AuditEvent.AuditEventSourceComponent;
import org.hl7.fhir.dstu3.model.CodeableConcept;
import org.hl7.fhir.dstu3.model.Coding;
import org.hl7.fhir.dstu3.model.InstantType;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.slf4j.MDC;

/**
 * Class to create AuditEvents and log them using slf4j
 *
 * @author ryan
 */
public class FHIREventAuditingFactory implements EventAuditingFactory<AuditEvent> {

	private final FhirContext ctx = FhirContext.forDstu3();
	private final IParser parser = ctx.newJsonParser();

	@Override
	public void trace( AuditEvent event ) {
		Logger log = LoggerFactory.getLogger( getCategory( event ) );
		log.trace( parser.encodeResourceToString( event ) );
	}

	@Override
	public void debug( AuditEvent event ) {
		Logger log = LoggerFactory.getLogger( getCategory( event ) );
		log.debug( parser.encodeResourceToString( event ) );
	}

	@Override
	public void info( AuditEvent event ) {
		String category = getCategory( event );
		Logger log = LoggerFactory.getLogger( category );
		String str = parser.encodeResourceToString( event );
		log.info( str );
	}

	@Override
	public void warn( AuditEvent event ) {
		Logger log = LoggerFactory.getLogger( getCategory( event ) );
		log.warn( parser.encodeResourceToString( event ) );
	}

	@Override
	public void error( AuditEvent event ) {
		Logger log = LoggerFactory.getLogger( getCategory( event ) );
		log.error( parser.encodeResourceToString( event ) );
	}

	public String getCategory( AuditEvent ae ) {
		String code = ae.getSubtypeFirstRep().getCode();
		return AuditingEvent.fromCode( code ).category;
	}

	@Override
	public void trace( AuditingEvent c, Class<?> klass ) {
		trace( newEvent( c, klass ) );
	}

	@Override
	public void debug( AuditingEvent c, Class<?> klass ) {
		debug( newEvent( c, klass ) );
	}

	@Override
	public void info( AuditingEvent c, Class<?> klass ) {
		info( newEvent( c, klass ) );
	}

	@Override
	public void warn( AuditingEvent c, Class<?> klass ) {
		warn( newEvent( c, klass ) );
	}

	@Override
	public void error( AuditingEvent c, Class<?> klass ) {
		error( newEvent( c, klass ) );
	}

	@Override
	public AuditEvent newEvent( AuditingEvent c, Class<?> source ) {
		AuditEventSourceComponent srccmp = new AuditEventSourceComponent();
		srccmp.setSite( source.getCanonicalName() );
		AuditEvent ae = new AuditEvent( c.type, new InstantType( new Date() ), srccmp );
		ae.addSubtype( c.coding );

		String txn = MDC.get( LogConstants.CORRELATION_MDCKEY );
		if ( null != txn ) {
			addEntity( ae, LogConstants.MESSAGING_URI, "correlationId", txn );
		}

		return ae;
	}

	@Override
	public AuditEvent newEvent( HttpServletRequest req, Class<?> source ) {
		AuditEvent ae = newEvent( AuditingEvent.INFO, source );

		if ( null != req ) {
			AuditEventAgentComponent aeac = ae.addAgent();
			AuditEventAgentNetworkComponent net = aeac.getNetwork();
			net.setAddress( req.getRemoteAddr() );

			if( null != req.getUserPrincipal() ){
				aeac.setName( req.getUserPrincipal().getName() );
			}
		}

		return ae;
	}

	@Override
	public AuditEvent newEvent( AssertionType info, AuditingEvent code, Class<?> source ) {
		AuditEvent ae = newEvent( code, source );

		if ( null != info ) {
			AuditEventAgentComponent aeac = null;
			if ( null != info.getPersonName() ) {
				aeac = ae.addAgent();
				aeac.setName( info.getPersonName().getFullName() );
			}

			if ( null != info.getSSN() ) {
				if ( null == aeac ) {
					aeac = ae.addAgent();
				}
				aeac.setAltId( info.getSSN() );
			}

			if( null != info.getUserInfo() ){
				aeac = ae.addAgent();
				UserType ui = info.getUserInfo();
				aeac.setName( ui.getPersonName().getFullName() );
				aeac.setAltId( ui.getUserName() );

				if( null != ui.getRoleCoded() ){
					CeType ct = ui.getRoleCoded();
					CodeableConcept cc = aeac.addRole();
					Coding coding = cc.addCoding();
					coding.setSystem( ct.getCodeSystem() );
					coding.setCode( ct.getCode() );
					coding.setDisplay( ct.getDisplayName() );
				}
			}

			if ( null != aeac ) {
				CodeableConcept cc = aeac.addRole();
				Coding c = cc.addCoding();
				c.setSystem( LogConstants.MESSAGING_URI );
				c.setCode( "userinfo" );
			}
		}

		return ae;
	}

	@Override
	public AuditEvent addAgent( AuditEvent event, String systemuri, String codename,
			String data ) {
		AuditEventAgentComponent aeac = event.addAgent();
		aeac.setName( data );
		CodeableConcept cc = aeac.addRole();
		Coding c = cc.addCoding();
		c.setSystem( systemuri );
		c.setCode( codename );
		return event;
	}

	@Override
	public AuditEvent addEntity( AuditEvent event, String systemuri, String codename,
			String data ) {
		AuditEventEntityComponent aeec = event.addEntity();
		aeec.setType( new Coding( systemuri, codename, null ) );
		aeec.setName( data );
		return event;
	}

	@Override
	public MessagingHelper<AuditEvent> messaging() {
		return new MessagingHelper() {
			@Override
			public AuditEvent partnerauth( AuditingEvent evt, Class source, String partner ) {
				return addEntity( newEvent( evt, source ), LogConstants.MESSAGING_URI,
						PARTNER, partner );
			}

			@Override
			public Object reqres( AuditingEvent evt, Class source, String partner ) {
				return addAgent( newEvent( evt, source ), LogConstants.MESSAGING_URI,
						REQUEST, partner );
			}
		};
	}
}
