package gov.va.med.nhin.adapter.datamanager.internalApp;

import gov.va.med.nhin.adapter.datamanager.DataDate;
import gov.va.med.nhin.adapter.datamanager.Reference;
import gov.va.med.nhin.adapter.datamanager.SmartHashMap;
import gov.va.med.nhin.adapter.datamanager.translators.FileManDateDataTranslator;

import java.math.BigDecimal;
import java.util.ArrayList;
import java.util.Calendar;
import java.util.Iterator;
import java.util.List;
import java.util.Map;

import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang3.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class InternalAppFilter
{
	private static final Logger logger = LoggerFactory.getLogger(InternalAppFilter.class.getName());

	// This is used for special cases where the getobject Method would error out
	// if there was a list in its path.
	private static class FoundList
	{
		private final String keysLeft;
		private final List<SmartHashMap> list;

		public FoundList(String keysLeft, List<SmartHashMap> list)
		{
			this.keysLeft = keysLeft;
			this.list = list;
		}

		public String getKeyList()
		{
			return keysLeft;
		}

		public List<SmartHashMap> getList()
		{
			return list;
		}
	}

	// Method will perform in place removal of objects that meet criteria to be
	// filtered.
	@SuppressWarnings("unchecked")
	public void filterObjects(Map<String, List<InternalAppFilterObj>> allSectionFilters, SmartHashMap allSectionsData )
	{
		if( null != allSectionFilters && null != allSectionsData )
		{
			for(Map.Entry<String, List<InternalAppFilterObj>> entry : allSectionFilters.entrySet())
			{
				String sectionName = entry.getKey();
				List<InternalAppFilterObj> sectionFilters = entry.getValue();
				// Get the section object.
				Object valueObj = allSectionsData.get( sectionName );
				// If we remove all of the items on the list we need to
				// remove the Map Entry.
				if( valueObj instanceof ArrayList )
				{
					List<SmartHashMap> sectionItemMaps = ( List<SmartHashMap> ) valueObj;
					filterSectionObjects( sectionFilters, sectionItemMaps );
					// Section is now empty. Remove from the map.
					if( CollectionUtils.isEmpty( sectionItemMaps ) )
					{
						allSectionsData.remove( sectionName );
					}
				}
				// Else it would be an object that is not a "Section" Level
				// Component. We will not filter those.
			}
		}
	}

	@SuppressWarnings("unchecked")
	public void filterSectionObjects(List<InternalAppFilterObj> filters, List<SmartHashMap> itemsToCheck)
	{
		List<InternalAppFilterObj> limitFilters = new ArrayList<>();
		for(InternalAppFilterObj filter : filters)
		{

			if(InternalAppOperations.LIST_LIMIT.equals(filter.getOperation()))
			{
				// This is a list limit filter. Run it last.
				limitFilters.add(filter);
			}
			else
			{
				// Not a list limit filter as usual.
				if(StringUtils.isNotBlank(filter.getSubRoot()))
				{
					// We need to look at a subList or map.
					for(SmartHashMap map : itemsToCheck)
					{
						Object obj = getObject(filter.getSubRoot(), map);
						if(obj instanceof List)
						{
							filterMapObjects(filter, (List<SmartHashMap>) obj);
							if(CollectionUtils.isEmpty((List<SmartHashMap>) obj))
							{
								map.remove(filter.getSubRoot());
							}
						}
					}
				}
				else
				{
					// We are comparing against the main object.
					filterMapObjects(filter, itemsToCheck);
				}
			}
		}

		// Always run limit filters after filtering other objects.
		for(InternalAppFilterObj limitFilter : limitFilters)
		{
			if(StringUtils.isBlank(limitFilter.getObjectName()))
			{
				// Section level limit
				limitList(limitFilter, itemsToCheck);
			}
			else
			{
				if(StringUtils.isNotBlank(limitFilter.getSubRoot()))
				{
					// We need to look at a subList or map.
					for(SmartHashMap map : itemsToCheck)
					{
						Object obj = getObject(limitFilter.getSubRoot(), map);
						if(obj instanceof List)
						{
							filterMapObjects(limitFilter, (List<SmartHashMap>) obj);
							if(CollectionUtils.isEmpty((List<SmartHashMap>) obj))
							{
								// removeObject(limitFilter.getObjectName(),
								// map);
								map.remove(limitFilter.getSubRoot());
							}
						}
					}
				}
				else
				{
					// We are comparing against a subObject of the main object.
					filterMapObjects(limitFilter, itemsToCheck);
				}
			}
		}
	}

	public void filterMapObjects(InternalAppFilterObj filter, List<SmartHashMap> items)
	{
		for(Iterator<SmartHashMap> iter = items.iterator(); iter.hasNext();)
		{
			SmartHashMap liMap = iter.next();
			Object value = getObject(filter.getObjectName(), liMap);
			if(value instanceof FoundList)
			{
				// We encountered a List of objects in the middle of getting
				// objects.
				// Need to loop through this list.
				FoundList found = (FoundList) value;
				InternalAppFilterObj internalFilter = new InternalAppFilterObj();
				internalFilter.setOperation(filter.getOperation());
				internalFilter.setFilterValue(filter.getFilterValue());
				internalFilter.setObjectName(found.getKeyList());
				filterMapObjects(internalFilter, found.getList());
			}
			else
			{
				if(!matchCriteria(filter, value))
				{
					iter.remove();
				}
			}
		}
	}

	@SuppressWarnings("rawtypes")
	private void limitList(InternalAppFilterObj limitFilter, List list)
	{
		if(limitFilter != null && list != null)
		{
			int limit = -1;
			try
			{
				limit = Integer.parseInt(limitFilter.getFilterValue());
			}
			catch(NumberFormatException ex)
			{
				logger.error("List limit has an incorrect value.");
			}

			if(list.size() > limit && limit > 0)
			{
				// List is longer than allowed. Remove
				// items to meet criteria.
				// perform in place removal of objects.
				while(list.size() > limit)
				{
					list.remove(limit);
				}
			}

			if(limit == 0)
			{
				list.clear();
			}
		}
	}

	/**
	 * This will loop through to get sub elements. If it encounters a list prior
	 * to the end of the keys it will return a FoundList.
	 * 
	 * @param objectName
	 * @param objMap
	 * @return
	 */
	@SuppressWarnings("unchecked")
	private static Object getObject(final String objectName, final SmartHashMap objMap)
	{
		if(StringUtils.isNotBlank(objectName) && objMap != null)
		{
			if(objectName.contains("."))
			{
				int location = objectName.indexOf(".");
				String name = objectName.substring(0, location);
				String toPass = objectName.substring(location + 1);
				Object val = objMap.get(name);
				if(val != null)
				{
					if(val instanceof SmartHashMap)
					{
						return getObject(toPass, (SmartHashMap) val);
					}
					else
					{
						if(val instanceof ArrayList)
						{
							return new FoundList(toPass, (List<SmartHashMap>) val);
						}
					}
				}
			}
			else
			{
				// We are just getting an item.
				return objMap.get(objectName);
			}
		}
		// Failed to find object return null;
		return null;
	}

	@SuppressWarnings("unchecked")
	private boolean matchCriteria(final InternalAppFilterObj filter, final Object value)
	{
		// Check input params.
		boolean ret = false;

		switch(filter.getOperation())
		{
		case CONTAINS_TEXT:
			if(value instanceof String || value == null)
			{
				ret = containsText((String) value, filter.getFilterValue());
			}
			else
			{
				logger.debug("Failed to run Contains Text.  Recieved wrong type. {} ", value);
			}
			break;
		case NOT_CONTAINS_TEXT:
			if(value instanceof String || value == null)
			{
				ret = !containsText((String) value, filter.getFilterValue());
			}
			else
			{
				logger.debug("Failed to run Not Contains Text.  Recieved wrong type. {} ", value);
			}
			break;
		case DATE_GREATER:
			if(value instanceof DataDate || value == null)
			{
				ret = dateGreater((DataDate) value, filter.getFilterValue());
			}
			else
			{
				logger.debug("Failed to run Date Greater.  Recieved wrong type. {} ", value);
			}
			break;
		case DATE_LESS:
			if(value instanceof DataDate || value == null)
			{
				ret = dateLess((DataDate) value, filter.getFilterValue());
			}
			else
			{
				logger.debug("Failed to run Date Less.  Recieved wrong type. {} ", value);
			}
			break;
		case EQUALS_TEXT:
			if(value instanceof String || value == null)
			{
				ret = equalsText((String) value, filter.getFilterValue());
			}
			else
			{
				logger.debug("Failed to run Equals Text.  Recieved wrong type. {} ", value);
			}
			break;
		case NOT_EQUALS_TEXT:
			if(value instanceof String || value == null)
			{
				ret = !equalsText((String) value, filter.getFilterValue());
			}
			else
			{
				logger.debug("Failed to run Not Equals Text.  Recieved wrong type. {}  " + value);
			}
			break;
		case HOLD_PERIOD:
			if(value instanceof DataDate || value == null)
			{
				ret = holdPeriod((DataDate) value, filter.getFilterValue());
			}
			else
			{
				logger.debug("Failed to run Hold period.  Recieved wrong type. {}  ", value);
			}
			break;
		case LIST_LIMIT:
			if(value instanceof List)
			{
				ret = true;
				limitList(filter, (List<Object>) value);
			}
			else
			{
				logger.debug("Failed to run List Limit.  Recieved wrong type. {} ", value);
			}
			break;
		default:
			// Need to log that an invalid parameter was sent in.
		}

		return ret;
	}

	private boolean containsText(final String value, final String filter)
	{
		return StringUtils.contains(value, filter);
	}

	private boolean equalsText(final String value, final String filter)
	{
		return StringUtils.equalsIgnoreCase(filter, value);
	}

	private DataDate convertDate(final String dateStr)
	{
		DataDate ret = null;

		if(StringUtils.isNotBlank(dateStr))
		{
			try
			{
				BigDecimal date = BigDecimal.valueOf(Long.parseLong(dateStr));
				ret = new FileManDateDataTranslator().translate(date, null, new Reference<>(null, null, null), null);
			}
			catch(NumberFormatException ex)
			{
				logger.error("Failed to convert date {} ", dateStr);
			}
		}

		return ret;
	}

	private boolean dateGreater(final DataDate valueDate, final String filter)
	{
		boolean ret = false;
		if(valueDate != null && StringUtils.isNotBlank(filter))
		{
			DataDate filterDate = convertDate(filter);
			if( filterDate != null)
			{
				ret = valueDate.after(filterDate);
			}
		}

		return ret;
	}

	private boolean dateLess(final DataDate valueDate, final String filter)
	{
		boolean ret = false;
		if(valueDate != null && StringUtils.isNotBlank(filter))
		{
			DataDate filterDate = convertDate(filter);
			if(filterDate != null)
			{
				ret = valueDate.before(filterDate);
			}
		}

		return ret;
	}

	private boolean holdPeriod(final DataDate valueDate, final String filter)
	{
		if(valueDate != null && StringUtils.isNotBlank(filter))
		{
			// only compare based on date, not hours, minutes, seconds, ms
			Calendar cutoff = Calendar.getInstance();
			cutoff.set(  Calendar.HOUR_OF_DAY, 0);
			cutoff.set( Calendar.MINUTE, 0 );
			cutoff.set( Calendar.SECOND, 0 );
			cutoff.set( Calendar.MILLISECOND, 0 );

			Calendar testcal = Calendar.getInstance();
			testcal.setTime( valueDate );
			testcal.set( Calendar.HOUR_OF_DAY, 0 );
			testcal.set( Calendar.MINUTE, 0 );
			testcal.set( Calendar.SECOND, 0 );
			testcal.set( Calendar.MILLISECOND, 0 );

			int modifier = Integer.valueOf(filter);
			testcal.add( Calendar.DATE, modifier );

			boolean ok = !testcal.after( cutoff );

			String str = testcal.getTime()
					+ "(" + valueDate + ") not after " + cutoff.getTime() + "? " + ok;
			// System.out.println( str );
			logger.debug( str );
			return ok;
		}

		return false;
	}
}
