package gov.va.med.nhin.adapter.datamanager.filters;

import gov.va.med.nhin.adapter.datamanager.DataDate;
import java.util.List;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import gov.va.med.nhin.adapter.datamanager.DataFilter;
import gov.va.med.nhin.adapter.datamanager.DataManagerException;
import gov.va.med.nhin.adapter.datamanager.DataQuery;
import gov.va.med.nhin.adapter.datamanager.DataTranslator;
import gov.va.med.nhin.adapter.datamanager.Reference;
import gov.va.med.nhin.adapter.datamanager.SmartHashMap;
import java.io.IOException;
import java.lang.reflect.InvocationTargetException;
import java.util.ArrayList;
import java.util.Collections;
import java.util.Comparator;
import java.util.HashMap;
import java.util.Map;
import org.joda.time.LocalDate;
import org.joda.time.LocalDateTime;

/**
 *
 * @author Ry Bobko
 */
public class VitalsFilter implements DataFilter, DataTranslator<List<SmartHashMap>> {

	private static final Logger logger = LoggerFactory.getLogger( VitalsFilter.class.getName() );

	@Override
	public boolean test( Object result, List results, Reference filterType, DataQuery dataQuery ) {
		if ( results.isEmpty() ) {
			return true;
		}

		String field = filterType.getProperty( "dataDateField" );

		SmartHashMap smh = SmartHashMap.class.cast( result );
		Object smhdate = smh.get( field );
		LocalDateTime dd = LocalDateTime.fromDateFields( DataDate.class.cast( smhdate ) );

		List<SmartHashMap> maplist = (List<SmartHashMap>) results;
		List<SmartHashMap> dailies
				= getPastResultsForTodaySorted( dd.toLocalDate(), maplist, field );

		if ( !dailies.isEmpty() ) {
			SmartHashMap mostRecentVitals = dailies.get( 0 );
			LocalDateTime lastVitalsDate = getDate( mostRecentVitals, field );

			if ( dd.isAfter( lastVitalsDate ) ) {
				results.remove( mostRecentVitals );
			}
			else {
				return false;
			}
		}

		return true;
	}

	private static LocalDateTime getDate( SmartHashMap smh, String datefield ) {
		Object date = smh.get( datefield );
		return LocalDateTime.fromDateFields( DataDate.class.cast( date ) );
	}

	private static List<SmartHashMap> getPastResultsForTodaySorted( LocalDate today,
			List<SmartHashMap> oldResults, final String comparatorField ) {

		List<SmartHashMap> todays = new ArrayList<>();

		for ( SmartHashMap smh : oldResults ) {
			LocalDateTime olddate = getDate( smh, comparatorField );
			if ( today.equals( olddate.toLocalDate() ) ) {
				todays.add( smh );
			}
		}

		Collections.sort( todays, new Comparator<SmartHashMap>() {
			@Override
			public int compare( SmartHashMap o1, SmartHashMap o2 ) {
				LocalDateTime d1 = getDate( o1, comparatorField );
				LocalDateTime d2 = getDate( o2, comparatorField );
				return d2.compareTo( d1 );
			}
		} );

		return todays;
	}

	@Override
	public List<SmartHashMap> translate( Object input, Object result,
			Reference translation, DataQuery dataQuery ) throws IOException {

		Map<LocalDate, SmartHashMap> mostRecents = new HashMap<>();
		final String field = translation.getProperty( "dataDateField" );

		//try {
		if ( input != null && input instanceof List ) {
			// only allow the most recent set of vitals per day
			List<SmartHashMap> list = List.class.cast( input );
			for ( SmartHashMap smh : list ) {
				LocalDateTime testtime = getDate( smh, field );
				LocalDate testdate = testtime.toLocalDate();

				if ( mostRecents.containsKey( testdate ) ) {
					LocalDateTime oldtime = getDate( mostRecents.get( testdate ), field );
					if ( testtime.isAfter( oldtime ) ) {
						mostRecents.put( testdate, smh );
					}
				}
				else {
					// first time we've seen this date
					mostRecents.put( testdate, smh );
				}
			}
		}
		//}
		//catch ( DataManagerException | IllegalAccessException | NoSuchMethodException | InvocationTargetException e ) {
		//throw new DataManagerException( "Exception occurred in VitalsFilter (Translator)", e );
		//}

		return new ArrayList<>( mostRecents.values() );
	}
}
