package gov.va.med.nhin.adapter.datamanager.adapters;

import com.sun.xml.ws.client.BindingProviderProperties;
import gov.hhs.fha.nhinc.common.nhinccommon.AssertionType;
import gov.va.med.nhin.adapter.datamanager.DataAdapter;
import gov.va.med.nhin.adapter.datamanager.DataQuery;
import gov.va.med.nhin.adapter.mvi.hl7parsers.HL7Parser201306;
import gov.va.oit.oed.vaww.VAIdMPort;
import java.math.BigInteger;
import java.util.ArrayList;
import java.util.List;
import java.util.Map;

import javax.xml.bind.JAXBContext;
import javax.xml.bind.JAXBException;
import javax.xml.bind.util.JAXBSource;
import javax.xml.ws.BindingProvider;
import javax.xml.ws.WebServiceException;
import javax.xml.ws.soap.SOAPFaultException;

import org.hl7.v3.CS;
import org.hl7.v3.INT;
import org.hl7.v3.PRPAIN201305UV02;
import org.hl7.v3.PRPAIN201305UV02QUQIMT021001UV01ControlActProcess;
import org.hl7.v3.PRPAIN201306UV02;
import org.hl7.v3.PRPAMT201306UV02LivingSubjectId;
import org.hl7.v3.PRPAMT201306UV02LivingSubjectName;
import org.hl7.v3.PRPAMT201306UV02ParameterList;
import org.hl7.v3.PRPAMT201306UV02PatientAddress;
import org.hl7.v3.PRPAMT201306UV02QueryByParameter;
import org.hl7.v3.ST;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

/**
 *
 * @author DNS   
 */
public class MVI1305DataAdapter extends MVIDataAdapter<PRPAIN201306UV02> implements DataAdapter<PRPAIN201306UV02>
{
	private static final Logger logger = LoggerFactory.getLogger(MVI1305DataAdapter.class.getName());
        
        private static JAXBContext context;
        static
        {
            try
            {
                context = JAXBContext.newInstance(PRPAIN201305UV02.class);
            }
            catch (JAXBException e)
            {
                logger.debug("JAXBException occurred at initialization: ", e);
            }
        }
        
	public List<PRPAIN201306UV02> getData(DataQuery dataQuery)
	{
		initializeProperties(dataQuery);

		List<PRPAIN201306UV02> ret = new ArrayList<>();
		PRPAIN201306UV02 response1306 = null;

		PRPAIN201305UV02 partnersRequest1305 = (PRPAIN201305UV02) dataQuery.getParameter("1305Request");
		PRPAIN201305UV02 request1305 = null;

		AssertionType assertion = (AssertionType) dataQuery.getParameter("assertion");
		String sendingFacilityOID = (String) dataQuery.getParameter("sendingFacilityOID");
		String sendingFacilityNumber = (String) dataQuery.getParameter("sendingFacilityNumber");

		// Implement a requirement to retry the operation on failure, up to a
		// retry limit
		int retriesRemaining = mviRetryLimit;
		while(true)
		{
			try
			{
				request1305 = createAdapters1305Request(partnersRequest1305, assertion, sendingFacilityOID, sendingFacilityNumber);
				VAIdMPort port = getVAIdMPort(wsdlURL);
				Map<String, Object> requestContext = ((BindingProvider) port).getRequestContext();
				requestContext.put(BindingProviderProperties.REQUEST_TIMEOUT, mviRequestTimeout);
				requestContext.put(BindingProviderProperties.CONNECT_TIMEOUT, mviConnectTimeout);
				response1306 = (PRPAIN201306UV02) port.prpaIN201305UV02(request1305);
				break;
			}
			catch(SOAPFaultException sfe)
			{
				String msg = sfe.getMessage();
				String sfStr = sfe.getFault().getFaultString();
				String sfCode = sfe.getFault().getFaultCode();

				throw new DataAdapterException("SOAPFaultException occurred in processing MVI 1305 :" + "Message: " + msg + "; " + "FaultString: " + sfStr + ";" + "FaultCode: " + sfCode, sfe);
			}
			catch(WebServiceException ste)
			{
				logger.error("WebServiceException connecting to MVI.  Message={}", ste.getMessage());

				// only execute retries on timeout
				if(ste.getCause() instanceof java.net.SocketTimeoutException)
				{
					if(--retriesRemaining == 0)
					{
						throw new DataAdapterException(ste);
					}
					else
					{
						logger.debug("Retrying connection to MVI due to SocketTimeoutException. {} retries remaining.", retriesRemaining);
					}
				}
				else
				{
					throw new DataAdapterException(ste);
				}
			}
			catch(Throwable t)
			{
				logger.error("There was an error getting data from MVI.  Message={}", t.getMessage());
				throw new DataAdapterException("There was an error getting data from MVI", t);
			}
		}

		ret.add(response1306);

		return ret;
	}

	/*
	 * Build 1305 request
	 * 
	 * @param PRPAIN201305UV02 the 1305 request sent to Adapter.
	 */
	private PRPAIN201305UV02 createAdapters1305Request(PRPAIN201305UV02 partners1305Request, AssertionType assertion, String sendingFacilityOID, String sendingFacilityNumber)
	{
		// ObjectFactory objFactory = new ObjectFactory();

		// Decided to create new instance of 1305 message to use for MVI request
		// so that
		// original inbound 1305 message from partner would be preserved.
		PRPAIN201305UV02 adapter1305Request = null;
		try
		{
			adapter1305Request = (PRPAIN201305UV02) context.createUnmarshaller().unmarshal(new JAXBSource(context, partners1305Request));
		}
		catch(JAXBException e)
		{
			throw new DataAdapterException("JAXBException occurred while making copy of inbound PRPAIN201305UV02 message", e);
		}

		// Set up message header fields
		adapter1305Request.setId(createMessageId(adapter1305Request.getId()));

		adapter1305Request.setCreationTime(createCreationTime());

		// Set Processing Code
		CS processingCode = new CS();
		processingCode.setCode(processingCodeStr);
		adapter1305Request.setProcessingCode(processingCode);

		// Set Acknowlegment Code
		CS acceptAckCode = new CS();
		acceptAckCode.setCode("AL");
		adapter1305Request.setAcceptAckCode(acceptAckCode);

		// Set Sender
		adapter1305Request.setSender(createSender(mviSiteKey1305, sendingFacilityOID, sendingFacilityNumber));

		// Set DataEnterer
		PRPAIN201305UV02QUQIMT021001UV01ControlActProcess controlActProcess = adapter1305Request.getControlActProcess();
		controlActProcess.getDataEnterer().add(createDataEnterer2(assertion));
		// adapter1305Request.setControlActProcess(controlActProcess);

		// QueryByParamter
		PRPAMT201306UV02QueryByParameter queryByParameter = adapter1305Request.getControlActProcess().getQueryByParameter().getValue();

		CS modifyCode = new CS();
		modifyCode.setCode("MVI.COMP1");
		queryByParameter.setModifyCode(modifyCode);

		INT initQty = new INT();
		initQty.setValue(BigInteger.ONE);
		queryByParameter.setInitialQuantity(initQty);

		// Query ParameterList
		PRPAMT201306UV02ParameterList paramList = queryByParameter.getParameterList();

		// MVI requires the use of the semanticsText element on each query
		// parameter....as mandated in the integration process (not enforced at
		// runtime however).
		// Since the VLER Health Exchange Partners have not been sending this
		// element...as it was not needed for the RPC calls...we are checking to
		// see if
		// it's there and if not adding it for them.
		// Gender
		if(paramList.getLivingSubjectAdministrativeGender() != null && paramList.getLivingSubjectAdministrativeGender().size() > 0 && paramList.getLivingSubjectAdministrativeGender().get(0) != null)
		// &&
		// (paramList.getLivingSubjectAdministrativeGender().get(0).getSemanticsText()
		// == null ||
		// NullChecker.isNullOrEmpty(paramList.getLivingSubjectAdministrativeGender().get(0).getSemanticsText().getValue())))
		{
			ST st2 = new ST();
			st2.setValue("Gender");
			paramList.getLivingSubjectAdministrativeGender().get(0).setSemanticsText(st2);
		}

		// Date of Birth
		if(paramList.getLivingSubjectBirthTime() != null && paramList.getLivingSubjectBirthTime().size() > 0 && paramList.getLivingSubjectBirthTime().get(0) != null)
		// && (paramList.getLivingSubjectBirthTime().get(0).getSemanticsText()
		// == null ||
		// NullChecker.isNullOrEmpty(paramList.getLivingSubjectBirthTime().get(0).getSemanticsText().getValue())))
		{
			ST st2 = new ST();
			st2.setValue("Date of Birth");
			paramList.getLivingSubjectBirthTime().get(0).setSemanticsText(st2);
		}

		// Place of Birth
		if(paramList.getLivingSubjectBirthPlaceAddress() != null && paramList.getLivingSubjectBirthPlaceAddress().size() > 0 && paramList.getLivingSubjectBirthPlaceAddress().get(0) != null)
		// &&
		// (paramList.getLivingSubjectBirthPlaceAddress().get(0).getSemanticsText()
		// == null ||
		// NullChecker.isNullOrEmpty(paramList.getLivingSubjectBirthPlaceAddress().get(0).getSemanticsText().getValue())))
		{
			ST st2 = new ST();
			st2.setValue("POB");
			paramList.getLivingSubjectBirthPlaceAddress().get(0).setSemanticsText(st2);
		}

		// SSN
		if(paramList.getLivingSubjectId() != null && paramList.getLivingSubjectId().size() > 0 && paramList.getLivingSubjectId().get(0) != null)
		{
			// only SSN is valid for the person trait search....so remove
			// everything else the Partner may have sent...namely their ID.
			removeNonSSNSubjectIds(paramList.getLivingSubjectId());

			PRPAMT201306UV02LivingSubjectId ssn = this.getSSN(paramList.getLivingSubjectId());
			if(ssn != null)
			// && (ssn.getSemanticsText() == null ||
			// NullChecker.isNullOrEmpty(ssn.getSemanticsText().getValue())))
			{
				ST st2 = new ST();
				st2.setValue("SSN");
				ssn.setSemanticsText(st2);
			}
		}

		// Legal Name
		if(paramList.getLivingSubjectName() != null && paramList.getLivingSubjectName().size() > 0 && paramList.getLivingSubjectName().get(0) != null)
		{
			PRPAMT201306UV02LivingSubjectName legalName = HL7Parser201306.extractSubjectName(paramList.getLivingSubjectName(), "L");
			if(legalName != null)
			// && (legalName.getSemanticsText() == null ||
			// NullChecker.isNullOrEmpty(legalName.getSemanticsText().getValue())))
			{
				if(legalName.getValue().get(0).getUse() == null || legalName.getValue().get(0).getUse().size() <= 0)
				{
					// if null set to "L" otherwise MVI will return "NF" for not
					// found.
					legalName.getValue().get(0).getUse().add("L");
				}
				ST st2 = new ST();
				st2.setValue("Legal Name");
				legalName.setSemanticsText(st2);
			}
		}

		// Mother's Maiden Name
		if(paramList.getLivingSubjectName() != null && paramList.getLivingSubjectName().size() > 0 && paramList.getLivingSubjectName().get(0) != null)
		{
			PRPAMT201306UV02LivingSubjectName maidenName = HL7Parser201306.extractSubjectName(paramList.getLivingSubjectName(), "C");
			if(maidenName != null)
			// && (maidenName.getSemanticsText() == null ||
			// NullChecker.isNullOrEmpty(maidenName.getSemanticsText().getValue())))
			{
				ST st2 = new ST();
				st2.setValue("Mother's Maiden Name");
				maidenName.setSemanticsText(st2);
			}
		}

		// Patient Address
		PRPAMT201306UV02PatientAddress address = HL7Parser201306.extractPatientAddress(paramList.getPatientAddress(), "PHYS");
		if(address != null)
		// && (paramList.getPatientAddress().get(0).getSemanticsText() == null
		// ||
		// NullChecker.isNullOrEmpty(paramList.getPatientAddress().get(0).getSemanticsText().getValue())))
		{
			if(address.getValue().get(0).getUse() == null || address.getValue().get(0).getUse().size() <= 0)
			{
				// if null set to "PHYS" otherwise MVI will not use this
				// parameter and may return a "NF" for not found.
				address.getValue().get(0).getUse().add("PHYS");
			}
			ST st2 = new ST();
			st2.setValue("ADDR");
			address.setSemanticsText(st2);
		}
		return adapter1305Request;
	}

	private void removeNonSSNSubjectIds(List<PRPAMT201306UV02LivingSubjectId> ids)
	{
		List<PRPAMT201306UV02LivingSubjectId> nonSSNIds = new ArrayList();

		for(PRPAMT201306UV02LivingSubjectId id : ids)
		{
			if(id.getValue() != null && id.getValue().size() > 0 && id.getValue().get(0) != null && id.getValue().get(0).getRoot() != null && id.getValue().get(0).getRoot().equals("2.16.840.1.113883.4.1"))
			{
				// it's an SSN so keep it and move on.
				continue;
			}
			else
			{
				// it's not an SSN so remove it since only SSNs are valid for
				// the 1305 queryByParameter/person trait search.
				nonSSNIds.add(id);
			}
		}

		for(PRPAMT201306UV02LivingSubjectId nonSSNId : nonSSNIds)
		{
			ids.remove(nonSSNId);
		}
	}

	private PRPAMT201306UV02LivingSubjectId getSSN(List<PRPAMT201306UV02LivingSubjectId> ids)
	{

		PRPAMT201306UV02LivingSubjectId ssn = null;

		for(PRPAMT201306UV02LivingSubjectId id : ids)
		{
			if(id.getValue() != null && id.getValue().size() > 0 && id.getValue().get(0) != null && id.getValue().get(0).getRoot() != null && id.getValue().get(0).getRoot().equals("2.16.840.1.113883.4.1"))
			{
				ssn = id;
				break;
			}
		}
		return ssn;
	}

}