package gov.va.med.nhin.adapter.datamanager;

import java.text.ParseException;
import java.text.SimpleDateFormat;
import java.util.ArrayList;
import java.util.Collections;
import java.util.Date;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Properties;

import org.apache.commons.beanutils.PropertyUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import gov.va.med.nhin.adapter.datamanager.config.DataTypeType;
import gov.va.med.nhin.adapter.datamanager.config.DeclarationType;
import gov.va.med.nhin.adapter.datamanager.config.FieldType;
import gov.va.med.nhin.adapter.datamanager.config.FieldTypeType;
import gov.va.med.nhin.adapter.datamanager.config.FieldsType;
import gov.va.med.nhin.adapter.datamanager.config.ProcessType;
import gov.va.med.nhin.adapter.datamanager.config.QueryType;
import gov.va.med.nhin.adapter.datamanager.config.ReferenceType;
import gov.va.med.nhin.adapter.datamanager.config.ReferencesType;
import gov.va.med.nhin.adapter.datamanager.config.ResultsType;
import gov.va.med.nhin.adapter.datamanager.config.SortDirectionType;
import gov.va.med.nhin.adapter.utils.LogUtil;
import gov.va.med.nhin.adapter.utils.NullChecker;
import gov.va.med.nhin.adapter.utils.Utils;
import gov.va.med.nhin.adapter.utils.config.PropertiesType;
import gov.va.med.nhin.adapter.utils.config.PropertyType;
import gov.va.med.nhin.adapter.utils.config.PropertyTypeUtils;
import java.lang.reflect.InvocationTargetException;
import java.math.BigInteger;
import java.util.Arrays;
import org.apache.commons.lang.StringUtils;

/**
 *
 * @author David Vazquez
 */
public class DataQueryImpl implements DataQuery
{

    private static final Logger logger = LoggerFactory.getLogger(DataQueryImpl.class.getName());

    private final String name;
    private final DataManagerImpl dataManager;
    private final QueryType queryType;
    private final Map<String, Object> arguements = new HashMap<>();
    private final Properties properties = new Properties();

    DataQueryImpl(DataManagerImpl dataManager, QueryType queryType, String name)
    {
        try {
            this.dataManager = dataManager;
            this.queryType = queryType;
            this.name = name;
        }
        catch (Throwable t) {
            throw new DataManagerException("Error occurred constructing DataQuery.", t);
        }
    }

    @Override
    public void setParameter(String name, Object value)
    {
        if (isParameter(name)) {
            arguements.put(name, value);
        }
        else if (System.getProperty("gov.va.med.nhin.adapter.datamanager.Strict") != null) {
            throw new InvalidArguementException(getName(), name);
        }
        else {
            logger.debug("Invalid argument ({}) passed to {}", LogUtil.cleanLogMessage(name), LogUtil.cleanLogMessage(getName()));
        }
    }

    @Override
    public String[] getParameterNames()
    {
        List<String> ret = new ArrayList<>();

        for (FieldType qp : getQueryType().getParameters().getField()) {
            ret.add(qp.getName());
        }

        return ret.toArray(new String[ret.size()]);
    }

    @Override
    public String[] getResultNames()
    {
        List<String> ret = new ArrayList<>();

        for (FieldType qr : getQueryType().getResults().getFields().getField()) {
            ret.add(qr.getName());
        }

        return ret.toArray(new String[ret.size()]);
    }

    @Override
    public boolean isParameter(String name)
    {
        return getQueryParameter(name) != null;
    }

    @Override
    public boolean isResult(String name)
    {
        return getQueryResult(name) != null;
    }

    @Override
    public String getName()
    {
        return name;
    }

    @Override
    public Object getParameter(String name)
    {
        Object ret = null;

        if (isParameter(name)) {
            ret = arguements.get(name);
        }
        else if (System.getProperty("gov.va.med.nhin.adapter.datamanager.Strict") != null) {
            throw new InvalidArguementException(getName(), name);
        }
        else {
            logger.debug("Invalid argument ({}) passed to {}", LogUtil.cleanLogMessage(name), LogUtil.cleanLogMessage(getName()));
        }

        return ret;
    }

    @Override
    public Object getParameterBySource(String name)
    {
        Object ret = null;
        FieldType fieldType = getQueryParameterBySource(name);

        if (fieldType != null) {
            ret = arguements.get(fieldType.getName());
        }

        return ret;
    }

    @Override
    public String[] getParameterBySourceNames()
    {
        List<String> ret = new ArrayList<>();

        for (FieldType qp : getQueryType().getParameters().getField()) {
            if (NullChecker.isNotNullOrEmpty(qp.getSource())) {
                ret.addAll(qp.getSource());
            }
        }

        return ret.toArray(new String[ret.size()]);
    }

    @Override
    public String[] getResultBySourceNames()
    {
        List<String> ret = new ArrayList<>();

        for (FieldType qr : getQueryType().getResults().getFields().getField()) {
            ret.addAll(qr.getSource());
        }

        return ret.toArray(new String[ret.size()]);
    }

    @Override
    public List getResults()
    {
        List ret = null;

        logger.debug("Entering query {}", getName());

        try {
            logger.trace("Parameters:\n{}\nProperties:\n{}", LogUtil.cleanLogMessage(arguements.toString()), LogUtil.cleanLogMessage(properties.toString()));

            checkRequiredParameters();
            processParameters();

            logger.trace("Parameters after processing:\n{}\nProperties:\n{}", LogUtil.cleanLogMessage(arguements.toString()), LogUtil.cleanLogMessage(properties.toString()));

            List results = getDataAdapter().getData(this);
            logger.trace("Results from adapter:\n{}", LogUtil.cleanLogMessage(results.toString()));

            if (getQueryType().getResults().isPassthrough()) {
                return results;
            }

            if (getQueryType().getParsers() != null) {
                for (ReferenceType parser : getQueryType().getParsers().getReference()) {
                    results = getDataManager().invokeDataParser(parser, results, this);
                }
                logger.trace("Results after parse:\n{}", LogUtil.cleanLogMessage(results.toString()));
            }

            ret = processResults(results, getQueryType().getResults());

            logger.trace("Returning Results:\n{}", LogUtil.cleanLogMessage(ret.toString()));
        }
        catch (DataManagerException dme) {
            throw dme;
        }
        catch (Throwable t) {
            logger.debug("Error occurred during processing of {}", getName());
            throw new DataManagerException("An error occurred.", t);
        }
        finally {
            logger.debug("Exitting query {}", getName());
        }

        return ret;
    }

    private void checkRequiredParameters()
    {
        for (FieldType parameter : getQueryType().getParameters().getField()) {
            if (parameter.isRequired() && NullChecker.isNullOrEmpty(getParameter(parameter.getName()))) {
                throw new RequiredParameterMissingException(parameter.getName());
            }
        }
    }

    private void processParameters() throws Exception
    {
        DataProxy sourceDataProxy = getDataProxy(getQueryType().getParameters().getSourceDataType());
        DataProxy targetDataProxy = getDataProxy(getQueryType().getParameters().getTargetDataType());
        Map<String, Object> processedArguements = new HashMap<>();

        sourceDataProxy.setObject(arguements);
        targetDataProxy.setObject(processedArguements);

        for (FieldType ft : getQueryType().getParameters().getField()) {
            if ((ft.getFieldType() == FieldTypeType.STRING || ft.getFieldType() == FieldTypeType.REAL || ft.getFieldType() == FieldTypeType.INTEGER || ft.getFieldType() == FieldTypeType.NUMBER || ft.getFieldType() == FieldTypeType.DATE) && !ft.isCollection()) {
                Object value = getValue(ft.getName(), ft.getDefaultValue(), sourceDataProxy, targetDataProxy);
                Object valueAsTargetType = convertValue(value, processedArguements, arguements, ft);
                targetDataProxy.set(ft.getName(), valueAsTargetType);
            }
            else if (ft.getFieldType() == FieldTypeType.OBJECT) {
                targetDataProxy.set(ft.getName(), getParameter(ft.getName()));
            }
        }

        arguements.clear();
        arguements.putAll(processedArguements);
    }

    @Override
    public DataManagerImpl getDataManager()
    {
        return dataManager;
    }

    private List processResults(List results, ResultsType resultsType) throws Exception
    {
        List ret = new ArrayList();
        List retSource = new ArrayList();

        DataProxy sourceDataProxy = null;
        DataProxy targetDataProxy = null;

        if (results != null) {
            if (NullChecker.isNotNullOrEmpty(resultsType.getFields())) {
                sourceDataProxy = getDataProxy(resultsType.getFields().getSourceDataType());
                targetDataProxy = getDataProxy(resultsType.getFields().getTargetDataType());

                for (Object result : results) {
                    Object retResult = createResult(resultsType.getFields().getClassName());
                    retResult = processResult(result, sourceDataProxy, targetDataProxy, resultsType.getFields(), ProcessType.PRE_FILTER, retResult);
                    if (isValidResult(retResult, ret, resultsType.getFilters())) {
                        if (queryType.isOverwrite()) {
                            ret.clear();
                        }
                        ret.add(retResult);
                        retSource.add(result);
                    }
                }
            }
            else {
                for (Object result : results) {
                    if (isValidResult(result, ret, resultsType.getFilters())) {
                        if (queryType.isOverwrite()) {
                            ret.clear();
                        }
                        ret.add(result);
                        retSource.add(result);
                    }
                }
            }
        }

        logger.trace("Results after pre-filter processing and filtering:\n{}", LogUtil.cleanLogMessage(ret.toString()));

        if (NullChecker.isNotNullOrEmpty(resultsType.getTranslations())) {
            ret = (List) translateValue(ret, null, null, resultsType.getTranslations());
        }

        logger.trace("Results after translating:\n{}", LogUtil.cleanLogMessage(ret.toString()));

        if (NullChecker.isNotNullOrEmpty(resultsType.getFields())) {
            for (int i = 0; i < ret.size(); ++i) {
                processResult(retSource.get(i), sourceDataProxy, targetDataProxy, resultsType.getFields(), ProcessType.POST_FILTER, ret.get(i));
            }
        }

        logger.trace("Results after post filter processing:\n{}", LogUtil.cleanLogMessage(ret.toString()));

        if (NullChecker.isNotNullOrEmpty(resultsType.getSortField())) {
            final List<String> sortField = resultsType.getSortField();
            final List<SortDirectionType> sortDirection = resultsType.getSortDirection();
            Collections.sort(ret, new java.util.Comparator()
            {
                @Override
                public int compare(Object o1, Object o2)
                {
                    int ret = 0;

                    try {
                        for (int i = 0; ret == 0 && i < sortField.size(); ++i) {
                            Comparable left = (Comparable) PropertyUtils.getProperty(o1, sortField.get(i));
                            Comparable right = (Comparable) PropertyUtils.getProperty(o2, sortField.get(i));
                            if (left == null && right == null) {
                                ret = 0;
                            }
                            else if (left == null) {
                                ret = 1;
                            }
                            else if (right == null) {
                                ret = -1;
                            }
                            else {
                                ret = left.compareTo(right);
                            }
                            if (ret != 0 && NullChecker.isNotNullOrEmpty(sortDirection)) {
                                if (sortDirection.size() > i) {
                                    if (sortDirection.get(i) == SortDirectionType.DESC) {
                                        ret *= -1;
                                    }
                                }
                            }
                        }
                    }
                    catch (IllegalAccessException | NoSuchMethodException | InvocationTargetException e) {
                        ret = 0;
                    }

                    return ret;
                }
            });
        }
        logger.trace("Results after sorting:\n{}", LogUtil.cleanLogMessage(ret.toString()));

        if (resultsType.getOffset() != null) {
            int offset = resultsType.getOffset().intValue();
            if (offset < ret.size()) {
                ret = ret.subList(resultsType.getOffset().intValue(), ret.size());
            }
            else {
                ret.clear();
            }
        }

        int maxResults;
        if (resultsType.getMaxResults() != null) {
            maxResults = resultsType.getMaxResults().intValue();
        }
        else {
            maxResults = getDefaultMaxResults();
        }

        if (NullChecker.isNotNullOrEmpty(resultsType.getMaxResultsOverrideParam())
            && isParameter(resultsType.getMaxResultsOverrideParam())
            && NullChecker.isNotNullOrEmpty(getParameter(resultsType.getMaxResultsOverrideParam()))) {
            if (resultsType.getMaxResultsOverride() != null
                && resultsType.getMaxResultsOverride().intValue() >= 0) {
                maxResults = resultsType.getMaxResultsOverride().intValue();
            }
            else {
                maxResults = getDefaultMaxResults();
            }
        }

        if (ret.size() > maxResults) {
            ret.subList(maxResults, ret.size()).clear();
        }

        logger.trace("Returning processed results:\n{}", LogUtil.cleanLogMessage(ret.toString()));

        return ret;
    }

    private Object createResult(String className) throws Exception
    {
        Object retResult;

        if (NullChecker.isNullOrEmpty(className)) {
            retResult = new SmartHashMap();
        }
        else {
            ClassLoader classLoader = Thread.currentThread().getContextClassLoader();
            Class c = classLoader.loadClass(className);
            if (c != null) {
                retResult = c.newInstance();
            }
            else {
                throw new DataManagerException("Could not make an instance of " + className + ".");
            }
        }

        return retResult;
    }

    private DataProxy getDataProxy(DataTypeType dataType)
    {
        DataProxy ret;

        if (dataType == DataTypeType.BEAN) {
            ret = new BeanDataProxy();
        }
        else {
            ret = new MappedDataProxy();
        }

        return ret;
    }

    private Object processResult(Object result, DataProxy sourceDataProxy, DataProxy targetDataProxy, FieldsType fields, ProcessType process, Object ret) throws Exception
    {
        sourceDataProxy.setObject(result);
        targetDataProxy.setObject(ret);

        for (FieldType qr : fields.getField()) {

            if (process == qr.getProcess()) {

                Object value = getValue(qr.getSource().get(0), qr.getDefaultValue(), sourceDataProxy, targetDataProxy);

                switch (qr.getFieldType()) {
                    case DATARESULT_ROOT:
                        if (NullChecker.isNotNullOrEmpty(value)) {
                            List listValue = (List) value;
                            if (NullChecker.isNotNullOrEmpty(listValue)) {
                                ((SmartHashMap) ret).putAll((Map) listValue.get(0));
                            }
                        }
                        break;

                    case DATARESULT:
                        if (NullChecker.isNotNullOrEmpty(value)) {
                            List listValue = (List) value;
                            if (NullChecker.isNotNullOrEmpty(listValue)) {
                                targetDataProxy.set(qr.getName(), listValue.get(0));
                            }
                        }
                        break;

                    case DATARESULTS:
                        if (NullChecker.isNotNullOrEmpty(value)) {
                            targetDataProxy.set(qr.getName(), value);
                        }
                        break;

                    case DATARESULTS_ROOT:
                        if (NullChecker.isNotNullOrEmpty(value)) {
                            List<Map> mapList = (List) value;
                            if (NullChecker.isNotNullOrEmpty(mapList)) {
                                SmartHashMap retMap = (SmartHashMap) ret;
                                for (Map m : mapList) {
                                    retMap.putAll(m);
                                }
                            }
                        }
                        break;

                    case DEEP_OBJECT:
                        if (NullChecker.isNotNullOrEmpty(value)) {
                            if (!qr.isCollection()) {
                                DataProxy subSourceDataProxy = getDataProxy(qr.getResults().getFields().getSourceDataType());
                                DataProxy subTargetDataProxy = getDataProxy(qr.getResults().getFields().getTargetDataType());
                                Object o = createResult(qr.getResults().getFields().getClassName());
                                targetDataProxy.set(qr.getName(), processResult(value, subSourceDataProxy, subTargetDataProxy, qr.getResults().getFields(), process, o));
                            }
                            else {
                                List r = processResults((List) value, qr.getResults());

                                /*
							 * The augment "feature" will add a value to each
							 * element in the list from the parent element so it
							 * can be leveraged within the mapping loop
                                 */
                                if (qr.getProperties() != null) {
                                    for (PropertyType pt : qr.getProperties().getProperty()) {
                                        if ("augment".equals(pt.getName()) && pt.getValue() != null) {
                                            Object o = sourceDataProxy.get(pt.getValue());
                                            if (o != null) {
                                                for (Object rr : r) {
                                                    if (rr instanceof HashMap) {
                                                        sourceDataProxy.get(pt.getValue());
                                                        ((HashMap) rr).put(pt.getValue(), targetDataProxy.get(pt.getValue()));
                                                    }
                                                }
                                            }
                                        }
                                    }
                                }

                                if (NullChecker.isNotNullOrEmpty(r)) {
                                    targetDataProxy.set(qr.getName(), r);
                                }
                            }
                        }
                        break;

                    default:
                        if (!qr.isCollection()) {
                            Object valueAsTargetType = convertValue(value, ret, result, qr);
                            targetDataProxy.set(qr.getName(), valueAsTargetType);
                        }
                        else {
                            List collectionAsTargetType = new ArrayList();
                            for (int i = 0; i < Utils.getCollectionLength(value); ++i) {
                                collectionAsTargetType.add(convertValue(PropertyUtils.getIndexedProperty(value, "", i), ret, result, qr));
                            }
                            targetDataProxy.set(qr.getName(), collectionAsTargetType);
                        }
                        break;
                }
            }
        }

        return ret;
    }

    private Object getValue(String property, String defaultValue, DataProxy sourceDataProxy, DataProxy targetDataProxy) throws Exception
    {
        Object ret = targetDataProxy.get(property);
        if (ret == null) {
            ret = sourceDataProxy.get(property);

            if (ret == null && defaultValue != null) {
                if (defaultValue.equals("$queryName")) {
                    ret = getName();
                }
                else if (defaultValue.equals("$source")) {
                    ret = sourceDataProxy.getObject();
                }
                else if (defaultValue.equals("$result")) {
                    ret = targetDataProxy.getObject();
                }
                else if (defaultValue.startsWith("$")) {
                    String fieldGeneratorName = getFieldGeneratorName(defaultValue.substring(1));
                    String[] fieldGeneratorParams = getFieldGeneratorParams(defaultValue.substring(1));
                    ret = getDataManager().getDataFieldGenerator(fieldGeneratorName).generate(fieldGeneratorParams);
                }
                else if (defaultValue.startsWith("=")) {
                    ret = getParameter(defaultValue.substring(1));
                }
                else if (defaultValue.startsWith("#")) {
                    ret = expandProperty(defaultValue.substring(1));
                }
                else {
                    ret = defaultValue;
                }
            }
        }

        return ret;
    }

    private String getFieldGeneratorName(String value)
    {
        String ret = value;
        int i = value.indexOf('(');
        if (i >= 0) {
            ret = value.substring(0, i);
        }
        return ret;
    }

    private String[] getFieldGeneratorParams(String value)
    {
        List<String> ret = new ArrayList<>();
        int i = value.indexOf("(");
        if (i >= 0) {
            int j = value.indexOf(")");
            if (j >= 1) {
                String paramsString = value.substring(i + 1, j);
                if (!paramsString.isEmpty()) {
                    ret.addAll(Arrays.asList(paramsString.split(",")));
                }
            }
        }
        return ret.toArray(new String[ret.size()]);
    }

    /*
	 * private Object doSubquery(FieldType field, Object result) throws
	 * Exception { List ret; DataQuery subQuery =
	 * getDataManager().getQuery(field.getSource().get(0));
	 * 
	 * for (String parameterName : getParameterNames()) { if
	 * (subQuery.isParameter(parameterName)) {
	 * subQuery.setParameter(parameterName, getParameter(parameterName)); } }
	 * 
	 * for (String resultName : getResultNames()) { if
	 * (subQuery.isParameter(resultName)) { subQuery.setParameter(resultName,
	 * PropertyUtils.getProperty(result, resultName)); } }
	 * 
	 * String forwardProperties =
	 * PropertyTypeUtils.getProperty(field.getProperties(),
	 * "forwardProperties"); if
	 * (NullChecker.isNotNullOrEmpty(forwardProperties)) { for (String
	 * forwardProperty : forwardProperties.split(",")) {
	 * subQuery.setProperty(forwardProperty, getProperty(forwardProperty)); } }
	 * 
	 * ret = subQuery.getResults();
	 * 
	 * return ret; }
     */
    private Object convertValue(Object value, Object result, Object sourceResult, FieldType fieldType) throws Exception
    {
        Object ret = null;

        switch (fieldType.getFieldType()) {
            case DATE:
                if (NullChecker.isNotNullOrEmpty(value)) {
                    if (value instanceof String) {
                        String valueAsString = (String) value;

                        if (fieldType.getSourceFormatString() == null) {
                            fieldType.setSourceFormatString("MMM dd, yyyy");
                        }

                        SimpleDateFormat formatter = new SimpleDateFormat(fieldType.getSourceFormatString());
                        try {
                            ret = new DataDate(formatter.parse(valueAsString).getTime(), fieldType.getTargetFormatString());
                        }
                        catch (ParseException pe) {
                            logger.warn("Unable to parse date. Return null.");
                            ret = null;
                        }
                    }
                    else if (value instanceof Number) {
                        Number valueAsNumber = (Number) value;
                        ret = new DataDate(valueAsNumber.longValue(), fieldType.getTargetFormatString());
                    }
                    else if (value instanceof Date) {
                        Date valueAsDate = (Date) value;
                        ret = new DataDate(valueAsDate.getTime(), fieldType.getTargetFormatString());
                    }
                    else if (value instanceof DataDate) {
                        DataDate valueAsDataDate = (DataDate) value;
                        ret = new DataDate(valueAsDataDate.getTime(), fieldType.getTargetFormatString());
                    }
                }
                break;

            case INTEGER:
                if (NullChecker.isNotNullOrEmpty(value)) {
                    if (value instanceof String) {
                        String valueAsString = (String) value;
                        try {
                            ret = Integer.parseInt(valueAsString);
                        }
                        catch (NumberFormatException nfe) {
                            ret = null;
                        }
                    }
                    else if (value instanceof Number) {
                        ret = ((Number) value).intValue();
                    }
                }
                break;

            case REAL:
            case NUMBER:
                if (NullChecker.isNotNullOrEmpty(value)) {
                    if (value instanceof String) {
                        String valueAsString = (String) value;
                        try {
                            ret = Double.parseDouble(valueAsString);
                        }
                        catch (NumberFormatException nfe) {
                            ret = null;
                        }
                    }
                    else if (value instanceof Number) {
                        ret = ((Number) value).doubleValue();
                    }
                }
                break;

            default:
                ret = value;
                break;
        }

        if (NullChecker.isNotNullOrEmpty(fieldType.getTranslations())) {
            ret = translateValue(ret, result, sourceResult, fieldType.getTranslations());
        }

        return ret;
    }

    private Object translateValue(Object value, Object result, Object sourceResult, ReferencesType translationsType) throws Exception
    {
        Object ret = value;

        for (ReferenceType tt : translationsType.getReference()) {
            ret = getDataManager().invokeDataTranslator(tt, ret, result, sourceResult, this);
        }

        return ret;
    }

    private boolean isValidResult(Object result, List results, ReferencesType filters) throws Exception
    {
        boolean ret = true;

        if (NullChecker.isNotNullOrEmpty(filters)) {
            for (ReferenceType filterType : filters.getReference()) {
                ret &= getDataManager().invokeDataFilter(filterType, result, results, this);
                if (!ret) {
                    break;
                }
            }
        }

        return ret;
    }

    @Override
    public <T> List<T> getResults(Class<T> cls)
    {
        return (List<T>) getResults();
    }

    public QueryType getQueryType()
    {
        return queryType;
    }

    private DataAdapter<?> getDataAdapter() throws Exception
    {
        return getDataManager().getDataAdapter(queryType.getAdapter());
    }

    @Override
    public void setProperty(String name, String value)
    {
        properties.setProperty(name, value);
    }

    @Override
    public String getProperty(String name)
    {
        return getProperty(name, null);
    }

    public String getProperty(String name, String defaultValue)
    {
        // FIXME: we never use defaultValue in this function?

        String ret = properties.getProperty(name);

        if (ret == null) {
            ret = PropertyTypeUtils.getProperty(getQueryType().getProperties(), name);
        }

        if (ret == null) {
            ret = PropertyTypeUtils.getProperty(getQueryType().getAdapter().getProperties(), name);
        }

        if (ret == null) {
            ret = PropertyTypeUtils.getProperty(getDataManager().getAdapterDef(getQueryType().getAdapter().getName()).getProperties(), name);
        }

        return ret;
    }

    @Override
    public PropertiesType getProperties(String name)
    {
        PropertiesType ret = null;

        if (ret == null) {
            ret = PropertyTypeUtils.getProperties(getQueryType().getProperties(), name);
        }

        if (ret == null) {
            ret = PropertyTypeUtils.getProperties(getQueryType().getAdapter().getProperties(), name);
        }

        if (ret == null) {
            ret = PropertyTypeUtils.getProperties(getDataManager().getAdapterDef(getQueryType().getAdapter().getName()).getProperties(), name);
        }

        return ret;
    }

    @Override
    public String[] getPropertyNames()
    {
        HashSet ret = new HashSet();

        ret.addAll(properties.keySet());

        if (getQueryType().getProperties() != null) {
            for (PropertyType pt : getQueryType().getProperties().getProperty()) {
                ret.add(pt.getName());
            }
        }

        if (getQueryType().getAdapter().getProperties() != null) {
            for (PropertyType pt : getQueryType().getAdapter().getProperties().getProperty()) {
                ret.add(pt.getName());
            }
        }

        DeclarationType adt = getDataManager().getAdapterDef(getQueryType().getAdapter().getName());
        if (adt.getProperties() != null) {
            for (PropertyType pt : adt.getProperties().getProperty()) {
                ret.add(pt.getName());
            }
        }

        return (String[]) ret.toArray(new String[ret.size()]);
    }

    @Override
    public String toString()
    {
        return getName();
    }

    private FieldType getQueryParameter(String name)
    {
        FieldType ret = null;

        for (FieldType f : queryType.getParameters().getField()) {
            if (f.getName().equals(name)) {
                ret = f;
                break;
            }
        }

        return ret;
    }

    private FieldType getQueryResult(String name)
    {
        FieldType ret = null;

        for (FieldType f : queryType.getResults().getFields().getField()) {
            if (f.getName().equals(name)) {
                ret = f;
                break;
            }
        }

        return ret;
    }

    public FieldType getQueryParameterBySource(String name)
    {
        FieldType ret = null;

        for (FieldType f : queryType.getParameters().getField()) {
            if (NullChecker.isNotNullOrEmpty(f.getSource())) {
                for (String src : f.getSource()) {
                    if (src.equals(name)) {
                        ret = f;
                    }
                }
            }
        }

        return ret;
    }

    private String expandProperty(String property)
    {
        String ret;
        String[] s = property.split(",");

        ret = getProperty(s[0]);
        if (ret == null) {
            ret = System.getProperty(s[0]);
        }

        if (ret == null && s.length > 1) {
            ret = s[1];
        }

        return ret;
    }

    private static final int DEFAULT_MAX_RESULTS = 5000;

    private int getDefaultMaxResults()
    {
        int ret = DEFAULT_MAX_RESULTS;

        String s = System.getProperty("gov.va.med.nhin.adapter.datamanager.DEFAULT_MAX_RESULTS");
        if (NullChecker.isNotNullOrEmpty(s) && StringUtils.isNumeric(s) && s.indexOf(".") == -1) {
            ret = Integer.parseInt(s);
        }

        return ret;
    }
}
