package gov.va.med.nhin.adapter.patientcorrelation;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import javax.ejb.EJB;
import javax.ejb.*;
import javax.jws.WebService;

import org.hl7.v3.ADExplicit;
import org.hl7.v3.AddPatientCorrelationRequestType;
import org.hl7.v3.AddPatientCorrelationResponseType;
import org.hl7.v3.CE;
import org.hl7.v3.II;
import org.hl7.v3.MCCIMT000300UV01Acknowledgement;
import org.hl7.v3.MFMIMT700711UV01QueryAck;
import org.hl7.v3.PRPAIN201301UV02;
import org.hl7.v3.PRPAIN201309UV02;
import org.hl7.v3.PRPAIN201310UV02;
import org.hl7.v3.PRPAMT201301UV02OtherIDs;
import org.hl7.v3.PRPAMT201301UV02Patient;
import org.hl7.v3.PRPAMT201301UV02Person;
import org.hl7.v3.PRPAMT201307UV02PatientIdentifier;
import org.hl7.v3.RetrievePatientCorrelationsRequestType;
import org.hl7.v3.RetrievePatientCorrelationsResponseType;
import org.hl7.v3.TELExplicit;
import org.hl7.v3.TSExplicit;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import gov.hhs.fha.nhinc.common.nhinccommon.AssertionType;
import gov.hhs.fha.nhinc.common.nhinccommon.PersonNameType;
import gov.hhs.fha.nhinc.patientcorrelation.nhinc.model.QualifiedPatientIdentifier;
import gov.hhs.fha.nhinc.patientdb.model.Address;
import gov.hhs.fha.nhinc.patientdb.model.Identifier;
import gov.hhs.fha.nhinc.patientdb.model.Patient;
import gov.hhs.fha.nhinc.patientdb.model.Personname;
import gov.va.med.nhin.adapter.adaptergateway.exception.AdapterPatientCorrelationException;
import gov.va.med.nhin.adapter.audit.Audit;
import gov.va.med.nhin.adapter.audit.AuditManager;
import gov.va.med.nhin.adapter.audit.AuditManagerLocal;
import gov.va.med.nhin.adapter.datamanager.DataManager;
import gov.va.med.nhin.adapter.datamanager.DataQuery;
import gov.va.med.nhin.adapter.datamanager.ejb.DataManagerLocal;
import gov.va.med.nhin.adapter.facilitymanager.Facility;
import gov.va.med.nhin.adapter.facilitymanager.FacilityManager;
import gov.va.med.nhin.adapter.facilitymanager.FacilityManagerLocal;
import gov.va.med.nhin.adapter.mvi.hl7parsers.HL7Parser201301;
import gov.va.med.nhin.adapter.mvi.hl7parsers.HL7Parser201305;
import gov.va.med.nhin.adapter.mvi.hl7parsers.HL7Parser201306;
import gov.va.med.nhin.adapter.patientcorrelation.ack.AckBuilder;
import gov.va.med.nhin.adapter.patientcorrelation.parsers.PRPAIN201301UV.PRPAIN201301UVParser;
import gov.va.med.nhin.adapter.patientcorrelation.parsers.PRPAIN201309UV.PRPAIN201309UVParser;
import gov.va.med.nhin.adapter.patientcorrelation.parsers.PRPAIN201309UV.PixRetrieveResponseBuilder;
import gov.va.med.nhin.adapter.propertylookup.PropertyLookup;
import gov.va.med.nhin.adapter.propertylookup.PropertyLookupLocal;
import gov.va.med.nhin.adapter.utils.AuditUtil;
import gov.va.med.nhin.adapter.utils.NullChecker;

/**
 *
 * @author David Vazquez
 */
// No need to be a web service right now.
//@WebService(serviceName = "PatientCorrelationService", portName = "PatientCorrelationPort", endpointInterface = "gov.hhs.fha.nhinc.nhinccomponentpatientcorrelation.PatientCorrelationPortType", targetNamespace = "urn:gov:hhs:fha:nhinc:nhinccomponentpatientcorrelation", wsdlLocation = "META-INF/wsdl/NhincComponentPatientCorrelation.wsdl")
@TransactionAttribute(value = TransactionAttributeType.SUPPORTS)
@Stateless(name = "AdapterPatientCorrelation")
public class AdapterPatientCorrelation implements PatientCorrelationPortTypeLocal
{
    private static final Logger logger = LoggerFactory.getLogger(AdapterPatientCorrelation.class.getName());

    static private String SSA_ORGANIZATION_ID = "2.16.840.1.113883.3.184";

    private PropertyLookup propertyLookup;
    private FacilityManager facilityManager;
    private AuditManager auditManager;
    private DataManager dataManager;

    @EJB(beanInterface = PropertyLookupLocal.class, beanName = "PropertyFileLookup")
    public void setPropertyLookup(PropertyLookup propertyLookup)
    {
        this.propertyLookup = propertyLookup;
    }

    @EJB(beanInterface = FacilityManagerLocal.class, beanName = "FacilityManager")
    public void setFacilityManager(FacilityManager facilityManager)
    {
        this.facilityManager = facilityManager;
    }

    @EJB(beanInterface = AuditManagerLocal.class, beanName = "AuditManager")
    public void setAuditManager(AuditManager auditManager)
    {
        this.auditManager = auditManager;
    }

    @EJB(beanInterface = DataManagerLocal.class, beanName = "DataManager")
    public void setDataManager(DataManager dataManager)
    {
        this.dataManager = dataManager;
    }

    public RetrievePatientCorrelationsResponseType retrievePatientCorrelations(RetrievePatientCorrelationsRequestType retrievePatientCorrelationsRequest)
    {
        logger.debug("retrievePatientCorrelations() invoked");

        RetrievePatientCorrelationsResponseType result = new RetrievePatientCorrelationsResponseType();

        PRPAIN201309UV02 IN201309 = retrievePatientCorrelationsRequest.getPRPAIN201309UV02();
        AssertionType assertion = retrievePatientCorrelationsRequest.getAssertion();

        PRPAMT201307UV02PatientIdentifier patIdentifier = PRPAIN201309UVParser.parseHL7PatientPersonFrom201309Message(IN201309);

        logger.debug("patient identifier {}", patIdentifier);

        if (patIdentifier == null) {
            return null;
        }
        List<II> listII = patIdentifier.getValue();
        if (listII == null) {
            return null;
        }
        if (listII.get(0) == null) {
            return null;
        }

        II inputPatientId = listII.get(0);
        List<String> dataSourceList = PRPAIN201309UVParser.extractDataSourceList(IN201309);

        QualifiedPatientIdentifier inputQualifiedPatientIdentifier = QualifiedPatientIdentifierFactory(inputPatientId);
        List<QualifiedPatientIdentifier> qualifiedPatientIdentifiers = findCorrelations(inputQualifiedPatientIdentifier, dataSourceList, assertion);

        // CCR 177986
        logger.debug("qualified Patient correlations/identifiers {} ", qualifiedPatientIdentifiers);

        if (qualifiedPatientIdentifiers != null && qualifiedPatientIdentifiers.size() > 0) {
            List<II> iiList = buildList(qualifiedPatientIdentifiers);
            PRPAIN201310UV02 IN201310 = PixRetrieveResponseBuilder.createPixRetrieveResponse(IN201309, iiList);
            result.setPRPAIN201310UV02(IN201310);

            // CCR 177986
            logger.debug("Patient correlations {} ", result);
        }

        return result;
    }

    public AddPatientCorrelationResponseType addPatientCorrelation(AddPatientCorrelationRequestType addPatientCorrelationRequest)
    {
        return addPatientCorrelation(addPatientCorrelationRequest, null, false);
    }

    public AddPatientCorrelationResponseType addPatientCorrelation(AddPatientCorrelationRequestType addPatientCorrelationRequest, Patient searchResultPatient, boolean correlationAlreadyExists)
    {
        logger.debug("addPatientCorrelation() invoked");

        AddPatientCorrelationResponseType result = new AddPatientCorrelationResponseType();

        PRPAIN201301UV02 IN201301 = addPatientCorrelationRequest.getPRPAIN201301UV02();
        PRPAMT201301UV02Patient patient = PRPAIN201301UVParser.ParseHL7PatientPersonFrom201301Message(IN201301);
        AssertionType assertion = addPatientCorrelationRequest.getAssertion();

        String remoteAssigningAuthorityID = HL7Parser201301.extractRemoteAssigningAuthorityID(IN201301);

        logger.debug("remote auth id {} ", remoteAssigningAuthorityID);

        /*
		 * Calls to addPatientCorrelation coming from
		 * AdapterGatewayPatientDiscovery currently currently do not pass in the
		 * remote Assigning Authority (although it could be extracted from the
		 * partner's PD 1306 response. So for now allow a null and the
		 * getRemoteId() method below will handle accordingly.
         */
 /*
		 * if (NullChecker.isNullOrEmpty(remoteAssigningAuthorityID)) { String
		 * errMsg = "Missing Remote Assigning Authority";
		 * logger.log(Level.SEVERE, errMsg); throw new RuntimeException(errMsg);
		 * }
         */
        II senderId = getSenderId(IN201301);
        II localId = getLocalId(patient);
        II remoteId = getRemoteId(patient, remoteAssigningAuthorityID);

        if (senderId != null && localId != null && remoteId != null) {
            // CCR 177986
            logger.debug("addPatientCorrelation preparing request for HCID {} ", senderId.getRoot());

            Facility facility = facilityManager.getFacilityByHomeCommunityId(senderId.getRoot());

            Map map = null;
            PRPAMT201301UV02Person person = PRPAIN201301UVParser.ParseHL7PatientPersonFromHL7Patient(patient);
            if (person != null) {
                map = getPatientMap(patient);
            }

            String icnValue = HL7Parser201306.extractICNValue(localId.getExtension());

            String auditDetails = correlationAlreadyExists ? "Correlation already exists" : "Correlation added";
            auditDetails += ", ICN=" + icnValue;
            auditDetails += ", correlatedPatientId=" + remoteId.getExtension();
            auditDetails += ", correlatedAssigningAuthority=" + remoteId.getRoot();
            auditDetails += ", correlatedAssigningFacility=" + facility.getFacilityNumber();
            if (map != null && !map.isEmpty()) {
                auditDetails += ", ssn=" + map.get("ssn");
                auditDetails += ", lastName=" + map.get("nameFamily");
                auditDetails += ", firstName=" + map.get("nameGiven");
                auditDetails += ", middleName=" + map.get("nameMiddle");
                auditDetails += ", gender=" + map.get("gender");
                auditDetails += ", dob=" + map.get("dob");

                // Get MMN from result patient if empty in source patient
                String mmn = (String)map.get("mothersMaidenName");
                if (NullChecker.isNullOrEmpty(mmn)) {
                    for (Personname name : searchResultPatient.getPersonnames()) {
                        if (name.getPersonnameId().equals(HL7Parser201305.MOTHERS_MAIDEN_NAME_ID)) {
                            mmn = name.getLastName();
                            break;
                        }
                    }
                }
                auditDetails += ", mothersMaidenName=" + mmn;

                auditDetails += ", street1=" + map.get("homeAddressStreet");
                auditDetails += ", city=" + map.get("homeAddressCity");
                auditDetails += ", state=" + map.get("homeAddressState");
                auditDetails += ", zip=" + map.get("homeAddressPostal");
                auditDetails += ", phoneNumber=" + map.get("homePhone");

                // Get POB city and state from result patient if empty in source patient
                String pobCity = (String)map.get("pobCity");
                String pobState = (String)map.get("pobState");
                if (NullChecker.isNullOrEmpty(pobCity) || NullChecker.isNullOrEmpty(pobState)) {
                    for (Address address : searchResultPatient.getAddresses()) {
                        if (address.getAddressId().equals(HL7Parser201305.POB_ADDRESS_ID)) {
                            pobCity = address.getCity();
                            pobState = address.getState();
                            break;
                        }
                    }
                }
                auditDetails += ", pobCity=" + pobCity;
                auditDetails += ", pobState=" + pobState;
            }

            if (!correlationAlreadyExists) {
                DataQuery dataQuery = dataManager.getQuery("MVI.linkPatient");
                dataQuery.setParameter("assertion", assertion);
                dataQuery.setParameter("patientId", localId.getExtension());
                dataQuery.setParameter("correlatedPatientId", remoteId.getExtension());
                dataQuery.setParameter("correlatedAssigningAuthority", remoteId.getRoot());
                dataQuery.setParameter("sendingFacilityOID", facility.getHomeCommunityId());
                dataQuery.setParameter("sendingFacilityNumber", facility.getFacilityNumber());
                if (map != null && !map.isEmpty()) {
                    dataQuery.setParameter("familyName", map.get("nameFamily"));
                    dataQuery.setParameter("firstName", map.get("nameGiven"));
                    dataQuery.setParameter("middleName", map.get("nameMiddle"));
                    dataQuery.setParameter("gender", map.get("gender"));
                    dataQuery.setParameter("dob", map.get("dob"));
                    dataQuery.setParameter("mothersMaidenName", map.get("mothersMaidenName"));
                    dataQuery.setParameter("homeAddressStreet1", map.get("homeAddressStreet"));
                    dataQuery.setParameter("homeAddressCity", map.get("homeAddressCity"));
                    dataQuery.setParameter("homeAddressState", map.get("homeAddressState"));
                    dataQuery.setParameter("homeAddressPostal", map.get("homeAddressPostal"));
                    dataQuery.setParameter("homePhone", map.get("homePhone"));
                    dataQuery.setParameter("pobCity", map.get("pobCity"));
                    dataQuery.setParameter("pobState", map.get("pobState"));
                    dataQuery.setParameter("ssn", map.get("ssn"));
                }

                if (facility != null && facility.getFacilityNumber() != null && !facility.getFacilityNumber().trim().isEmpty()) {

                    dataQuery.setParameter("correlatedAssigningFacility", facility.getFacilityNumber());
                }
                else {
                    throw new RuntimeException("Could not find facility matching the Full Home Commnity ID: '" + senderId.getRoot() + "'");
                }

                // CCR 177986
                logger.debug("query results for addPatientCorrelation Length: {} ", dataQuery.getResults().size());

                dataQuery.getResults();
            }

            facilityManager.addAssigningAuthorityToFacility(senderId.getRoot(), remoteId.getRoot(), remoteId.getAssigningAuthorityName());

            processAuditLogging(searchResultPatient, assertion, icnValue, senderId, auditDetails);
        }
        else {
            String senderIdRoot = (senderId != null) ? senderId.getRoot() : null;
            String localIdExtension = (localId != null) ? localId.getExtension() : null;
            String remoteIdExtension = (remoteId != null) ? remoteId.getExtension() : null;
            String remoteIdRoot = (remoteId != null) ? remoteId.getRoot() : null;

            // CCR 177986
            logger.error("Could not process addPatientCorrelation due to missing parameters: " + senderId.getRoot() + " {} " + localId.getExtension() + " {} " + remoteId.getExtension() + " {} " + remoteId.getRoot() + " {}", new Object[]{senderIdRoot, localIdExtension, remoteIdExtension, remoteIdRoot});
        }

        result.setMCCIIN000002UV01(AckBuilder.BuildAck(IN201301));

        return result;
    }

    private Map getPatientMap(PRPAMT201301UV02Patient patient)
    {
        Map<String, String> map = new HashMap();

        PRPAMT201301UV02Person person = PRPAIN201301UVParser.ParseHL7PatientPersonFromHL7Patient(patient);

        // CCR 177986
        logger.debug("201301UV02 Person {} ", person);

        if (person != null) {
            map = HL7Parser201301.extractPatientLegalName(person);

            String mmn = HL7Parser201301.extractMothersMaidenName(person);
            if (mmn != null) {
                map.put("mothersMaidenName", mmn);
            }

            CE administrativeGenderCode = person.getAdministrativeGenderCode();
            if (administrativeGenderCode != null) {
                map.put("gender", administrativeGenderCode.getCode());
            }

            TSExplicit birthTime = person.getBirthTime();
            if (birthTime != null) {
                map.put("dob", birthTime.getValue());
            }

            if (NullChecker.isNotNullOrEmpty(person.getAddr())) {
                map.putAll(HL7Parser201306.extractHomeAddressMap(person.getAddr().get(0).getContent()));
            }

            if (person.getBirthPlace() != null && person.getBirthPlace().getValue() != null && person.getBirthPlace().getValue().getAddr() != null && NullChecker.isNotNullOrEmpty(person.getBirthPlace().getValue().getAddr().getContent()) && person.getBirthPlace().getValue().getAddr().getContent().get(0) != null) {
                ADExplicit ade = (ADExplicit) person.getBirthPlace().getValue().getAddr().getContent().get(0);
                map.putAll(HL7Parser201306.extractPlaceOfBirth(ade.getContent()));
            }

            if (NullChecker.isNotNullOrEmpty(person.getTelecom())) {
                TELExplicit telecom = person.getTelecom().get(0);
                String str = telecom.getValue();
                if (!NullChecker.isNullOrEmpty(str) && str.startsWith("tel:+")) {
                    str = str.substring(5);
                }
                map.put("homePhone", str);
            }

            if (NullChecker.isNotNullOrEmpty(person.getAsOtherIDs())) {
                for (PRPAMT201301UV02OtherIDs asOtherID : person.getAsOtherIDs()) {
                    for (II id : asOtherID.getId()) {
                        if (id.getRoot().equals("2.16.840.1.113883.4.1")) {
                            String ssn = id.getExtension();
                            map.put("ssn", ssn);
                        }
                    }
                }
            }
        }

        return map;
    }

    private void processAuditLogging(Patient searchResultPatient, AssertionType assertion, String localIdValue, II senderId, String auditDetails)
    {
        String organization = null;

        Audit audit = new Audit();

        // Change action to exclude correlation if sender is SSA
        if (senderId.getRoot().contains(SSA_ORGANIZATION_ID)) {
            audit.setAction("NoCorrelationPatientDiscovery");
        }
        else {
            audit.setAction("AddPatientCorrelation");
        }

        audit.setPatientId(localIdValue);
        if (searchResultPatient != null) {
            audit.setPatientSSN(searchResultPatient.getSsn());

            // set Patient's name
            for (Personname name : searchResultPatient.getPersonnames()) {
                if (name.getPersonnameId().equals(HL7Parser201305.PATIENT_NAME_ID)) {
                    audit.setPatientLastName(name.getLastName());
                    audit.setPatientGivenName(name.getFirstName());
                    audit.setPatientMiddleName(name.getMiddleName());
                    break;
                }
            }

            // set Mothers Maiden Name
            for (Personname name : searchResultPatient.getPersonnames()) {
                if (name.getPersonnameId().equals(HL7Parser201305.MOTHERS_MAIDEN_NAME_ID)) {
                    audit.setMothersMaidenName(name.getLastName());
                    break;
                }
            }

            // Place of Birth (City & State)
            for (Address address : searchResultPatient.getAddresses()) {
                if (address.getAddressId().equals(HL7Parser201305.POB_ADDRESS_ID)) {
                    audit.setPobCity(address.getCity());
                    audit.setPobState(address.getState());
                    break;
                }
            }

            Identifier patientFacilityId = getAndRemovePatientFacilityIdFromPatient(searchResultPatient);
            if (patientFacilityId != null) {
                String[] parts = patientFacilityId.getId().split("~", 2);
                audit.setPatientFacilityNumber(parts[0]);
                audit.setPatientFacilityName(parts[1]);
            }
        }

        if (assertion != null) {
            audit.setUserId(assertion.getUserInfo().getUserName());
            
            audit.setSystemId(AuditUtil.checkSystemId(assertion));
            
            audit.setPurposeForUse(assertion.getPurposeOfDisclosureCoded().getCode());
            audit.setUserRole(assertion.getUserInfo().getRoleCoded().getCode());
            audit.setUserFacilityNumber(assertion.getUserInfo().getOrg().getHomeCommunityId());
            audit.setUserFacilityName(assertion.getUserInfo().getOrg().getName());
            if (assertion.getHomeCommunity() != null && !NullChecker.isNullOrEmpty(assertion.getHomeCommunity().getHomeCommunityId())) {
                organization = assertion.getHomeCommunity().getHomeCommunityId();
            }
            if (assertion.getUserInfo().getPersonName() != null) {
                PersonNameType personName = assertion.getUserInfo().getPersonName();
                if (!NullChecker.isNullOrEmpty(personName.getFullName())) {
                    audit.setUserName(personName.getFullName());
                }
                else {
                    StringBuilder userName = new StringBuilder();

                    if (NullChecker.isNotNullOrEmpty(personName.getGivenName())) {
                        userName.append(personName.getGivenName());
                    }

                    if (NullChecker.isNotNullOrEmpty(personName.getFamilyName())) {
                        if (userName.length() > 0) {
                            userName.append(' ');
                        }

                        userName.append(personName.getFamilyName());
                    }

                    audit.setUserName(userName.toString());
                }
            }
        }

        Facility homeFacility = facilityManager.getFacilityByFacilityNumber("VA");
        if (organization == null || organization.isEmpty() || organization.equals(homeFacility.getFullHomeCommunityId())) {
            organization = homeFacility.getFullHomeCommunityId();
            audit.setRemoteOrganizationId("urn:oid:" + senderId.getRoot());
        }
        else {
            audit.setRemoteOrganizationId(facilityManager.getFacilityByFacilityNumber("VA").getFullHomeCommunityId());
        }

        audit.setOrganizationId(organization);
        audit.setDetails(auditDetails);

        auditManager.storeAudit(audit);
        // PD Audit Report - End
    }

    private Identifier getAndRemovePatientFacilityIdFromPatient(Patient patient)
    {
        Identifier ret = null;
        String assigningAuthority = "PATIENT_FACILITY";
        int index = 0;

        for (Identifier id : patient.getIdentifiers()) {
            if (id.getOrganizationId().equals(assigningAuthority)) {
                ret = id;
                break;
            }
            ++index;
        }

        if (ret != null) {
            patient.getIdentifiers().remove(index);
        }

        // CCR 177986
        logger.info("Id removed {}", ret);

        return ret;
    }

    private II getSenderId(PRPAIN201301UV02 message)
    {
        II ret = null;

        if (message.getSender() != null && message.getSender().getDevice() != null && message.getSender().getDevice().getAsAgent() != null && message.getSender().getDevice().getAsAgent().getValue() != null && message.getSender().getDevice().getAsAgent().getValue().getRepresentedOrganization() != null && message.getSender().getDevice().getAsAgent().getValue().getRepresentedOrganization().getValue() != null && !NullChecker.isNullOrEmpty(message.getSender().getDevice().getAsAgent().getValue().getRepresentedOrganization().getValue().getId())) {
            ret = message.getSender().getDevice().getAsAgent().getValue().getRepresentedOrganization().getValue().getId().get(0);

            // CCR 177986
            logger.debug("assign auth name for Sender Id when message has no sender {} ", ret.getAssigningAuthorityName());

        }
        else if (message.getSender() != null && message.getSender().getDevice() != null && !NullChecker.isNullOrEmpty(message.getSender().getDevice().getId())) {
            ret = message.getSender().getDevice().getId().get(0);

            // CCR 177986
            logger.debug("assign auth name for Sender Id when message has sender info {} ", ret.getAssigningAuthorityName());
        }

        return ret;
    }

    private II getLocalId(PRPAMT201301UV02Patient patient)
    {
        II icn = null;

        icn = HL7Parser201306.extractICNId(patient.getId());
        if (icn == null) {
            String assigningAuthority = propertyLookup.getProperty("AssigningAuthority");
            for (II id : patient.getId()) {
                if (id.getRoot().equals(assigningAuthority) && HL7Parser201306.isValidICNValue(id.getExtension())) {
                    icn = id;
                    break;
                }
            }
        }

        // CCR 177986
        logger.debug("icn as local id {} ", icn);

        return icn;
    }

    private II getRemoteId(PRPAMT201301UV02Patient patient, String remoteAAID)
    {
        II ret = null;
        // for addPatientCorrelation calls coming from
        // AdapterGatewayPatientDiscovery (i.e. outbound PD requests) we
        // are not getting the remote Assigning Authority since our usage of the
        // HL7PRPA201301Transforms doesn't set it.
        // So for that scenario we look for the ID that's not a VA issued.
        // Therefore it must be the remote/partner's patient id...
        // assuming the only other one in ID list is the ICN (VA issued).
        if (NullChecker.isNullOrEmpty(remoteAAID)) {
            String vaAAID = propertyLookup.getProperty("AssigningAuthority");
            for (II id : patient.getId()) {
                if (!id.getRoot().equals(vaAAID)) {
                    ret = id;

                    // CCR 177986
                    logger.debug("remote partner's patient id {}", ret);

                    break;
                }
            }
        }
        else { // since we have the remote Assigning Authority, let's find remote
            // patient ID in more explicit way.
            for (II id : patient.getId()) {
                if (id.getRoot().equals(remoteAAID)) {
                    ret = id;

                    // CCR 177986
                    logger.debug("remote id from assigning auth {} ", ret);

                    break;
                }
            }
        }

        return ret;
    }

    private List<QualifiedPatientIdentifier> findCorrelations(QualifiedPatientIdentifier patientIdentifier, List<String> dataSourceList, AssertionType assertion)
    {
        List<QualifiedPatientIdentifier> resultQualifiedPatientIdentifiers = new ArrayList<QualifiedPatientIdentifier>();

        String fullHCID = assertion.getHomeCommunity().getHomeCommunityId();
        if (fullHCID == null || !fullHCID.startsWith("urn:oid:")) {
            throw new AdapterPatientCorrelationException("findCorrelations: missing or invalid full HCID: '" + fullHCID);
        }

        Facility facility = facilityManager.getFacilityByFullHomeCommunityId(fullHCID);
        if (facility == null || NullChecker.isNullOrEmpty(facility.getHomeCommunityId())) {
            throw new AdapterPatientCorrelationException("findCorrelations: could not find facility using full HCID: ' " + fullHCID + "'");
        }

        DataQuery dp = dataManager.getQuery("MVI.findNHINCorrelations");
        dp.setParameter("icn", patientIdentifier.getPatientId());
        dp.setParameter("assertion", assertion);
        dp.setParameter("sendingFacilityOID", facility.getHomeCommunityId());
        dp.setParameter("sendingFacilityNumber", facility.getFacilityNumber());

        List<PRPAIN201310UV02> dpResults = dp.getResults();
        PRPAIN201310UV02 result1310 = null;
        List<Map> correlatedIDsMap = null;

        if (dpResults != null && dpResults.size() == 1 && dpResults.get(0) != null) {
            result1310 = dpResults.get(0);

            logger.debug("result 1310 {} ", result1310);
        }
        else {
            return null;
        }

        MCCIMT000300UV01Acknowledgement ack = result1310.getAcknowledgement().get(0);
        MFMIMT700711UV01QueryAck queryAck = result1310.getControlActProcess().getQueryAck();

        if (ack.getTypeCode().getCode().equalsIgnoreCase("AE")) {
            // MVI returned with failure code.
            logger.error("MVI 1309 request returned acknowlegment code of 'AE' for application error for ICN  {} ", patientIdentifier.getPatientId());
            return null;
        }
        else if (queryAck.getQueryResponseCode().getCode().equalsIgnoreCase("NF")) {
            logger.error("MVI 1309 request returned query acknowlegment code of 'NF' for not found for ICN ' {} ", patientIdentifier.getPatientId());
            return null;
        }
        else if (queryAck.getQueryResponseCode().getCode().equalsIgnoreCase("OK")) {
            logger.debug("MVI 1309 request returned successful response for ICN  {} ", patientIdentifier.getPatientId());
            correlatedIDsMap = map1310toResult(result1310);
        }
        else {
            throw new AdapterPatientCorrelationException("MVI response does not map to valid acknowledgement for ICN '" + patientIdentifier.getPatientId() + "'");
        }

        for (Map correlatedIDMap : correlatedIDsMap) {
            String correlatedPatientId = (String) correlatedIDMap.get("correlatedPatientId");
            String correlatedAssigningAuthority = (String) correlatedIDMap.get("correlatedAssigningAuthority");
            String correlatedAssigningFacility = (String) correlatedIDMap.get("correlatedAssigningFacility");

            if (NullChecker.isNullOrEmpty(dataSourceList) || dataSourceList.contains(correlatedAssigningFacility)) {
                QualifiedPatientIdentifier qpi = new QualifiedPatientIdentifier();
                qpi.setAssigningAuthority(correlatedAssigningAuthority);
                qpi.setPatientId(correlatedPatientId);
                resultQualifiedPatientIdentifiers.add(qpi);
            }
        }

        return resultQualifiedPatientIdentifiers;
    }

    /*
	 * @param PRPAIN201310UV02
	 *
	 * Extract correlated patient information from the 1310 response and build
	 * result List<Map> to be returned to DataQueryImpl.getResults() method
     */
    private ArrayList<Map> map1310toResult(PRPAIN201310UV02 response1310)
    {
        ArrayList<Map> result = new ArrayList<Map>();
        HashMap correlatedInfo = null;

        // retrieve patient IDs
        List<II> patientIds = response1310.getControlActProcess().getSubject().get(0).getRegistrationEvent().getSubject1().getPatient().getId();

        // CCR 177986
        logger.debug("patient IDs from 1310 response {} ", patientIds);

        String patientExtension;
        String[] splittedExtension;

        for (II patientId : patientIds) {

            patientExtension = patientId.getExtension();
            splittedExtension = patientExtension.split("\\^", -1);

            if (splittedExtension != null && splittedExtension.length >= 4) {
                correlatedInfo = new HashMap(3);
                correlatedInfo.put("correlatedPatientId", splittedExtension[0]);
                correlatedInfo.put("correlatedAssigningFacility", splittedExtension[2]);
                correlatedInfo.put("correlatedAssigningAuthority", splittedExtension[3]);
                result.add(correlatedInfo);
            }
        }

        logger.debug("correlated patient Ids {} ", result.toArray());

        return result;
    }

    private static List<II> buildList(List<QualifiedPatientIdentifier> qualifiedPatientIdentifiers)
    {
        if (qualifiedPatientIdentifiers == null) {
            return null;
        }
        List<II> iiList = new ArrayList<II>();

        for (QualifiedPatientIdentifier qualifiedPatientIdentifier : qualifiedPatientIdentifiers) {
            iiList.add(IIFactory(qualifiedPatientIdentifier));
        }

        // CCR 177986
        logger.debug("ii list {} ", iiList);

        return iiList;
    }

    private static II IIFactory(QualifiedPatientIdentifier qualifiedPatientIdentifier)
    {
        II ii = new II();
        ii.setRoot(qualifiedPatientIdentifier.getAssigningAuthorityId());
        ii.setExtension(qualifiedPatientIdentifier.getPatientId());

        // CCR 177986
        logger.debug("II {} ", ii);

        return ii;
    }

    private static QualifiedPatientIdentifier QualifiedPatientIdentifierFactory(II ii)
    {
        QualifiedPatientIdentifier qualifiedPatientIdentifier = new QualifiedPatientIdentifier();
        qualifiedPatientIdentifier.setAssigningAuthority(ii.getRoot());
        qualifiedPatientIdentifier.setPatientId(ii.getExtension());

        // CCR 177986
        logger.debug("qualified Patient Identifier {} ", qualifiedPatientIdentifier);

        return qualifiedPatientIdentifier;
    }

    private String getSenderOID(PRPAIN201301UV02 request)
    {
        String ret = null;

        if (request.getSender() != null
            && request.getSender().getDevice() != null
            && request.getSender().getDevice().getAsAgent() != null
            && request.getSender().getDevice().getAsAgent().getValue() != null
            && request.getSender().getDevice().getAsAgent().getValue().getRepresentedOrganization() != null
            && request.getSender().getDevice().getAsAgent().getValue().getRepresentedOrganization().getValue() != null
            && request.getSender().getDevice().getAsAgent().getValue().getRepresentedOrganization().getValue().getId() != null
            && NullChecker.isNotNullOrEmpty(request.getSender().getDevice().getAsAgent().getValue().getRepresentedOrganization().getValue().getId())) {
            ret = request.getSender().getDevice().getAsAgent().getValue().getRepresentedOrganization().getValue().getId().get(0).getRoot();

            // CCR 177986
            logger.debug("Sender OID {}", ret);
        }

        return ret;
    }

    private String getReceiverOID(PRPAIN201301UV02 request)
    {
        String ret = null;

        if (request != null
            && !NullChecker.isNullOrEmpty(request.getReceiver())
            && request.getReceiver().get(0) != null
            && request.getReceiver().get(0).getDevice() != null
            && request.getReceiver().get(0).getDevice().getAsAgent() != null
            && request.getReceiver().get(0).getDevice().getAsAgent().getValue() != null
            && request.getReceiver().get(0).getDevice().getAsAgent().getValue().getRepresentedOrganization() != null
            && request.getReceiver().get(0).getDevice().getAsAgent().getValue().getRepresentedOrganization().getValue() != null
            && !NullChecker.isNullOrEmpty(request.getReceiver().get(0).getDevice().getAsAgent().getValue().getRepresentedOrganization().getValue().getId())
            && request.getReceiver().get(0).getDevice().getAsAgent().getValue().getRepresentedOrganization().getValue().getId().get(0) != null
            && !NullChecker.isNullOrEmpty(request.getReceiver().get(0).getDevice().getAsAgent().getValue().getRepresentedOrganization().getValue().getId().get(0).getRoot())) {
            ret = request.getReceiver().get(0).getDevice().getAsAgent().getValue().getRepresentedOrganization().getValue().getId().get(0).getRoot();
        }

        return ret;
    }
}
