/*
 * DO NOT ALTER OR REMOVE COPYRIGHT NOTICES OR THIS HEADER.
 *
 * Copyright 2011(Year date of delivery) United States Government, as represented by the Secretary of Health and Human Services.  All rights reserved.
 *
 */
package gov.va.med.nhin.adapter.mpi.hl7parsers;

import java.io.Serializable;
import java.math.BigInteger;
import java.text.SimpleDateFormat;
import java.util.ArrayList;
import java.util.Date;
import java.util.List;
import java.util.TimeZone;

import javax.xml.bind.JAXBElement;

import org.hl7.v3.ADExplicit;
import org.hl7.v3.ActClassControlAct;
import org.hl7.v3.AdxpExplicitCity;
import org.hl7.v3.AdxpExplicitPostalCode;
import org.hl7.v3.AdxpExplicitState;
import org.hl7.v3.AdxpExplicitStreetAddressLine;
import org.hl7.v3.CD;
import org.hl7.v3.CE;
import org.hl7.v3.COCTMT090003UV01AssignedEntity;
import org.hl7.v3.COCTMT090300UV01AssignedDevice;
import org.hl7.v3.COCTMT150002UV01Organization;
import org.hl7.v3.COCTMT150003UV03ContactParty;
import org.hl7.v3.COCTMT150003UV03Organization;
import org.hl7.v3.CS;
import org.hl7.v3.CommunicationFunctionType;
import org.hl7.v3.ENExplicit;
import org.hl7.v3.EntityClassDevice;
import org.hl7.v3.II;
import org.hl7.v3.INT;
import org.hl7.v3.MCCIMT000100UV01Receiver;
import org.hl7.v3.MCCIMT000100UV01Sender;
import org.hl7.v3.MCCIMT000300UV01Acknowledgement;
import org.hl7.v3.MCCIMT000300UV01Agent;
import org.hl7.v3.MCCIMT000300UV01Device;
import org.hl7.v3.MCCIMT000300UV01Organization;
import org.hl7.v3.MCCIMT000300UV01Receiver;
import org.hl7.v3.MCCIMT000300UV01Sender;
import org.hl7.v3.MFMIMT700711UV01AuthorOrPerformer;
import org.hl7.v3.MFMIMT700711UV01Custodian;
import org.hl7.v3.MFMIMT700711UV01QueryAck;
import org.hl7.v3.PNExplicit;
import org.hl7.v3.PRPAIN201305UV02;
import org.hl7.v3.PRPAIN201306UV02;
import org.hl7.v3.PRPAIN201306UV02MFMIMT700711UV01ControlActProcess;
import org.hl7.v3.PRPAIN201306UV02MFMIMT700711UV01RegistrationEvent;
import org.hl7.v3.PRPAIN201306UV02MFMIMT700711UV01Subject1;
import org.hl7.v3.PRPAIN201306UV02MFMIMT700711UV01Subject2;
import org.hl7.v3.PRPAMT201306UV02LivingSubjectName;
import org.hl7.v3.PRPAMT201306UV02ParameterList;
import org.hl7.v3.PRPAMT201306UV02QueryByParameter;
import org.hl7.v3.PRPAMT201310UV02OtherIDs;
import org.hl7.v3.PRPAMT201310UV02Patient;
import org.hl7.v3.PRPAMT201310UV02Person;
import org.hl7.v3.PRPAMT201310UV02QueryMatchObservation;
import org.hl7.v3.PRPAMT201310UV02Subject;
import org.hl7.v3.ParticipationTargetSubject;
import org.hl7.v3.RoleClassContact;
import org.hl7.v3.TELExplicit;
import org.hl7.v3.TSExplicit;
import org.hl7.v3.XActMoodIntentEvent;
import org.hl7.v3.XParticipationAuthorPerformer;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import gov.hhs.fha.nhinc.mpi.adapter.component.hl7parsers.MessageIdGenerator;
import gov.hhs.fha.nhinc.nhinclib.NullChecker;
import gov.hhs.fha.nhinc.patientdb.model.Address;
import gov.hhs.fha.nhinc.patientdb.model.Patient;
import gov.hhs.fha.nhinc.patientdb.model.Personname;
import gov.hhs.fha.nhinc.patientdb.model.Phonenumber;
import gov.hhs.fha.nhinc.transform.subdisc.HL7Constants;
import gov.hhs.fha.nhinc.transform.subdisc.HL7DataTransformHelper;
import gov.hhs.fha.nhinc.util.format.UTCDateUtil;

/**
 *
 * @author richard.ettema
 */
public class HL7DbParser201306
{
	// CCR 177986-logging framework updates
	private static final Logger logger = LoggerFactory.getLogger(HL7DbParser201306.class.getName());

	public static PRPAIN201306UV02 BuildMessageFromMpiPatients(List<Patient> patients, PRPAIN201305UV02 query, String homeCommunityId, String assigningAuthority)
	{
		logger.debug("Entering HL7Parser201306.BuildMessageFromMpiPatients method...");

		PRPAIN201306UV02 msg = new PRPAIN201306UV02();

		// Set up message header fields
		msg.setITSVersion("XML_1.0");

		II id = new II();
		id.setRoot(assigningAuthority);
		id.setExtension(MessageIdGenerator.generateMessageId());
		msg.setId(id);

		// Set up the creation time string
		String timestamp = "";
		try
		{
			Date now = new Date();
			SimpleDateFormat formatter = new SimpleDateFormat("yyyyMMddHHmmss");
			formatter.setTimeZone(TimeZone.getTimeZone("UTC"));
			timestamp = formatter.format(now);
		}
		catch(Exception e)
		{
			// CCR 177986
			logger.error("Exception occurred when creating XMLGregorian Date", e);
		}

		TSExplicit creationTime = new TSExplicit();
		creationTime.setValue(timestamp);
		msg.setCreationTime(creationTime);

		II interactionId = new II();
		interactionId.setRoot("2.16.840.1.113883.1.6");
		interactionId.setExtension("PRPA_IN201306UV02");
		msg.setInteractionId(interactionId);

		CS processingCode = new CS();
		processingCode.setCode("P");
		msg.setProcessingCode(processingCode);

		CS processingModeCode = new CS();
		processingModeCode.setCode("T");
		msg.setProcessingModeCode(processingModeCode);

		CS ackCode = new CS();
		ackCode.setCode("NE");
		msg.setAcceptAckCode(ackCode);

		msg.getAcknowledgement().add(createAck(query));

		// Set the receiver and sender
		msg.getReceiver().add(createReceiver(query.getSender()));
		msg.setSender(createSender(query.getReceiver().get(0)));

		msg.setControlActProcess(createControlActProcess(patients, query, homeCommunityId, assigningAuthority));

		logger.debug("Exiting HL7Parser201306.BuildMessageFromMpiPatient method...");

		return msg;
	}

	private static PRPAIN201306UV02MFMIMT700711UV01ControlActProcess createControlActProcess(List<Patient> patients, PRPAIN201305UV02 query, String homeCommunityId, String assigningAuthority)
	{
		PRPAIN201306UV02MFMIMT700711UV01ControlActProcess controlActProcess = new PRPAIN201306UV02MFMIMT700711UV01ControlActProcess();

		controlActProcess.setMoodCode(XActMoodIntentEvent.EVN);
		controlActProcess.setClassCode(ActClassControlAct.CACT);
		CD code = new CD();
		code.setCode("PRPA_TE201306UV02");
		code.setCodeSystem("2.16.840.1.113883.1.6");
		controlActProcess.setCode(code);
		controlActProcess.getAuthorOrPerformer().add(createAuthorOrPerformer(assigningAuthority));

		boolean hasPatients = false;

		// Add a subject for each value unique patient
		if(patients != null && patients.size() > 0)
		{
			hasPatients = true;
			for(Patient patient : patients)
			{
				controlActProcess.getSubject().add(createSubject(patient, homeCommunityId));

				// CCR 177986- only if debugger enabled
				logger.debug("Patient Subject from Control Act Process {} ", controlActProcess.getSubject().add(createSubject(patient, homeCommunityId)));
			}
		}
		else
		{
			// CCR 177986
			logger.debug("createControlActProcess - No patients found to create subject");
		}

		// Add in query parameters
		if(query.getControlActProcess() != null && query.getControlActProcess().getQueryByParameter() != null && query.getControlActProcess().getQueryByParameter().getValue() != null)
		{
			controlActProcess.setQueryByParameter(createQueryByParameter(query.getControlActProcess().getQueryByParameter().getValue()));
		}

		controlActProcess.setQueryAck(createQueryAck(query, hasPatients));

		return controlActProcess;
	}

	// ========== TO BE REMOVED ==========
	// CAN'T BE REMOVED JUST YET!!! CONNECT 2.4.x uses this to create AA to HCID
	// mappings.
	static private MFMIMT700711UV01AuthorOrPerformer createAuthorOrPerformer(String assigningAuthority)
	{
		MFMIMT700711UV01AuthorOrPerformer authorOrPerformer = new MFMIMT700711UV01AuthorOrPerformer();
		authorOrPerformer.setTypeCode(XParticipationAuthorPerformer.AUT);

		COCTMT090300UV01AssignedDevice assignedDevice = new COCTMT090300UV01AssignedDevice();
		II id = new II();
		id.setRoot(assigningAuthority);
		assignedDevice.setClassCode(HL7Constants.ASSIGNED_DEVICE_CLASS_CODE);
		assignedDevice.getId().add(id);

		javax.xml.namespace.QName xmlqname = new javax.xml.namespace.QName("urn:hl7-org:v3", "assignedDevice");
		JAXBElement<COCTMT090300UV01AssignedDevice> assignedDeviceJAXBElement = new JAXBElement<COCTMT090300UV01AssignedDevice>(xmlqname, COCTMT090300UV01AssignedDevice.class, assignedDevice);

		authorOrPerformer.setAssignedDevice(assignedDeviceJAXBElement);

		return authorOrPerformer;
	}
	// ========== END TO BE REMOVED ==========

	private static MFMIMT700711UV01QueryAck createQueryAck(PRPAIN201305UV02 query, boolean hasPatients)
	{
		MFMIMT700711UV01QueryAck result = new MFMIMT700711UV01QueryAck();

		if(query.getControlActProcess() != null && query.getControlActProcess().getQueryByParameter() != null && query.getControlActProcess().getQueryByParameter().getValue() != null && query.getControlActProcess().getQueryByParameter().getValue().getQueryId() != null)
		{
			result.setQueryId(query.getControlActProcess().getQueryByParameter().getValue().getQueryId());
		}

		CS respCode = new CS();
		if(hasPatients)
		{
			respCode.setCode("OK");
		}
		else
		{
			respCode.setCode("NF"); // Not Found
		}
		result.setQueryResponseCode(respCode);

		// INT totalQuanity = new INT();
		// totalQuanity.setValue(BigInteger.valueOf(1));
		// result.setResultTotalQuantity(totalQuanity);

		// INT currQuanity = new INT();
		// currQuanity.setValue(BigInteger.valueOf(1));
		// result.setResultCurrentQuantity(currQuanity);

		// INT remainQuanity = new INT();
		// remainQuanity.setValue(BigInteger.valueOf(0));
		// result.setResultRemainingQuantity(remainQuanity);

		return result;
	}

	private static PRPAIN201306UV02MFMIMT700711UV01Subject1 createSubject(Patient patient, String homeCommunityId)
	{
		PRPAIN201306UV02MFMIMT700711UV01Subject1 subject = new PRPAIN201306UV02MFMIMT700711UV01Subject1();

		subject.getTypeCode().add("SUBJ");

		subject.setRegistrationEvent(createRegEvent(patient, homeCommunityId));

		return subject;
	}

	private static PRPAIN201306UV02MFMIMT700711UV01RegistrationEvent createRegEvent(Patient patient, String homeCommunityId)
	{
		PRPAIN201306UV02MFMIMT700711UV01RegistrationEvent regEvent = new PRPAIN201306UV02MFMIMT700711UV01RegistrationEvent();
		regEvent.getMoodCode().add("EVN");
		regEvent.getClassCode().add("REG");
		II id = new II();
		id.getNullFlavor().add("NA");
		regEvent.getId().add(id);

		CS statusCode = new CS();
		statusCode.setCode("active");

		regEvent.setStatusCode(statusCode);

		regEvent.setSubject1(createSubject1(patient));

		regEvent.setCustodian(createCustodian(patient, homeCommunityId));

		return regEvent;
	}

	private static MFMIMT700711UV01Custodian createCustodian(Patient patient, String homeCommunityId)
	{
		MFMIMT700711UV01Custodian result = new MFMIMT700711UV01Custodian();
		result.getTypeCode().add("CST");
		result.setAssignedEntity(createAssignEntity(homeCommunityId));

		return result;
	}

	private static COCTMT090003UV01AssignedEntity createAssignEntity(String homeCommunityId)
	{
		COCTMT090003UV01AssignedEntity assignedEntity = new COCTMT090003UV01AssignedEntity();
		assignedEntity.setClassCode(HL7Constants.ASSIGNED_DEVICE_CLASS_CODE);
		II id = new II();
		id.setRoot(homeCommunityId);
		assignedEntity.getId().add(id);
		CE ce = new CE();
		ce.setCode("NotHealthDataLocator");
		ce.setCodeSystem("1.3.6.1.4.1.19376.1.2.27.2");
		assignedEntity.setCode(ce);

		return assignedEntity;
	}

	private static PRPAIN201306UV02MFMIMT700711UV01Subject2 createSubject1(Patient patient)
	{
		PRPAIN201306UV02MFMIMT700711UV01Subject2 subject = new PRPAIN201306UV02MFMIMT700711UV01Subject2();
		subject.setTypeCode(ParticipationTargetSubject.SBJ);
		// Add in patient
		subject.setPatient(createPatient(patient));

		return subject;
	}

	private static PRPAMT201310UV02Patient createPatient(Patient patient)
	{
		PRPAMT201310UV02Patient subjectPatient = new PRPAMT201310UV02Patient();

		subjectPatient.getClassCode().add("PAT");

		// TODO: Temporary value until confirmation can be made on the actual
		// value
		CS statusCode = new CS();
		statusCode.setCode("active");
		subjectPatient.setStatusCode(statusCode);

		// Add in patient id
		subjectPatient.getId().add(createSubjectId(patient));

		// Add in patient person
		subjectPatient.setPatientPerson(createPatientPerson(patient));

		// Add in provider organization
		subjectPatient.setProviderOrganization(createProviderOrg(patient));

		// Add in query match observation
		subjectPatient.getSubjectOf1().add(createSubjectOf1());

		return subjectPatient;
	}

	private static PRPAMT201310UV02Subject createSubjectOf1()
	{
		PRPAMT201310UV02Subject result = new PRPAMT201310UV02Subject();

		result.setQueryMatchObservation(createQueryMatch());
		return result;
	}

	private static PRPAMT201310UV02QueryMatchObservation createQueryMatch()
	{
		PRPAMT201310UV02QueryMatchObservation queryMatch = new PRPAMT201310UV02QueryMatchObservation();
		queryMatch.getMoodCode().add("EVN");
		queryMatch.getClassCode().add("CASE");
		CD code = new CD();
		code.setCode("IHE_PDQ");
		queryMatch.setCode(code);

		INT intValue = new INT();
		intValue.setValue(BigInteger.valueOf(100));
		queryMatch.setValue(intValue);

		return queryMatch;
	}

	private static JAXBElement<COCTMT150003UV03Organization> createProviderOrg(Patient patient)
	{
		COCTMT150003UV03Organization org = new COCTMT150003UV03Organization();
		org.setDeterminerCode("INSTANCE");
		org.setClassCode("ORG");
		II id = new II();

		if(patient.getIdentifiers() != null && patient.getIdentifiers().size() > 0 && patient.getIdentifiers().get(0).getOrganizationId() != null && patient.getIdentifiers().get(0).getOrganizationId().length() > 0)
		{
			id.setRoot(patient.getIdentifiers().get(0).getOrganizationId());
		}
		org.getId().add(id);

		COCTMT150003UV03ContactParty cp = new COCTMT150003UV03ContactParty();
		cp.setClassCode(RoleClassContact.CON);
		org.getContactParty().add(cp);

		javax.xml.namespace.QName xmlqname = new javax.xml.namespace.QName("urn:hl7-org:v3", "providerOrganization");
		JAXBElement<COCTMT150003UV03Organization> result = new JAXBElement<COCTMT150003UV03Organization>(xmlqname, COCTMT150003UV03Organization.class, org);

		return result;
	}

	private static II createSubjectId(Patient patient)
	{
		II id = new II();

		if(patient.getIdentifiers() != null && patient.getIdentifiers().size() > 0 && patient.getIdentifiers().get(0) != null)
		{

			if(patient.getIdentifiers().get(0).getOrganizationId() != null && patient.getIdentifiers().get(0).getOrganizationId().length() > 0)
			{
				logger.debug("Setting Patient Id root in 201306 {}: ", patient.getIdentifiers().get(0).getOrganizationId()); // CCR
																																// 177986
				id.setRoot(patient.getIdentifiers().get(0).getOrganizationId());
			}

			if(patient.getIdentifiers().get(0).getId() != null && patient.getIdentifiers().get(0).getId().length() > 0)
			{
				logger.debug("Setting Patient Id extension in 201306 {}: ", patient.getIdentifiers().get(0).getId()); // CCR
																														// 177986
				id.setExtension(patient.getIdentifiers().get(0).getId());
			}
		}

		return id;
	}

	private static JAXBElement<PRPAMT201310UV02Person> createPatientPerson(Patient patient)
	{
		PRPAMT201310UV02Person person = new PRPAMT201310UV02Person();

		// Set classCode
		person.getClassCode().add("PSN");

		// Set determinerCode
		person.setDeterminerCode("INSTANCE");

		// Set the Subject Gender
		if(patient.getGender() != null && patient.getGender().length() > 0)
		{
			person.setAdministrativeGenderCode(createGender(patient));
		}

		// Set the Subject Name
		if(patient.getPersonnames().size() > 0)
		{
			for(Personname name : patient.getPersonnames())
			{
				person.getName().add(createSubjectName(name));
			}
		}
		else
		{
			person.getName().add(createSubjectName(patient));
		}

		// Set the Birth Time
		if(patient.getDateOfBirth() != null)
		{
			person.setBirthTime(createBirthTime(patient));
		}

		// Set the Addresses
		if(patient.getAddresses().size() > 0)
		{
			for(Address add : patient.getAddresses())
			{
				person.getAddr().add(createAddress(add));
			}
		}

		// Set the phone Numbers
		if(patient.getPhonenumbers().size() > 0)
		{
			for(Phonenumber number : patient.getPhonenumbers())
			{
				TELExplicit tele = HL7DataTransformHelper.createTELExplicit(number.getValue());

				person.getTelecom().add(tele);
			}
		}

		// Set the SSN
		if(patient.getSsn() != null && patient.getSsn().length() > 0)
		{
			person.getAsOtherIDs().add(createOtherIds(patient));
		}

		javax.xml.namespace.QName xmlqname = new javax.xml.namespace.QName("urn:hl7-org:v3", "patientPerson");
		JAXBElement<PRPAMT201310UV02Person> result = new JAXBElement<PRPAMT201310UV02Person>(xmlqname, PRPAMT201310UV02Person.class, person);

		// CCR 177986
		logger.debug("PRPAMT201310UV02Person Patient Person {} ");

		return result;
	}

	private static PRPAMT201310UV02OtherIDs createOtherIds(Patient patient)
	{
		PRPAMT201310UV02OtherIDs otherIds = new PRPAMT201310UV02OtherIDs();

		// TODO: Temporary assignment until actual value can be determined
		otherIds.getClassCode().add("SD");

		// Set the SSN
		if(patient.getSsn() != null && patient.getSsn().length() > 0)
		{
			II ssn = new II();
			ssn.setExtension(patient.getSsn());
			ssn.setRoot("2.16.840.1.113883.4.1");
			otherIds.getId().add(ssn);

			COCTMT150002UV01Organization scopingOrg = new COCTMT150002UV01Organization();
			scopingOrg.setClassCode("ORG");
			scopingOrg.setDeterminerCode("INSTANCE");
			II orgId = new II();
			orgId.setRoot(ssn.getRoot());
			scopingOrg.getId().add(orgId);
			otherIds.setScopingOrganization(scopingOrg);
		}

		// CCR 177986
		logger.debug("PRPAMT201310UV02OtherIDs otherIds {} ", otherIds);

		return otherIds;
	}

	private static TSExplicit createBirthTime(Patient patient)
	{
		TSExplicit birthTime = new TSExplicit();

		if(patient.getDateOfBirth() != null)
		{
			logger.debug("Setting Patient Birthday in 201306: " + patient.getDateOfBirth());

			UTCDateUtil utcDateUtil = new UTCDateUtil();
			// Format for date only, no time portion
			birthTime.setValue(utcDateUtil.formatUTCDateOnly(patient.getDateOfBirth()));
		}

		// CCR 177986
		logger.debug("TSExplicit birthTime {} ", birthTime);

		return birthTime;
	}

	private static PNExplicit createSubjectName(Patient patient)
	{
		// CCR 177986
		logger.debug("subject name from patient info {} ", createSubjectName(patient.getPersonnames().get(0)));

		return createSubjectName(patient.getPersonnames().get(0));
	}

	private static PNExplicit createSubjectName(Personname personname)
	{
		org.hl7.v3.ObjectFactory factory = new org.hl7.v3.ObjectFactory();
		PNExplicit name = (PNExplicit) (factory.createPNExplicit());

		String lastName = personname.getLastName();
		String firstName = personname.getFirstName();
		String middleName = personname.getMiddleName();
		String prefix = personname.getPrefix();
		String suffix = personname.getSuffix();

		name = HL7DataTransformHelper.createPNExplicit(firstName, middleName, lastName, prefix, suffix);

		// CCR 177986
		logger.debug("subject name from person info {} ", name);

		return name;
	}

	private static CE createGender(Patient patient)
	{
		CE gender = new CE();

		if(patient.getGender() != null && patient.getGender().length() > 0)
		{
			logger.debug("Setting Patient Gender in 201306 {}: ", patient.getGender());

			gender.setCode(patient.getGender());
		}
		return gender;
	}

	private static MCCIMT000300UV01Acknowledgement createAck(PRPAIN201305UV02 query)
	{
		MCCIMT000300UV01Acknowledgement ack = new MCCIMT000300UV01Acknowledgement();
		ack.setTypeId(query.getInteractionId());

		CS typeCode = new CS();
		typeCode.setCode("AA");

		ack.setTypeCode(typeCode);

		// CCR 177986
		logger.debug("ack code {} ", ack);

		return ack;
	}

	private static MCCIMT000300UV01Receiver createReceiver(MCCIMT000100UV01Sender querySender)
	{
		MCCIMT000300UV01Receiver receiver = new MCCIMT000300UV01Receiver();
		String app = null;
		String oid = null;

		receiver.setTypeCode(CommunicationFunctionType.RCV);

		if(querySender.getDevice() != null && NullChecker.isNotNullish(querySender.getDevice().getId()) && querySender.getDevice().getId().get(0) != null && NullChecker.isNotNullish(querySender.getDevice().getId().get(0).getRoot()))
		{
			app = querySender.getDevice().getId().get(0).getRoot();

			// CCR 177986
			logger.info("APP is {} ", app);
		}

		if(querySender.getDevice() != null && querySender.getDevice().getAsAgent() != null && querySender.getDevice().getAsAgent().getValue() != null && querySender.getDevice().getAsAgent().getValue().getRepresentedOrganization() != null && querySender.getDevice().getAsAgent().getValue().getRepresentedOrganization().getValue() != null && NullChecker.isNotNullish(querySender.getDevice().getAsAgent().getValue().getRepresentedOrganization().getValue().getId()) && querySender.getDevice().getAsAgent().getValue().getRepresentedOrganization().getValue().getId().get(0) != null && NullChecker.isNotNullish(querySender.getDevice().getAsAgent().getValue().getRepresentedOrganization().getValue().getId().get(0).getRoot()))
		{
			oid = querySender.getDevice().getAsAgent().getValue().getRepresentedOrganization().getValue().getId().get(0).getRoot();

			// CCR 177986
			logger.info("OID is {} ", oid);
		}

		if(NullChecker.isNullish(oid))
		{
			oid = app;
		}

		MCCIMT000300UV01Device receiverDevice = new MCCIMT000300UV01Device();
		receiverDevice.setDeterminerCode(HL7Constants.RECEIVER_DETERMINER_CODE);
		receiverDevice.setClassCode(EntityClassDevice.DEV);

		// CCR 177986
		logger.debug("Setting receiver device id (applicationId) to query sender's device id {} ", app);

		receiverDevice.getId().add(HL7DataTransformHelper.IIFactory(app));

		MCCIMT000300UV01Agent agent = new MCCIMT000300UV01Agent();
		MCCIMT000300UV01Organization org = new MCCIMT000300UV01Organization();
		org.setClassCode(HL7Constants.ORG_CLASS_CODE);
		org.setDeterminerCode(HL7Constants.RECEIVER_DETERMINER_CODE);
		II id = HL7DataTransformHelper.IIFactory(oid);
		org.getId().add(id);

		javax.xml.namespace.QName xmlqnameorg = new javax.xml.namespace.QName("urn:hl7-org:v3", "representedOrganization");
		JAXBElement<MCCIMT000300UV01Organization> orgElem = new JAXBElement<MCCIMT000300UV01Organization>(xmlqnameorg, MCCIMT000300UV01Organization.class, org);
		agent.setRepresentedOrganization(orgElem);
		agent.getClassCode().add(HL7Constants.AGENT_CLASS_CODE);

		javax.xml.namespace.QName xmlqnameagent = new javax.xml.namespace.QName("urn:hl7-org:v3", "asAgent");
		JAXBElement<MCCIMT000300UV01Agent> agentElem = new JAXBElement<MCCIMT000300UV01Agent>(xmlqnameagent, MCCIMT000300UV01Agent.class, agent);

		receiverDevice.setAsAgent(agentElem);

		receiver.setDevice(receiverDevice);

		// CCR 177986
		logger.debug("receiver details {} ");

		return receiver;
	}

	private static MCCIMT000300UV01Sender createSender(MCCIMT000100UV01Receiver queryReceiver)
	{
		MCCIMT000300UV01Sender sender = new MCCIMT000300UV01Sender();
		String app = null;
		String oid = null;

		sender.setTypeCode(CommunicationFunctionType.SND);

		if(queryReceiver.getDevice() != null && NullChecker.isNotNullish(queryReceiver.getDevice().getId()) && queryReceiver.getDevice().getId().get(0) != null && NullChecker.isNotNullish(queryReceiver.getDevice().getId().get(0).getRoot()))
		{
			app = queryReceiver.getDevice().getId().get(0).getRoot();

			// CCR 177986
			logger.debug("OID is {} ", oid);
		}

		if(queryReceiver.getDevice() != null && queryReceiver.getDevice().getAsAgent() != null && queryReceiver.getDevice().getAsAgent().getValue() != null && queryReceiver.getDevice().getAsAgent().getValue().getRepresentedOrganization() != null && queryReceiver.getDevice().getAsAgent().getValue().getRepresentedOrganization().getValue() != null && NullChecker.isNotNullish(queryReceiver.getDevice().getAsAgent().getValue().getRepresentedOrganization().getValue().getId()) && queryReceiver.getDevice().getAsAgent().getValue().getRepresentedOrganization().getValue().getId().get(0) != null && NullChecker.isNotNullish(queryReceiver.getDevice().getAsAgent().getValue().getRepresentedOrganization().getValue().getId().get(0).getRoot()))
		{
			oid = queryReceiver.getDevice().getAsAgent().getValue().getRepresentedOrganization().getValue().getId().get(0).getRoot();

			// CCR 177986
			logger.debug("OID is {} ", oid);
		}

		if(NullChecker.isNullish(oid))
		{
			oid = app;
		}

		MCCIMT000300UV01Device senderDevice = new MCCIMT000300UV01Device();
		senderDevice.setDeterminerCode(HL7Constants.SENDER_DETERMINER_CODE);
		senderDevice.setClassCode(EntityClassDevice.DEV);

		// CCR 177986
		logger.debug("Setting sender device id (applicationId) to query receiver's device id {}", app);

		senderDevice.getId().add(HL7DataTransformHelper.IIFactory(app));

		MCCIMT000300UV01Agent agent = new MCCIMT000300UV01Agent();
		MCCIMT000300UV01Organization org = new MCCIMT000300UV01Organization();
		org.setClassCode(HL7Constants.ORG_CLASS_CODE);
		org.setDeterminerCode(HL7Constants.SENDER_DETERMINER_CODE);
		II id = HL7DataTransformHelper.IIFactory(oid);
		org.getId().add(id);

		javax.xml.namespace.QName xmlqnameorg = new javax.xml.namespace.QName("urn:hl7-org:v3", "representedOrganization");
		JAXBElement<MCCIMT000300UV01Organization> orgElem = new JAXBElement<MCCIMT000300UV01Organization>(xmlqnameorg, MCCIMT000300UV01Organization.class, org);
		agent.setRepresentedOrganization(orgElem);
		agent.getClassCode().add(HL7Constants.AGENT_CLASS_CODE);

		javax.xml.namespace.QName xmlqnameagent = new javax.xml.namespace.QName("urn:hl7-org:v3", "asAgent");
		JAXBElement<MCCIMT000300UV01Agent> agentElem = new JAXBElement<MCCIMT000300UV01Agent>(xmlqnameagent, MCCIMT000300UV01Agent.class, agent);

		senderDevice.setAsAgent(agentElem);

		sender.setDevice(senderDevice);

		// CCR 177986
		logger.debug("sender is {} ", sender);

		return sender;
	}

	private static ADExplicit createAddress(Address add)
	{
		org.hl7.v3.ObjectFactory factory = new org.hl7.v3.ObjectFactory();
		ADExplicit result = (ADExplicit) (factory.createADExplicit());
		List<Serializable> addrlist = result.getContent();

		if(add != null)
		{
			if(add.getStreet1() != null && add.getStreet1().length() > 0)
			{
				AdxpExplicitStreetAddressLine street = new AdxpExplicitStreetAddressLine();
				street.setContent(add.getStreet1());

				addrlist.add(factory.createADExplicitStreetAddressLine(street));
			}

			if(add.getStreet2() != null && add.getStreet2().length() > 0)
			{
				AdxpExplicitStreetAddressLine street = new AdxpExplicitStreetAddressLine();
				street.setContent(add.getStreet2());

				addrlist.add(factory.createADExplicitStreetAddressLine(street));
			}
			if(add.getCity() != null && add.getCity().length() > 0)
			{
				AdxpExplicitCity city = new AdxpExplicitCity();
				city.setContent(add.getCity());

				addrlist.add(factory.createADExplicitCity(city));
			}
			if(add.getState() != null && add.getState().length() > 0)
			{
				AdxpExplicitState state = new AdxpExplicitState();
				state.setContent(add.getState());

				addrlist.add(factory.createADExplicitState(state));
			}
			if(add.getPostal() != null && add.getPostal().length() > 0)
			{
				AdxpExplicitPostalCode zip = new AdxpExplicitPostalCode();
				zip.setContent(add.getPostal());

				addrlist.add(factory.createADExplicitPostalCode(zip));
			}
		}

		// CCR 177986
		logger.debug("Address listr {}");

		return result;
	}

	static private JAXBElement<PRPAMT201306UV02QueryByParameter> createQueryByParameter(PRPAMT201306UV02QueryByParameter fromQueryByParameter)
	{
		PRPAMT201306UV02QueryByParameter ret = new PRPAMT201306UV02QueryByParameter();
		ret.setExecutionAndDeliveryTime(fromQueryByParameter.getExecutionAndDeliveryTime());
		ret.setInitialQuantity(fromQueryByParameter.getInitialQuantity());
		ret.setInitialQuantityCode(fromQueryByParameter.getInitialQuantityCode());
		ret.setMatchCriterionList(fromQueryByParameter.getMatchCriterionList());
		ret.setModifyCode(fromQueryByParameter.getModifyCode());
		ret.setParameterList(createParameterList(fromQueryByParameter.getParameterList()));
		ret.setQueryId(fromQueryByParameter.getQueryId());
		ret.setResponseModalityCode(fromQueryByParameter.getResponseModalityCode());
		ret.setResponsePriorityCode(fromQueryByParameter.getResponsePriorityCode());
		ret.setStatusCode(fromQueryByParameter.getStatusCode());
		ret.setTypeId(fromQueryByParameter.getTypeId());

		if(!fromQueryByParameter.getNullFlavor().isEmpty())
		{
			ret.getNullFlavor().addAll(fromQueryByParameter.getNullFlavor());
		}

		if(!fromQueryByParameter.getRealmCode().isEmpty())
		{
			ret.getRealmCode().addAll(fromQueryByParameter.getRealmCode());
		}

		if(!fromQueryByParameter.getResponseElementGroupId().isEmpty())
		{
			ret.getResponseElementGroupId().addAll(fromQueryByParameter.getResponseElementGroupId());
		}

		if(!fromQueryByParameter.getSortControl().isEmpty())
		{
			ret.getSortControl().addAll(fromQueryByParameter.getSortControl());
		}

		if(!fromQueryByParameter.getTemplateId().isEmpty())
		{
			ret.getTemplateId().addAll(fromQueryByParameter.getTemplateId());
		}

		org.hl7.v3.ObjectFactory objFactory = new org.hl7.v3.ObjectFactory();
		return objFactory.createPRPAIN201306UV02MFMIMT700711UV01ControlActProcessQueryByParameter(ret);
	}

	static private PRPAMT201306UV02ParameterList createParameterList(PRPAMT201306UV02ParameterList fromParameterList)
	{
		PRPAMT201306UV02ParameterList ret = new PRPAMT201306UV02ParameterList();
		ret.setId(fromParameterList.getId());
		ret.setTypeId(fromParameterList.getTypeId());

		if(!fromParameterList.getLivingSubjectAdministrativeGender().isEmpty())
		{
			ret.getLivingSubjectAdministrativeGender().addAll(fromParameterList.getLivingSubjectAdministrativeGender());
		}

		if(!fromParameterList.getLivingSubjectBirthPlaceAddress().isEmpty())
		{
			ret.getLivingSubjectBirthPlaceAddress().addAll(fromParameterList.getLivingSubjectBirthPlaceAddress());
		}

		if(!fromParameterList.getLivingSubjectBirthPlaceName().isEmpty())
		{
			ret.getLivingSubjectBirthPlaceName().addAll(fromParameterList.getLivingSubjectBirthPlaceName());
		}

		if(!fromParameterList.getLivingSubjectBirthTime().isEmpty())
		{
			ret.getLivingSubjectBirthTime().addAll(fromParameterList.getLivingSubjectBirthTime());
		}

		if(!fromParameterList.getLivingSubjectDeceasedTime().isEmpty())
		{
			ret.getLivingSubjectDeceasedTime().addAll(fromParameterList.getLivingSubjectDeceasedTime());
		}

		if(!fromParameterList.getLivingSubjectId().isEmpty())
		{
			ret.getLivingSubjectId().addAll(fromParameterList.getLivingSubjectId());
		}

		if(!fromParameterList.getLivingSubjectName().isEmpty())
		{
			ret.getLivingSubjectName().addAll(createLivingSubjectName(fromParameterList.getLivingSubjectName()));
		}

		if(!fromParameterList.getMothersMaidenName().isEmpty())
		{
			ret.getMothersMaidenName().addAll(fromParameterList.getMothersMaidenName());
		}

		if(!fromParameterList.getNullFlavor().isEmpty())
		{
			ret.getNullFlavor().addAll(fromParameterList.getNullFlavor());
		}

		if(!fromParameterList.getOtherIDsScopingOrganization().isEmpty())
		{
			ret.getOtherIDsScopingOrganization().addAll(fromParameterList.getOtherIDsScopingOrganization());
		}

		if(!fromParameterList.getPatientAddress().isEmpty())
		{
			ret.getPatientAddress().addAll(fromParameterList.getPatientAddress());
		}

		if(!fromParameterList.getPatientStatusCode().isEmpty())
		{
			ret.getPatientStatusCode().addAll(fromParameterList.getPatientStatusCode());
		}

		if(!fromParameterList.getPatientTelecom().isEmpty())
		{
			ret.getPatientTelecom().addAll(fromParameterList.getPatientTelecom());
		}

		if(!fromParameterList.getPrincipalCareProviderId().isEmpty())
		{
			ret.getPrincipalCareProviderId().addAll(fromParameterList.getPrincipalCareProviderId());
		}

		if(!fromParameterList.getPrincipalCareProvisionId().isEmpty())
		{
			ret.getPrincipalCareProvisionId().addAll(fromParameterList.getPrincipalCareProvisionId());
		}

		if(!fromParameterList.getRealmCode().isEmpty())
		{
			ret.getRealmCode().addAll(fromParameterList.getRealmCode());
		}

		if(!fromParameterList.getTemplateId().isEmpty())
		{
			ret.getTemplateId().addAll(fromParameterList.getTemplateId());
		}

		// CCR 177986
		logger.debug("Living Subjects list size in PRPAMT201306UV02ParameterList is {} ", ret.getLivingSubjectName().size());

		return ret;
	}

	static private List<PRPAMT201306UV02LivingSubjectName> createLivingSubjectName(List<PRPAMT201306UV02LivingSubjectName> fromLivingSubjectName)
	{
		List<PRPAMT201306UV02LivingSubjectName> ret = new ArrayList<PRPAMT201306UV02LivingSubjectName>();

		for(PRPAMT201306UV02LivingSubjectName fromLSN : fromLivingSubjectName)
		{
			PRPAMT201306UV02LivingSubjectName lsn = new PRPAMT201306UV02LivingSubjectName();

			lsn.setSemanticsText(fromLSN.getSemanticsText());
			lsn.setTypeId(fromLSN.getTypeId());

			if(!fromLSN.getNullFlavor().isEmpty())
			{
				lsn.getNullFlavor().addAll(fromLSN.getNullFlavor());
			}

			if(!fromLSN.getRealmCode().isEmpty())
			{
				lsn.getRealmCode().addAll(fromLSN.getRealmCode());
			}

			if(!fromLSN.getTemplateId().isEmpty())
			{
				lsn.getTemplateId().addAll(fromLSN.getTemplateId());
			}

			if(!fromLSN.getValue().isEmpty())
			{
				lsn.getValue().addAll(createValue(fromLSN.getValue()));
			}

			ret.add(lsn);
		}

		return ret;
	}

	static private List<ENExplicit> createValue(List<ENExplicit> fromValue)
	{
		List<ENExplicit> ret = new ArrayList<ENExplicit>();

		for(ENExplicit fromE : fromValue)
		{
			ENExplicit e = new ENExplicit();

			if(!fromE.getContent().isEmpty())
			{
				e.getContent().addAll(fromE.getContent());
			}

			if(!fromE.getNullFlavor().isEmpty())
			{
				e.getNullFlavor().addAll(fromE.getNullFlavor());
			}

			if(!fromE.getUse().isEmpty())
			{
				e.getUse().addAll(fromE.getUse());
			}

			ret.add(e);
		}

		// CCR 177986
		logger.debug("ENExplicit list was exited");

		return ret;
	}
}
