package gov.va.med.nhin.adapter.adaptergateway.docretrieve;

import java.util.concurrent.Callable;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.ExecutorCompletionService;
import java.util.concurrent.ExecutorService;

import javax.ejb.EJB;
import javax.ejb.*;

import org.apache.commons.lang3.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import gov.hhs.fha.nhinc.adapterpolicyengine.AdapterPolicyEnginePortType;
import gov.hhs.fha.nhinc.common.eventcommon.DocRetrieveEventType;
import gov.hhs.fha.nhinc.common.eventcommon.DocRetrieveMessageType;
import gov.hhs.fha.nhinc.common.nhinccommon.AssertionType;
import gov.hhs.fha.nhinc.common.nhinccommon.HomeCommunityType;
import gov.hhs.fha.nhinc.common.nhinccommonadapter.CheckPolicyRequestType;
import gov.hhs.fha.nhinc.common.nhinccommonadapter.CheckPolicyResponseType;
import gov.hhs.fha.nhinc.common.nhinccommonentity.RespondingGatewayCrossGatewayRetrieveRequestType;
import gov.hhs.fha.nhinc.nhinclib.NhincConstants;
import gov.hhs.fha.nhinc.transform.policy.PolicyEngineTransformer;
import gov.va.med.nhin.adapter.adaptergateway.workmanager.WorkManagerExecutorServiceLocal;
import gov.va.med.nhin.adapter.facilitymanager.Facility;
import gov.va.med.nhin.adapter.facilitymanager.FacilityManager;
import gov.va.med.nhin.adapter.facilitymanager.FacilityManagerLocal;
import gov.va.med.nhin.adapter.facilitymanager.OperationOnOff;
import gov.va.med.nhin.adapter.logging.CheckPolicy;
import gov.va.med.nhin.adapter.logging.EntityRetrieveError;
import gov.va.med.nhin.adapter.logging.ErrorMessage;
import gov.va.med.nhin.adapter.logging.EventAuditingFactory;
import gov.va.med.nhin.adapter.logging.EventAuditingFactoryImpl;
import gov.va.med.nhin.adapter.logging.LogConstants.AuditingEvent;
import gov.va.med.nhin.adapter.logging.MaintLog;
import gov.va.med.nhin.adapter.policyengine.AdapterPolicyEnginePortTypeLocal;
import ihe.iti.xds_b._2007.RetrieveDocumentSetRequestType;
import ihe.iti.xds_b._2007.RetrieveDocumentSetRequestType.DocumentRequest;
import ihe.iti.xds_b._2007.RetrieveDocumentSetResponseType;
import oasis.names.tc.ebxml_regrep.xsd.rs._3.RegistryError;
import oasis.names.tc.ebxml_regrep.xsd.rs._3.RegistryErrorList;
import oasis.names.tc.ebxml_regrep.xsd.rs._3.RegistryResponseType;
import org.hl7.fhir.dstu3.model.AuditEvent;

/**
 *
 * @author DNS   
 */

@TransactionAttribute(value = TransactionAttributeType.SUPPORTS)
@Stateless(name = "AdapterGatewayDocRetrieve")
public class AdapterGatewayDocRetrieve implements EntityDocRetrievePortTypeLocal
{
	private static final Logger logger = LoggerFactory.getLogger(AdapterGatewayDocRetrieve.class.getName());

	private class SendRequestResponse
	{
		public RetrieveDocumentSetResponseType retrieveDocumentSetResponse;
		public SendRequestException sre = null;

		public boolean hasError()
		{
			return sre != null;
		}
	}

	private class SendRequestException extends Exception
	{
		private static final long serialVersionUID = 1L;
		
		public String remoteHomeCommunityId;
	}

	private class SendRequestCallable implements Callable<SendRequestResponse>
	{
		public final RetrieveDocumentSetRequestType retrieveDocumentSetRequest;
		public final AssertionType assertion;
		public final String remoteHomeCommunityId;

		public SendRequestCallable(RetrieveDocumentSetRequestType retrieveDocumentSetRequest, AssertionType assertion, String remoteHomeCommunityId)
		{
			this.retrieveDocumentSetRequest = retrieveDocumentSetRequest;
			this.assertion = assertion;
			this.remoteHomeCommunityId = remoteHomeCommunityId;

			logger.info("RD request callable created.");
		}

		@Override
		public SendRequestResponse call() throws Exception
		{
			logger.info("RD request callable sending request...");
			return sendRequest(retrieveDocumentSetRequest, assertion, remoteHomeCommunityId);
		}
	}

	private FacilityManager facilityManager;
	private AdapterPolicyEnginePortType adapterPolicyEngine;
	private NHINDocRetrieveRequestSender nhinDocRetrieveRequestSender;
	private ExecutorService executorService;

	@EJB(beanInterface = FacilityManagerLocal.class, beanName = "FacilityManager")
	public void setFacilityManager(FacilityManager facilityManager)
	{
		this.facilityManager = facilityManager;
	}

	@EJB(beanInterface = AdapterPolicyEnginePortTypeLocal.class, beanName = "AdapterPolicyEngine")
	public void setAdapterPolicyEngine(AdapterPolicyEnginePortType adapterPolicyEngine)
	{
		this.adapterPolicyEngine = adapterPolicyEngine;
	}

	@EJB(beanInterface = NHINDocRetrieveRequestSender.class, beanName = "NhincProxyNHINDocRetrieveRequestSender")
	public void setNhinDocQueryRequestSender(NHINDocRetrieveRequestSender nhinDocQueryRequestSender)
	{
		this.nhinDocRetrieveRequestSender = nhinDocQueryRequestSender;
	}

	//@EJB(beanInterface = WorkManagerExecutorServiceLocal.class, beanName = "WorkManagerExecutorService")
	@EJB(beanInterface = WorkManagerExecutorServiceLocal.class, beanName = "OutboundRDWorkManagerExecutorService")  //changed 1/24/2017
        public void setExecutorService(ExecutorService executorService)
	{
		this.executorService = executorService;
	}

	@Override
	public RetrieveDocumentSetResponseType respondingGatewayCrossGatewayRetrieve(RespondingGatewayCrossGatewayRetrieveRequestType respondingGatewayCrossGatewayRetrieveRequest)
	{

		EventAuditingFactory<AuditEvent> afac
				= EventAuditingFactoryImpl.getFactory( AuditEvent.class );
		afac.debug( afac.newEvent( respondingGatewayCrossGatewayRetrieveRequest.getAssertion(),
				AuditingEvent.RDOUT_BEGIN, getClass() ) );
		// CCR 177986
		logger.debug("respondingGatewayCrossGatewayRetrieve() invoked");

		RetrieveDocumentSetResponseType ret = new RetrieveDocumentSetResponseType();
		RetrieveDocumentSetRequestType retrieveDocumentSetRequest
				= respondingGatewayCrossGatewayRetrieveRequest.getRetrieveDocumentSetRequest();

		// CCR 177986
		logger.debug("RetrieveDocumentSetRequestType retrieveDocumentSetRequest {} ", retrieveDocumentSetRequest);

		AssertionType assertion = respondingGatewayCrossGatewayRetrieveRequest.getAssertion();

		// CCR 177986
		logger.debug(" AssertionType assertion {} ", assertion);

		ExecutorCompletionService<SendRequestResponse> completionService
				= new ExecutorCompletionService<>(executorService);
		int numSubmitted = 0;

		for(DocumentRequest documentRequest : retrieveDocumentSetRequest.getDocumentRequest())
		{
			RetrieveDocumentSetRequestType newRequest = createRetrieveDocumentSetRequest(documentRequest);

			logger.debug("RetrieveDocumentSetRequestType newRequest {} ", newRequest); // CCR
																						// 177986
			logger.debug("Home Community Id {} : ", documentRequest.getHomeCommunityId());

			if(checkPolicy(newRequest, assertion, documentRequest.getHomeCommunityId()))
			{
				if(facilityManager.isPartnerAllowed(documentRequest.getHomeCommunityId(), OperationOnOff.ONBOARD))
				{
					if(facilityManager.isPartnerAllowed(documentRequest.getHomeCommunityId(), OperationOnOff.OUT_DR))
					{
						afac.debug( afac.messaging().partnerauth( AuditingEvent.RDOUT_PARTNERAUTH,
								getClass(), documentRequest.getHomeCommunityId() ) );

						SendRequestCallable callable = new SendRequestCallable(newRequest, assertion, documentRequest.getHomeCommunityId());
						afac.debug( afac.messaging().reqres( AuditingEvent.RDOUT_RD, getClass(),
								documentRequest.getHomeCommunityId() ) );
						completionService.submit(callable);
						++numSubmitted;

						logger.debug("numSubmitted {} ", numSubmitted);
					}
					else
					{
						// Outbound DR Error
						addErrorToResponse(ret, "Internal Registry/Repository Error: " + ErrorMessage.OUT_DR_DISABLED.getMessage(), documentRequest.getHomeCommunityId());
						MaintLog.queryError(null, ErrorMessage.OUT_DQ_DISABLED, documentRequest.getHomeCommunityId(), logger);
					}
				}
				else
				{
					// Outbound no partner
					addErrorToResponse(ret, "Internal Registry/Repository Error: " + ErrorMessage.OUT_DR_NOT_A_PARTNER.getMessage(), documentRequest.getHomeCommunityId());
					MaintLog.queryError(null, ErrorMessage.OUT_DR_NOT_A_PARTNER, documentRequest.getHomeCommunityId(), logger);
				}
			}
		}

		for(int i = 0; i < numSubmitted; ++i)
		{
			try
			{
				SendRequestResponse response = completionService.take().get();
				if(!response.hasError())
				{
					addResultToResponse(ret, response.retrieveDocumentSetResponse);
				}
				else
				{
					SendRequestException sre = response.sre;
					addErrorToResponse(ret, "Internal Registry/Repository Error - HCID = " + sre.remoteHomeCommunityId, sre.remoteHomeCommunityId);
					logger.warn("Error sending DocRetrieve for community id: {} with exception: {} ", sre.remoteHomeCommunityId, sre); // CCR
																																		// 177986-
																																		// logging

				}
			}
			catch(ExecutionException ee)
			{
				logger.error("An Unknown error occured processing an outbound doc retrieve request.", ee);
				EntityRetrieveError.retrieveError(respondingGatewayCrossGatewayRetrieveRequest, ErrorMessage.OUT_DR_UKNOWN, ee.getMessage());
			}
			catch(InterruptedException ie)
			{
				logger.error("Outbound Doc Retrieve Threads interrupted.  Return will not be complete.", ie);
			}
		}

		afac.debug( AuditingEvent.RDOUT_END, getClass() );
		return ret;
	}

	private RetrieveDocumentSetRequestType createRetrieveDocumentSetRequest(DocumentRequest documentRequest)
	{
		RetrieveDocumentSetRequestType ret = new RetrieveDocumentSetRequestType();
		ret.getDocumentRequest().add(documentRequest);
		return ret;
	}

	private boolean checkPolicy(RetrieveDocumentSetRequestType retrieveDocumentSetRequest, AssertionType assertion, String remoteHomeCommunityId)
	{
		logger.debug("DR Check Policy");

		DocRetrieveMessageType docRetrieveMessage = new DocRetrieveMessageType();
		docRetrieveMessage.setRetrieveDocumentSetRequest(retrieveDocumentSetRequest);
		docRetrieveMessage.setAssertion(assertion);
		logger.info("Request docRetrieveMessage created");

		HomeCommunityType receivingHomeCommunity = new HomeCommunityType();
		receivingHomeCommunity.setHomeCommunityId(remoteHomeCommunityId);
		logger.debug("Request receivingHomeCommunity: {}", remoteHomeCommunityId);

		HomeCommunityType sendingHomeCommunity = new HomeCommunityType();
		sendingHomeCommunity.setHomeCommunityId(getHomeCommunityId());
		logger.debug("Request sendingHomeCommunity: {}", sendingHomeCommunity.getHomeCommunityId());

		DocRetrieveEventType docRetrieveEvent = new DocRetrieveEventType();
		docRetrieveEvent.setMessage(docRetrieveMessage);
		docRetrieveEvent.setDirection(NhincConstants.POLICYENGINE_OUTBOUND_DIRECTION);
		logger.debug("Request docRetrieveEvent Direction: {}", NhincConstants.POLICYENGINE_OUTBOUND_DIRECTION);
		docRetrieveEvent.setInterface(NhincConstants.AUDIT_LOG_ENTITY_INTERFACE);
		logger.debug("Request docRetrieveEvent Interface: {}", NhincConstants.AUDIT_LOG_ENTITY_INTERFACE);
		docRetrieveEvent.setReceivingHomeCommunity(receivingHomeCommunity);
		docRetrieveEvent.setSendingHomeCommunity(sendingHomeCommunity);
		logger.debug("Request docRetrieveEvent created");

		PolicyEngineTransformer transformer = new PolicyEngineTransformer();
		CheckPolicyRequestType checkPolicyRequest = transformer.transformDocRetrieveToCheckPolicy(docRetrieveEvent);
		logger.debug("checkPolicyRequest completed");

        if (facilityManager.getFacilityByFacilityNumber("200NSS") == null || !remoteHomeCommunityId.contains(facilityManager.getFacilityByFacilityNumber("200NSS").getFullHomeCommunityId()))
		{
			logger.info("Going to checkPolicy in adapterPolicyEngine...");
			CheckPolicyResponseType checkPolicyResponse = adapterPolicyEngine.checkPolicy(checkPolicyRequest);
			return CheckPolicy.checkPolicy(checkPolicyRequest, checkPolicyResponse);
		}

		return true;
	}

	private void addResultToResponse(RetrieveDocumentSetResponseType retrieveDocumentSetResponse, RetrieveDocumentSetResponseType result)
	{
		logger.debug("In addResultToResponse");
		if(retrieveDocumentSetResponse.getRegistryResponse() == null)
		{
			if(result.getRegistryResponse() != null)
			{
				retrieveDocumentSetResponse.setRegistryResponse(result.getRegistryResponse());
			}
			else
			{
				retrieveDocumentSetResponse.setRegistryResponse(new RegistryResponseType());
			}
		}

		if(result.getRegistryResponse().getRegistryErrorList() != null)
		{
			logger.debug("Registry Error List is not null. Adding errors to Registry Response");
			if(retrieveDocumentSetResponse.getRegistryResponse() == null)
			{
				retrieveDocumentSetResponse.setRegistryResponse(new RegistryResponseType());
			}
			if(retrieveDocumentSetResponse.getRegistryResponse().getRegistryErrorList() == null)
			{
				retrieveDocumentSetResponse.getRegistryResponse().setRegistryErrorList(new RegistryErrorList());
			}
			retrieveDocumentSetResponse.getRegistryResponse().getRegistryErrorList().getRegistryError().addAll(result.getRegistryResponse().getRegistryErrorList().getRegistryError());
			retrieveDocumentSetResponse.getRegistryResponse().getRegistryErrorList().setHighestSeverity(result.getRegistryResponse().getRegistryErrorList().getHighestSeverity());
		}

		if(result.getRegistryResponse().getStatus() != null)
		{
			if(retrieveDocumentSetResponse.getRegistryResponse() != null)
			{
				retrieveDocumentSetResponse.getRegistryResponse().setStatus(result.getRegistryResponse().getStatus());
			}
		}
		retrieveDocumentSetResponse.getDocumentResponse().addAll(result.getDocumentResponse());
	}

	private void addErrorToResponse(RetrieveDocumentSetResponseType retrieveDocumentSetResponse, String errorString, final String hcid)
	{
		logger.debug("In addErrorToResponse");
		if(retrieveDocumentSetResponse != null)
		{
			if(retrieveDocumentSetResponse.getRegistryResponse() == null)
			{
				retrieveDocumentSetResponse.setRegistryResponse(new RegistryResponseType());
			}

			if(retrieveDocumentSetResponse.getRegistryResponse().getRegistryErrorList() == null)
			{
				retrieveDocumentSetResponse.getRegistryResponse().setRegistryErrorList(new RegistryErrorList());
			}

			retrieveDocumentSetResponse.getRegistryResponse().getRegistryErrorList().setHighestSeverity("urn:oasis:names:tc:ebxml-regrep:ErrorSeverityType:Error");

			RegistryError registryError = new RegistryError();
			registryError.setErrorCode("XDSRegistryError");
			registryError.setCodeContext(errorString);
			
			if(StringUtils.isNotBlank(hcid))
			{
				registryError.setLocation(hcid);
			}
			
			registryError.setSeverity("urn:oasis:names:tc:ebxml-regrep:ErrorSeverityType:Error");
			retrieveDocumentSetResponse.getRegistryResponse().getRegistryErrorList().getRegistryError().add(registryError);
		}
	}

	private boolean communitySupports2011Specs(String hieCommunityOid)
	{
		logger.debug("In communitySupports2011Specs");
		// fetch the facility information
		Facility facility = facilityManager.getFacilityByFullHomeCommunityId(hieCommunityOid);
		if(facility == null)
		{
			logger.error("Error getting facility from {}", hieCommunityOid);
			return false;
		}
		return Facility.SPEC_VERSION_2011.equals(facility.getUseSpecVersion());
	}

	private SendRequestResponse sendRequest(RetrieveDocumentSetRequestType retrieveDocumentSetRequest, AssertionType assertion, String remoteHomeCommunityId) throws SendRequestException
	{
		logger.info("DR Send Request...");

		SendRequestResponse ret = new SendRequestResponse();

		try
		{
			boolean use2011Spec = false;
			if(communitySupports2011Specs(remoteHomeCommunityId))
			{
				use2011Spec = true;
			}
			ret.retrieveDocumentSetResponse = nhinDocRetrieveRequestSender.sendRequest(retrieveDocumentSetRequest, assertion, remoteHomeCommunityId, use2011Spec);
			logger.debug("RD response recieved from SSA");
		}
		catch(Throwable e)
		{
			SendRequestException sre = new SendRequestException();
			sre.remoteHomeCommunityId = remoteHomeCommunityId;
			sre.initCause(e);
			ret.sre = sre;
			logger.debug("Error sending RD to SSA: {}", e.getMessage());
		}
		return ret;
	}

	private String getHomeCommunityId()
	{
		String ret = null;
		Facility facility = facilityManager.getFacilityByFacilityNumber("VA");
		if(facility != null)
		{
			ret = facility.getHomeCommunityId();
		}
		return ret;
	}
}
