package gov.va.med.nhin.adapter.terminology;

import java.sql.Connection;
import java.sql.DatabaseMetaData;
import java.sql.PreparedStatement;
import java.sql.ResultSet;
import java.sql.SQLException;
import java.util.Map;
import java.util.HashMap;

import javax.annotation.Resource;
import javax.ejb.*;
import javax.persistence.EntityManager;
import javax.persistence.NoResultException;
import javax.persistence.PersistenceContext;
import javax.persistence.Query;
import javax.sql.DataSource;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import gov.va.med.nhin.adapter.utils.NullChecker;

/**
 *
 * @author David Vazquez
 */
// Does not have to be a web service for now.
//@WebService(serviceName = "TerminologyService",
//            wsdlLocation = "META-INF/wsdl/Terminology.wsdl",
//            portName = "TerminologyPort",
//            endpointInterface = "gov.va.med.nhin.adapter.terminology.TerminologyPortType",
//            targetNamespace = "urn:gov:va:med:nhin:adapter:terminology")
@TransactionAttribute(value = TransactionAttributeType.SUPPORTS)
@Stateless(name = "Terminology", mappedName = "Terminology")
public class TerminologyBean implements TerminologyLocal, TerminologyRemote, TerminologyPortType
{
    private static final Logger logger = LoggerFactory.getLogger(TerminologyBean.class.getName());

    @Resource(name = "jdbc/adapter")
    private DataSource dataSource;

    @PersistenceContext
    private EntityManager entityManager;

    @Override
    public String lookup(String codeSet,
                         String code,
                         String field)
    {
        String ret = "";
        Connection connection = null;

        try {
            connection = getConnection();
            Codeset codeset = getCodeset(codeSet);
            if (codeset != null) {
                ret = getValue(connection, codeset, code, field);
            }
        }
        catch (SQLException se) {
            throw new RuntimeException("A database error occurred.", se);
        }
        finally {
            closeConnection(connection);
        }

        return ret;
    }

    @Override
    public String translate(String sourceCodeset,
                            String code,
                            String targetCodeset,
                            String field)
    {
        String ret = "";
        Connection connection = null;

        try {
            connection = getConnection();
            Translation translation = getTranslation(sourceCodeset, code, targetCodeset);
            if (translation != null) {
                Codeset codeset = getCodeset(translation.getTargetCodeset());
                if (codeset != null) {
                    ret = getValue(connection, codeset, translation.getTargetCode(), field);
                }
            }
        }
        catch (SQLException se) {
            throw new RuntimeException("A database error occurred.", se);
        }
        finally {
            closeConnection(connection);
        }

        return ret;
    }

    private Connection getConnection() throws SQLException
    {
        return dataSource.getConnection();
    }

    private void closeConnection(Connection connection)
    {
        try {
            if (connection != null) {
                connection.close();
            }
        }
        catch (SQLException se) {
            logger.error("SQL Error occurred in closeConnection(): ", se);
        }
    }

    private Codeset getCodeset(String codeset)
    {
        Codeset ret;
        Query query = entityManager.createNamedQuery("Codeset.findByCodeset");
        query.setParameter("codeset", codeset);

        try {
            ret = (Codeset) query.getSingleResult();
        }
        catch (NoResultException nre) {
            ret = null;
            logger.error("Error occurred in getCodeset(): ", nre);
        }

        return ret;
    }

    private String getValue(Connection connection, Codeset codeset, String code, String field) throws SQLException
    {
        String ret = null;
        Map<String, String> metaData = getValidTableAndColumnNames(connection, codeset, field);
        if (NullChecker.isNotNullOrEmpty(metaData) && metaData.size() == 3) {
            String sql = "select " + metaData.get("field") + " from " + metaData.get("lookupTable") + " where " + metaData.get("keyField") + "= ?";
            PreparedStatement statement = connection.prepareStatement(sql);
            statement.setString(1, code);
            ResultSet rs = statement.executeQuery();
            if (rs.next()) {
                ret = rs.getString(1);
            }
        }
        else {
            logger.warn("Codeset is not set up correctly (e. g. configuration references objects that are not in the database).");
        }

        logger.debug("value {}", ret);

        return ret;
    }

    private Map<String, String> getValidTableAndColumnNames(Connection connection, Codeset codeset, String field)
            throws SQLException
    {
        Map<String, String> ret = new HashMap<>();
        DatabaseMetaData meta = connection.getMetaData();

        ResultSet columns = meta.getColumns(null, null, codeset.getLookupTable(), null);
        while (columns.next()) {
            ret.put("lookupTable", columns.getString("TABLE_NAME"));
            if (columns.getString("COLUMN_NAME").equalsIgnoreCase(codeset.getKeyField())) {
                ret.put("keyField", columns.getString("COLUMN_NAME"));
            }
            if (columns.getString("COLUMN_NAME").equalsIgnoreCase(field)) {
                ret.put("field", columns.getString("COLUMN_NAME"));
            }
        }

        return ret;
    }

    private Translation getTranslation(String sourceCodeset, String sourceCode, String targetCodeset)
    {
        Translation ret;
        Query query = entityManager.createNamedQuery("Translation.findTranslation");
        query.setParameter("sourceCodeset", sourceCodeset);
        query.setParameter("sourceCode", sourceCode);
        query.setParameter("targetCodeset", targetCodeset);

        try {
            ret = (Translation) query.getSingleResult();
        }
        catch (NoResultException nre) {
            logger.warn("No results for getTranslation( \"{}\", \"{}\", \"{}\" ): {}",
								sourceCodeset, sourceCode, targetCodeset, nre.getLocalizedMessage() );
            ret = null;
        }

        return ret;
    }
}
