package gov.va.med.nhin.adapter.audit;

import java.io.File;
import java.io.FileInputStream;
import java.io.InputStream;
import java.io.Serializable;
import java.util.Properties;

import javax.annotation.Resource;
import javax.ejb.*;
import javax.ejb.Timeout;
import javax.ejb.Timer;
import javax.ejb.TimerService;
import javax.sql.DataSource;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import gov.va.med.nhin.adapter.utils.DatabaseUtils;
import gov.va.med.nhin.adapter.utils.Utils;


@TransactionAttribute(value = TransactionAttributeType.NOT_SUPPORTED)
@Stateless(mappedName = "AuditBulkExport")
public class BulkExportBean implements BulkExport
{
   public static final String DQ_EXPORT_PROPS = "dqExport.properties";
   public static final String PROP_WATCH_DIR = "watch.directory";
   public static final String PROP_UNPROCESSED_EXT = "unprocessed.file.extension";
   public static final String PROP_PROCESSED_EXT = "processed.file.extension";
   public static final String PROP_TIMER_INTERVAL_SECS = "timer.interval.seconds";

   @Resource
   TimerService timerService;
   
   private static final Logger LOGGER = LoggerFactory.getLogger(BulkExportBean.class);

   private DataSource dataSource;

   @Resource(name = "jdbc/adapter")
   public void setDataSource(DataSource dataSource)
   {
      this.dataSource = dataSource;
   }

   /**
    * Creates a repeating timer to check for DQ Bulk Export trigger files.
    */
   @Override
   public void createTimer()
   {
   	// remove any old timers first
   	for(Object timerObj: timerService.getTimers())
   	{
   		Timer timer = (Timer)timerObj;
   		timer.cancel();
   	}

   	InputStream propInputStream = null;
		try
		{
			propInputStream = Utils.getResourceAsStream(DQ_EXPORT_PROPS);
			Properties props = new Properties();
			props.load(propInputStream);

			DQExportInfo dqExportInfo = new DQExportInfo();
			dqExportInfo.watchDirectoryPath = props.getProperty(PROP_WATCH_DIR);
			dqExportInfo.unprocessedFileExtension = props.getProperty(PROP_UNPROCESSED_EXT);
			dqExportInfo.processedFileExtension = props.getProperty(PROP_PROCESSED_EXT);

			String intervalStr = props.getProperty(PROP_TIMER_INTERVAL_SECS);
			long interval = Integer.parseInt(intervalStr) * 1000;
			timerService.createTimer(interval, interval, dqExportInfo);
			
			LOGGER.info("App will watch '" + dqExportInfo.watchDirectoryPath + "' every " + intervalStr + " seconds.");
		}
		catch (Exception ex)
		{
			LOGGER.error("Create DQ Export timer failed.", ex);
		}
		finally
		{
			DatabaseUtils.closeIO(propInputStream, LOGGER);
		}
   }
   
   @Timeout
   public void timeout(Timer timer) 
   {
	   try 
	   {
	   	DQExportInfo dqExportInfo = (DQExportInfo)timer.getInfo();
	   	File folder = new File(dqExportInfo.watchDirectoryPath);
	   	if(!folder.isDirectory())
	   	{
	   		return;
	   	}
		   File[] listOfFiles = folder.listFiles();
		   
		   for (File file : listOfFiles) 
		   {
			   if (file.isFile() && file.getName().endsWith(dqExportInfo.unprocessedFileExtension)) 
			   {
			   	processDqExportTrigger(file, dqExportInfo);
			   }
		   } 
	   } 
	   catch (Exception ex) 
	   {
		   LOGGER.error("DQ Bulk Export timer service failed.", ex);
	   }
   }

   /**
    * Sample file contents:
    * start.audit.datetime = 2016-11-01T00:00
    * end.audit.datetime = 2016-12-01T00:00
    * export.root.dir = /opt/dqExport
    * @param triggerPropsFile
    * @param dqExportInfo
    */
   private void processDqExportTrigger(File triggerPropsFile, DQExportInfo dqExportInfo)
   {
   	FileInputStream fis = null;
   	Properties triggerProps = new Properties();
   	try
   	{
   		fis = new FileInputStream(triggerPropsFile);
   		triggerProps.load(fis);

   		// loaded OK, now rename so we won't load it again
   		String fileName = triggerPropsFile.getName();
		   int extensionIndex = fileName.lastIndexOf(dqExportInfo.unprocessedFileExtension);
		   String processedFilename = fileName.substring(0, extensionIndex) + dqExportInfo.processedFileExtension;
		   boolean renamed = triggerPropsFile.renameTo(new File(triggerPropsFile.getParentFile(), processedFilename));
		   if(!renamed)
		   {
		   	LOGGER.warn("Unable to rename " + triggerPropsFile.getAbsolutePath() + ". It will be processed again.");
		   }
		   LOGGER.info("Bulk export triggered by '" + triggerPropsFile.getAbsolutePath() + "'.");
   	}
   	catch(Exception ex)
   	{
   		LOGGER.error("Reading of data quality bulk export trigger file failed.", ex);
   		return;
   	}
   	finally
   	{
   		DatabaseUtils.closeIO(fis, LOGGER);
   	}
   	
	   // now use info in there to do the actual export
   	try
   	{
      	BulkExportLib bel = new BulkExportLib(dataSource);
   		bel.exportRetrievedDocs(triggerProps);
	   }
	   catch(Exception ex)
	   {
	   	LOGGER.error("exportRetrievedDocs failed.", ex);
	   }
   }

   /**
    * Simple structure to hold the information needed to process a timed iteration of
    * the Data Quality Export.
    */
   public static class DQExportInfo implements Serializable
   {
		private static final long serialVersionUID = 1L;
		public String watchDirectoryPath;
      public String unprocessedFileExtension;
      public String processedFileExtension;
   }
}
