package gov.va.vss.web.interceptor;

import java.time.ZoneId;
import java.time.ZonedDateTime;
import java.time.format.TextStyle;
import java.util.Locale;
import java.util.Map;

import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;
import javax.servlet.http.HttpSession;

import org.apache.commons.lang3.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.core.env.Environment;
import org.springframework.stereotype.Component;

import gov.va.shared.model.AppUser;
import gov.va.shared.util.SecurityUtil;
import gov.va.vss.model.lookup.TemplateType;
import gov.va.vss.service.VelocityService;
import gov.va.vss.web.AbstractAppController;

@Component
public class CommonReferenceDataInterceptor extends AbstractReferenceDataInterceptor {
	private static final Logger log = LoggerFactory.getLogger(AbstractReferenceDataInterceptor.class);

	@Value("${solr.baseURL}")
	private String solrBaseURL;
	@Value("${session.staff.idleAfterSeconds}")
	private int sessionIdleAfterSeconds;
	@Value("${session.staff.expirationSeconds}")
	private int sessionExpirationSeconds;

	@Override
	protected void addAdditionalReferenceData(HttpServletRequest request, HttpServletResponse response, Object handler,
			Map<String, Object> model) throws Exception {
		HttpSession session = request.getSession();

		session.setMaxInactiveInterval(sessionExpirationSeconds);
		session.setAttribute("sessionIdleAfterSeconds", new Integer(sessionIdleAfterSeconds));

		AppUser appUser = SecurityUtil.getCurrentUserAs(AppUser.class);

		ZoneId timeZone = null;
		if (appUser != null) {
			model.put("currentUser", appUser);
			model.put("multipleStationsAssigned", appUser.getFacilities().size() > 1);
			timeZone = appUser.getTimeZone();

			if (timeZone != null) {
				model.put("userTimeZone", timeZone);
				model.put("userTimeZoneName", timeZone.getDisplayName(TextStyle.FULL, Locale.US));
			}
		}

		ZonedDateTime now = ZonedDateTime.now();
		if (timeZone != null)
			now = now.withZoneSameInstant(timeZone);
		model.put("currentTime", now);

		model.put("solrBaseURL", solrBaseURL);

		/* Sensible default for consistency in JSP layer - CPB */
		if (model.get(AbstractAppController.FORM_READ_ONLY) == null)
			model.put(AbstractAppController.FORM_READ_ONLY, false);

		/* Velocity caches these vals per the engine setup - CPB */
		model.put("referenceDataLinks",
				velocityService.mergeTemplateIntoString(TemplateType.REFERENCE_DATA_LINKS.getName()));
		populateBasicGlobalReferenceData(model, velocityService, env);
		model.put("systemNotification",
				StringUtils.trim(velocityService.mergeTemplateIntoString(TemplateType.SYSTEM_NOTIFICATION.getName())));
	}

	public static void populateBasicGlobalReferenceData(Map<String, Object> model, VelocityService velocityService,
			Environment env) {
		AbstractReferenceDataInterceptor.populateBasicGlobalReferenceData(model, velocityService, env);
		model.put("footerContent", velocityService.mergeTemplateIntoString(TemplateType.FOOTER_CONTENT.getName()));
	}

}
