const _ = require("underscore");
const Backbone = require("backbone");
const Marionette = require("backbone.marionette");
const Radio = require("backbone.radio");
const SelectPicker = require("../../behaviors/select-picker");
const authHeaders = require("../authHeaders");

module.exports = Marionette.StateView.extend({
	className: "entitySelect",
	template: require("./template.hbs"),

	CollectionClass: Backbone.Collection,
	label: "name",
	value: "id",
	selected: [],
	bootstrapSelect: {},

	defaultState: {
		loaded: false
	},

	ui: {
		input: ".js-selectpicker"
	},

	events: {
		"change @ui.input": "handleValueChange"
	},

	behaviors() {
		const defaultBSOptions = {
			container: "body",
			liveSearch: true,
			showTick: true
		};

		const providedBSOptions = this.getOption("bootstrapSelect");
		const spOptions = Object.assign({}, defaultBSOptions, providedBSOptions);

		return {
			Select: {
				behaviorClass: SelectPicker,
				selector: ".js-selectpicker",
				options: spOptions
			}
		};
	},

	initialize() {
		Marionette.StateView.prototype.initialize.apply(this, arguments);

		const dataKey = this.getOption("dataKey");
		const Collection = this.getOption("CollectionClass");

		if (!dataKey) {
			throw new Error("dataKey option required for EntitySelect view");
		}

		this.collection = new Collection();
		this.channel = Radio.channel("store");

		this._getData();
	},

	templateContext() {
		const isLoaded = this.state.get("loaded");
		const isEmpty = this.collection.isEmpty();

		let title = "Select one of the following";
		if (!isLoaded) { title = "Loading data..."; }
		else if (isEmpty) { title = "No data available"; }

		const ret = {
			title,
			ariaLabel: this.getOption("ariaLabel"),
			selected: [].concat(this.getOption("selected")),
			isDisabled: !isLoaded || isEmpty,
			options: this.collection.map(model => ({
				label: this._getLabel(model),
				value: this._getValue(model)
			}))
		};
		return ret;
	},

	onAttach() {
		const dataKey = this.getOption("dataKey");
		this._getData();
		this.listenTo(this.channel, `change:entity-select.${dataKey}`, this._applyData);
	},

	onBeforeDestroy() {
		this.stopListening(this.channel);
	},

	_getData() {
		const dataKey = this.getOption("dataKey");
		const forceFetch = this.getOption("forceFetch");
		const data = this.channel.request("get", `entity-select.${dataKey}`);
		const isLoading = this.channel.request("get", `entity-select.${dataKey}.loading`);

		// If it's already loading we'll just wait for the data to load
		if (isLoading) { return; }

		// Otherwise, fetch clean data if none exists or has been
		// explicitly requested
		if (forceFetch || !data) { return this._fetchData(); }

		// Apply the existing data
		this._applyData(data);
	},

	_fetchData() {
		const dataKey = this.getOption("dataKey");

		this.channel.request("set", `entity-select.${dataKey}.loading`, true);

		let fetchOpts = authHeaders();
		if (this.getOption("fetchOptions")) {
			fetchOpts.data = this.getOption("fetchOptions");
			fetchOpts.method = "POST";
		}
		return this.collection.fetch(fetchOpts).then(() => {
			this.channel.request("set", `entity-select.${dataKey}.loading`, false);
			this.channel.request("set", `entity-select.${dataKey}`, this.collection.toJSON());
		});
	},

	_applyData(data) {
		if (!data) { return; }
		this.state.set("loaded", true);

		const Collection = this.getOption("CollectionClass");
		this.collection = new Collection(data);

		if (this.isRendered()) { this.render(); }
	},

	_getLabel(model) {
		const label = this.getOption("label");
		return _.isFunction(label) ? label(model) : model.get(label);
	},

	_getValue(model) {
		const value = this.getOption("value");
		return _.isFunction(value) ? value(model) : model.get(value);
	},

	handleValueChange() {
		this.triggerMethod("change", this.getValue());
	},

	getValue() {
		return this.getUI("input").val();
	},

	setValue(value) {
		this.getUI("input").val(value);
		this.triggerMethod("selectpicker:update");
	}
});
