const Marionette = require("backbone.marionette");
const Radio = require("backbone.radio");
const authHeaders = require("../../utils/authHeaders");
const CaseCollection = require("../../entities/case/collection");
const CaseModel = require("../../entities/case/model");

module.exports = Marionette.StateView.extend({
	className: "selCase",
	template: require("./template.hbs"),

	defaultState: {
		loadingList: true,
		loadingCase: false,
		selectedCase: null,
	},

	templateContext() {
		return {
			// Add any/all selected case properties we want to display here
			casePropertyLookup: [
				{ prop: "ien", label: "IEN" },
				{ prop: "caseCreateDate", label: "Creation Date" },
				{ prop: "caseDateTime", label: "Date/Time" },
				{ prop: "cancelDate", label: "Cancel Date" },
				{ prop: "patient", label: "Patient" },
				{ prop: "hospitalAdmissionStatus", label: "Hospital Admission Status" },
				{ prop: "serviceConnected", label: "Service Connected" },
			],
		};
	},

	ui: {
		input: ".js-selCase-selection-input",
		confirm: ".js-selCase-confirm",
	},

	stateEvents: {
		"change": "render",
	},

	events: {
		"change @ui.input": "loadCase"
	},

	triggers: {
		"click @ui.confirm": "confirm"
	},

	// Note: use of "ECMAScript 6 Spread Operator" (aka ...) here
	// - See https://ariya.io/2013/03/es6-and-spread-operator for reference
	serializeData(...args) {
		const data = {
			...Marionette.StateView.prototype.serializeData.apply(this, args),
		};

		// By default Marionette will only pass the model data into
		// the template context if provided, but we want it to serialize
		// both the model & collection.
		if (this.model && this.collection) {
			Object.assign(data, {
				items: this.serializeCollection(),
			});
		}

		return data;
	},

	onAttach() {
		this.loadCaseList();
	},

	/**
	 * Method that gets called when view is first displayed,
	 * fetching the full list of cases for selection
	 */
	loadCaseList() {
		const list = new CaseCollection();

		// Render loading message in case request takes a while
		this.state.set("loadingList", true);

		return list.fetch(authHeaders())
			.then(() => {
				this.collection = list;
				this.state.set("loadingList", false);
			});
	},

	/**
	 * Method that gets called when a case is selected from
	 * dropdown list. Fetches model & displays information
	 */
	loadCase() {
		this.state.set("selectedCase", this.ui.input.val(), { silent: true });

		// If no case has been selected, remove reference to any
		// previously selected cases and return out -- no need to
		// fetch anything
		if (!this.state.get("selectedCase")) {
			this.model = undefined;
			return this.render();
		}

		const caseModel = new CaseModel({ ien: this.state.get("selectedCase") });

		// Render loading message in case request takes a while
		this.state.set("loadingCase", true);

		// Fetch case & re-render once complete
		return caseModel.fetch(authHeaders())
			.then(() => {
				this.model = caseModel;
				this.state.set("loadingCase", false);
			});
	},

	/**
	 * If case is selected & confirmed, store it for use & display elsewhere
	 */
	onConfirm() {
		Radio.request("store", "set", "activeCase", this.model);
		Radio.request("root", "close:modal");
	},
});
