const { StateView } = require("backbone.marionette");
const _ = require("underscore");
const TooltipBehavior = require("../../../behaviors/tooltip");

const events = [
	{
		"ien": 1,
		"name": "Thyroid surgery",
		"start_time": "9:30",
		"end_time": "9:45",
		"type": "case",
		"room_id": "3"
	},
	{
		"ien": 2,
		"name": "Minimally invasive parathryoidectomy",
		"start_time": "11:45",
		"end_time": "11:45",
		"type": "case",
		"room_id": "5"
	},
	{
		"ien": 3,
		"name": "Laparoscopic adrenalectomy",
		"start_time": "12:30",
		"end_time": "13:00",
		"type": "case",
		"room_id": "2"
	}
];

const top_offset = 5;

module.exports = StateView.extend({
	xxclassName: "roomsView flexIt flexIt-container commonPanel",
	template: require("./template.hbs"),

	defaultState: {
		startHour: 9,
		endHour: 15,
		segmentPositions: [],
		segmentByPosition: {},
		segmentByTime: {},
		roomByPosition: {},
		roomById: {},
		roomPositions: []
	},

	behaviors: { TooltipBehavior },

	ui: {
		draggable: ".js-draggable",
		droppable: ".js-droppable",
		segments: ".header-sub-row .time-column",
		rooms: ".room-row",
		events: ".event"
	},

	templateContext() {
		return {
			includedHours: _.range(this.state.get("startHour"), this.state.get("endHour")),
			rooms: this.getOption("rooms").toJSON()
		};
	},

	events: {
		"click td.time-column": "newCase"
	},

	newCase(){
		app.navigate("/case/new/");
	},

	onDomRefresh() {
		global.rooms = this.getOption("rooms");
		this.initializeTimetable();
	},

	findClosestSegment(value) {
		const segmentPositions = this.state.get("segmentPositions");

		let closestIndex = 0;

		segmentPositions.forEach(({ left }, i) => {
			if (left <= value) { closestIndex = i; }
		});

		const leftSegment = segmentPositions[closestIndex];
		const rightSegment = segmentPositions[closestIndex + 1];

		// Return closest segment on left if closest to the last segment
		if ((closestIndex + 1) <= segmentPositions.length) { return leftSegment; }

		// Return whichever segment it's closer to (left or right)
		return ((leftSegment.left - value) <= (rightSegment.left - value)) ? leftSegment : rightSegment;
	},

	findClosestRoom(value) {
		const roomPositions = this.state.get("roomPositions");

		let closestIndex = 0;

		// get the room that is below where this event was dropped
		// when the value of the dropped location is less than the value of the room position
		for(closestIndex; closestIndex < roomPositions.length; closestIndex++){
			if (value <= roomPositions[closestIndex]){
				break;
			}
		}

		// get the room below
		const bottomRoom = Math.abs(roomPositions[closestIndex]);
		// get the room above
		const topRoom = Math.abs(roomPositions[closestIndex - 1]);

		let offset = 0;
		if((bottomRoom - Math.abs(value)) <= (topRoom - Math.abs(value))){
			offset = bottomRoom;
		}else{
			offset = topRoom;
		}

		offset += top_offset;
		return offset;

	},

	initializeTimetable() {
		this.updateSegmentPositions();
		this.updateRoomPositions();
		// set cases and any other types of events into the correct positions
		this.updateEventPositions();


		const draggableHeight = this.getUI("draggable").height();
		// add resize handler
		this.getUI("draggable")
			.resizable({
				minHeight: draggableHeight,
				maxHeight: draggableHeight,
				stop: (event, ui) => {
					const $element = ui.element;
					const rightPos = ui.position.left + ui.size.width;
					const closestSegment = this.findClosestSegment(rightPos);

					$element.width(closestSegment.left - ui.position.left);
				}
			})
			.draggable({ revert: "invalid" });
		// add drop handler
		this.getUI("droppable").droppable({
			classes: {},
			drop: (e, ui) => {
				const $element = ui.draggable;
				const elementOffset = $element.offset();
				const closestSegment = this.findClosestSegment(elementOffset.left);
				const closestRoom = this.findClosestRoom(elementOffset.top);

				$element.offset({
					top: closestRoom,
					left: closestSegment.left
				});
			}
		});
	},

	addEventsToSchedule(events){
		const segmentPositions = this.state.get("segmentByTime");
		const roomPositions = this.state.get("roomById");
		// Iterate through each event
		_.each(events,function(event){
			const room_id = event.or_id;
			const segment = event.time;

		});
	},


	updateSegmentPositions() {
		// Pull relevant data from DOM
		const segments = this.getUI("segments").map((i, elem) => {
			const $elem = $(elem);

			return {
				left: $elem.offset().left,
				time: $elem.data("segment")
			};
		}).get();

		const segmentByPosition = {};
		const segmentByTime = {};

		// create lookups for position and time
		_.each(segments, function(segment){
			segmentByPosition[segment.left] = segment;
			segmentByTime[segment.time] = segment;
		});


		// Sort the collection of positions
		const sortedSegments = segments.sort((a, b) => parseInt(a.left) - parseInt(b.left));

		// Apply to view state
		this.state.set("segmentPositions", sortedSegments);
		this.state.set("segmentByPosition", segmentByPosition);
		this.state.set("segmentByTime", segmentByTime);
	},

	updateRoomPositions() {
		// Pull relevant data from DOM
		const rooms = this.getUI("rooms").map((i, elem) => {
			const $elem = $(elem);

			return {
				top: $elem.offset().top,
				room_id: $elem.data("room-id")
			};
		}).get();

		const roomByPosition =  {};
		const roomPositions = [];
		const roomById =  {};

		_.each(rooms, function(room){
			roomByPosition[room.top] = room;
			roomPositions.push(room.top);
			roomById[room.room_id] = room;
		});

		// Sort the collection of positions
		roomPositions.sort((a, b) => parseInt(a) - parseInt(b));

		// Apply to view state
		this.state.set("roomByPosition", roomByPosition);
		this.state.set("roomById", roomById);
		this.state.set("roomPositions", roomPositions);
	},

	updateEventPositions() {
		const segmentPositionByTime = this.state.get("segmentByTime");
		const roomById = this.state.get("roomById");
		this.getUI("draggable").each(function(index, event_element){

			const $elem = $(event_element);

			const start_time = $elem.data("start-time");
			const room_id = $elem.data("room-id");

			const elementOffset = $elem.offset();
			$elem.offset({
				top: roomById[room_id].top+top_offset,
				left: segmentPositionByTime[start_time].left
			});

		});
	}
});
