const Marionette		= require("backbone.marionette");
const Radio				= require("backbone.radio");
const authHeaders		= require("../../../utils/authHeaders");
const CaseCollection	= require("../../../entities/case/collection");
const CaseModel			= require("../../../entities/case/model");
const PatientModel		= require("../../../entities/patient/model");
const IndexView			= require("../../../modules/index/view");
const SaveCaseView		= require("../close_case/view");
const CancelCaseView	= require("../cancel_case/view");
const CloseCaseView		= require("../close_case/view");

module.exports = Marionette.StateView.extend({

	template: require("./template.hbs"),
	ui: {
		formxxx: "#cfForm",
		filterType: "#filterType option:selected",
		typeaheadField: "#filterData",
		titleAddition: "#titleAddition",
		formContainer: "#formContainer",
		patientCB: "#patientCB",
		placeholderCB: "#placeholderCB",
		rows: ".cf-li",
		tabs: ".tab-li"
	},

	defaultState: {
		caseScreenTitle: "Cases",
		titleAddition: "",
		toggleECClass: true,
		toggleTabPanelClass: true,
		showSubtitlePanel: true,
		showCaseFilterPanel: true,
		showResultsPanel: false,
		showTabsPanel: false,
		resultsObj: new Object(),
		expandedMode: true,
		filterTypeValue: "Specialty",
		filterDataValue: "",
		patientCBDisabled: false,
		patientCBChecked: false,
		placeholderCBDisabled: false,
		placeholderCBChecked: false,
		requestedChecked: false,
		scheduledChecked: false,
		showResultMessage: false,
		resultMessage: "",
		isNewCase: false,
		ien: null,
		tabPanel1Height: "",
		tabPanel2Height: "",
		tabPanel3Height: "",
		tabPanel4Height: "",
		tabPanel5Height: "",
		tabPanel6Height: "",

		caseTypeValue: "Patient",

		filterTypes: [{
			"option": {
				"value": "Specialty",
				"text": "Specialty" }
		}, {
			"option": {
				"value": "Provider",
				"text": "Provider" }
		}, {
			"option": {
				"value": "Staff Member",
				"text": "Staff Member" }
		},  {
			"option": {
				"value": "ORs",
				"text": "ORs" }
		}, {
			"option": {
				"value": "SSLQI",
				"text": "SSLQI" }
		}, {
			"option": {
				"value": "Reasons for Unavailability",
				"text": "Reasons for Unavailability" }
		}],

		caseTypes: [{
			"option": {
				"value": "Patient",
				"text": "Patient" }
		}, {
			"option": {
				"value": "Placeholder",
				"text": "Placeholder" }
		}, {
			"option": {
				"value": "Requested",
				"text": "Requested" }
		},  {
			"option": {
				"value": "Scheduled",
				"text": "Scheduled" }
		}, {
			"option": {
				"value": "Non OR",
				"text": "Non OR" }
		}]
	},

	stateEvents: {
		"change": "render",
	},

	events: {

		"click #expand-collapse": function(e) {

			let currentTab = this.getCurrentTab();

			this.state.set({
				showCaseFilterPanel: !this.state.get("showCaseFilterPanel"),
				toggleECClass: !this.state.get("toggleECClass"),
				toggleTabPanelClass: !this.state.get("toggleTabPanelClass"),
				expandedMode: !this.state.get("expandedMode")
			});

			// Reinitialize tab functionality
			this.reinitializeTabs();

			// Update tab panel height
			this.updateBodyPanelCSS();

			// Highlight the selected case a different color and reset the other case colors in the list
			this.highlightCurrentRow();

			// This is what allows the the tab panel to stay in the current tab
			this.returnToCurrentTab(currentTab);
		},

		"click #patientCB":  function (e) {

			let currentTab = this.getCurrentTab();

			this.state.set({
				patientCBChecked:  !this.state.get("patientCBChecked"),
				placeholderCBDisabled: !this.state.get("placeholderCBDisabled")
			});

			// Reinitialize tab functionality
			this.reinitializeTabs();

			// Highlight the selected case a different color and reset the other case colors in the list
			this.highlightCurrentRow();

			// This is what allows the the tab panel to stay in the current tab
			this.returnToCurrentTab(currentTab);
		},

		"click #placeholderCB":  function (e) {
			let currentTab = this.getCurrentTab();

			this.state.set({
				placeholderCBChecked:  !this.state.get("placeholderCBChecked"),
				patientCBDisabled: !this.state.get("patientCBDisabled")
			});

			// Reinitialize tab functionality
			this.reinitializeTabs();

			// Highlight the selected case a different color and reset the other case colors in the list
			this.highlightCurrentRow();

			// This is what allows the the tab panel to stay in the current tab
			this.returnToCurrentTab(currentTab);
		},

		"click #requestedCB":  function (e) {
			this.state.set("requestedChecked", !this.state.get("requestedChecked"));
		},

		"click #scheduledCB":  function (e) {
			this.state.set("scheduledChecked", !this.state.get("scheduledChecked"));
		},

		// TODO: Get rid of this JQuery
		"change #filterType":  function () {
			this.state.set("filterTypeValue", $("#filterType option:selected").val());
			console.log("Filter type value selected is: " + this.state.get("filterTypeValue"));
			this.ui.typeaheadField = "";
		},

		"click #cancel":  function (e) {
			Backbone.history.navigate( "/", { trigger: false, replace: true } );
			Radio.request("root", "body", new IndexView({}));
		},

		"click #clear":  function (e) {

			// Clear all fields
			Object.assign(this.options, {
				currentCase: new CaseModel(),
				activePatient: new PatientModel()
			});

			this.state.set({
				showResultMessage: false,
				showResultsPanel: false,
				showTabsPanel: false,
				titleAddition: "",
				filterTypeValue: "Specialty",
				filterDataValue: ""
			});

			// This would not clear if put in above set call as an empty string
			this.state.unset("filterDataValue");
		},

		"click #newCase":  function (e) {

			// Clear all fields
			Object.assign(this.options, {
				currentCase: new CaseModel(),
				activeCases: new CaseCollection(),
				activePatient: new PatientModel()
			});

			this.state.set({
				caseScreenTitle: "New Case",
				showSubtitlePanel: false,
				showCaseFilterPanel: false,
				showResultMessage: false,
				showResultsPanel: false,
				showTabsPanel: true,
				//toggleTabPanelClass: !this.state.get("toggleTabPanelClass"),
				expandedMode: !this.state.get("expandedMode"),
				isNewCase: true
			});

			// Reinitialize tab functionality
			this.reinitializeTabs();

			// Update tab panel height
			this.updateBodyPanelCSS();
		},

		"click #search": function(e) {
			let filterType = this.getUI("filterType").val();
			let filterData = this.getUI("typeaheadField").val();

			// Value gets cleared from field if this is not done
			this.state.set("filterDataValue", filterData);

			if (filterType == "" || filterData == "") {

				// Remove error div (if there), show error message, and hide other page components
				let message = "Please enter relevant data";

				this.state.set({
					resultMessage: message,
					showResultMessage: true,
					showResultsPanel: false,
					showTabsPanel: false
				});

			} else {

				this.state.set({
					showResultsPanel: true,
					showTabsPanel: false
				});

				let liResultsObj = new Object();

				// Loop through case list results and put them in object?
				$.each(hcJSONDataResponse, function(key, value) {
					//$.each(activeCases, function(key, value) {
					liResultsObj[value.ien] = value;
				});

				this.state.set("resultsObj", liResultsObj);

				// Remove error message (if there) and reset title
				this.state.set("showResultMessage", false);
				this.getUI("titleAddition").text("").removeClass("cf-title-addition");
			}
		},

		"click #list": function(e) {

			this.reinitializeTabs();

			// Get the current active tab
			let currentTab = this.getCurrentTab();

			// Set up a CaseModel object for the current case and put it into options
			let selectedCaseObj = this.state.get("resultsObj")[e.target.id];
			Object.assign(this.options, {
				currentCase: new CaseModel(selectedCaseObj)
			});

			// Show the actual tab panel below the case list and display the principal procedure near title
			this.state.set({
				showTabsPanel: true,
				titleAddition: `${e.target.id} - ${selectedCaseObj.principalProcedure}`,
				ien: e.target.id
			});

			// Update tab panel height
			this.updateBodyPanelCSS();

			// Highlight the selected case a different color and reset the other case colors in the list
			this.highlightCurrentRow();

			// This is what allows the the tab panel to stay in the current tab
			this.returnToCurrentTab(currentTab);
		},

		/*
		"click .tabs .tab-links a": function(e) {
			this.reinitializeTabsHB();
		},
		*/

		"click #saveCaseTab1, #saveCaseTab2, #saveCaseTab3, #saveCaseTab4, #saveCaseTab5, #saveCaseTab6":  function (e) {
			Radio.request("store", "set", "activeCase", this.currentCase);
			Radio.request("root", "modal", { view: new SaveCaseView() });
		},

		"click #closeCaseTab1, #closeCaseTab2, #closeCaseTab3, #closeCaseTab4, #closeCaseTab5, #closeCaseTab6":  function (e) {
			Radio.request("root", "modal", { view: new CloseCaseView() });
		},

		"click #cancelCaseTab1, #cancelCaseTab2, #cancelCaseTab3, #cancelCaseTab4, #cancelCaseTab5, #cancelCaseTab6":  function (e) {
			Radio.request("root", "modal", { view: new CancelCaseView() });
		},

		"click #cancelTab1, #cancelTab2, #cancelTab3, #cancelTab4, #cancelTab5, #cancelTab6":  function (e) {
			Backbone.history.navigate( "/", { trigger: false, replace: true } );
			Radio.request("root", "body", new IndexView({}));
		},
	},

	// 1: intialize; 2: onRender; 3: onAttach
	initialize: function() {

		// Needed when using StateView - calls its constructor
		Marionette.StateView.prototype.initialize.apply(this, arguments);

		// On window resize, call updateBodyPanelCSS function
		$(window).on("resize", this.updateBodyPanelCSS());

		// Collect relevant data from Radio channel
		Object.assign(this.options, {
			activePatient: Radio.request("store", "get", "activePatient"),
			activeCases: new CaseCollection(hcJSONDataResponse),
			currentCase: new CaseModel()
		});

		this.state.set({
			showResultsPanel: false,
			showTabsPanel: false
		});
	},

	//onRender: function() {
		//this.loadCaseList();
	//},

	onAttach: function() {
		this.ui.formxxx.show();

		this.listenTo(Radio.channel("store"), {
			"change:activePatient": this.updatePatient
		}, this);

		this.updateBodyPanelCSS();
		this.loadCaseList();
	},

	templateContext() {
		const activePatient = this.getOption("activePatient");
		const activeCases = this.getOption("activeCases");
		const currentCase = this.getOption("currentCase");

		console.log("Rendering from case_filter view:\n", {
			activePatient: activePatient ? activePatient.toJSON() : undefined,
			activeCases: activeCases ? activeCases.toJSON() : undefined,
			currentCase: currentCase ? currentCase.toJSON() : undefined
		});

		return {
			activePatient: activePatient ? activePatient.toJSON() : undefined,
			activeCases: activeCases ? activeCases.toJSON() : undefined,
			currentCase: currentCase ? currentCase.toJSON() : undefined
		};
	},

	updatePatient(activePatient) {

		if(activePatient != undefined) {

			let currentTab = this.getCurrentTab();

			Object.assign(this.options, {activePatient});

			this.state.set({
				showResultsPanel: false,
				showTabsPanel: false
			});

			this.state.unset("filterDataValue");

			// Reinitialize tab functionality
			this.reinitializeTabs();

			// This is what allows the the tab panel to stay in the current tab
			this.returnToCurrentTab(currentTab)
		}
	},

	// TODO: Change this to template context
	reinitializeTabs() {

		// Re-initiate dynamic tab functionality
		jQuery(document).ready(function() {
			jQuery(".tabs .tab-links a").on("click", function(e)  {
				var currentAttrValue = jQuery(this).attr("href");

				// Show/Hide Tabs
				jQuery(`.tabs ${currentAttrValue}`).show().siblings().hide();

				// Change/remove current tab to active
				jQuery(this).parent("li").addClass("active").siblings().removeClass("active");

				e.preventDefault();
			});
		});
	},

	reinitializeTabsHB() {

		// Re-initiate dynamic tab functionality
		let currentAttrValue = this.getUI(e.target.id).attr("href");

		// Show/Hide Tabs
		jQuery(`.tabs ${currentAttrValue}`).show().siblings().hide();

		// Change/remove current tab to active
		jQuery(this).parent("li").addClass("active").siblings().removeClass("active");

		e.preventDefault();
	},

	// TODO: Change this to template context
	highlightCurrentRow() {

		let currentIen = this.state.get("ien");
		this.getUI("rows").each(function() {
			if(this.id == currentIen) $(this).addClass("highlighted-li");
			else $(this).removeClass("highlighted-li");
		});
	},

	// TODO: Change this to template context
	getCurrentTab() {

		let currTab = "tabStub1";
		this.getUI("tabs").each(function() {
			if($(this).attr("class").indexOf("active") > -1)
				currTab = this.id;
		});
		return currTab;
	},

	// TODO: Change this to template context
	returnToCurrentTab(tab) {

		$("#" + tab).addClass("active").siblings().removeClass("active");
		let currentAttrValue = "#tab" + tab.substr(tab.length - 1);
		jQuery(`.tabs ${currentAttrValue}`).show().siblings().hide();
	},

	updateBodyPanelCSS() {

		let zoomLevel = null;
		let screenCssPixelRatio = (window.outerWidth - 8) / window.innerWidth;

		console.log("screenCssPixelRatio: " + screenCssPixelRatio);

		if (this.state.get("expandedMode")) {

			if (screenCssPixelRatio >= .46 && screenCssPixelRatio <= .51) {
				zoomLevel = "-5";
				this.state.set({
					tabPanel1Height: "",
					tabPanel2Height: "",
					tabPanel3Height: "",
					tabPanel4Height: "",
					tabPanel5Height: "",
					tabPanel6Height: ""
				});
			} else if (screenCssPixelRatio <= .68) {
				zoomLevel = "-4";
				this.state.set({
					tabPanel1Height: "",
					tabPanel2Height: "",
					tabPanel3Height: "height:546px;",
					tabPanel4Height: "",
					tabPanel5Height: "",
					tabPanel6Height: "height:546px;"
				});
			} else if (screenCssPixelRatio <= .76) {
				zoomLevel = "-3";
				this.state.set({
					tabPanel1Height: "",
					tabPanel2Height: "",
					tabPanel3Height: "height:432px;",
					tabPanel4Height: "",
					tabPanel5Height: "",
					tabPanel6Height: "height:432px;"
				});
			} else if (screenCssPixelRatio <= .81) {
				zoomLevel = "-2";
				this.state.set({
					tabPanel1Height: "",
					tabPanel2Height: "",
					tabPanel3Height: "height:373px;",
					tabPanel4Height: "",
					tabPanel5Height: "",
					tabPanel6Height: "height:373px;"
				});
			} else if (screenCssPixelRatio <= .91) {
				zoomLevel = "-1";
				this.state.set({
					tabPanel1Height: "",
					tabPanel2Height: "",
					tabPanel3Height: "height:284px;",
					tabPanel4Height: "height:284px;",
					tabPanel5Height: "height:284px;",
					tabPanel6Height: "height:284px;"
				});
			} else if (screenCssPixelRatio <= 1.01) {
				zoomLevel = "0";
				this.state.set({
					tabPanel1Height: "",
					tabPanel2Height: "",
					tabPanel3Height: "height:373px;",
					tabPanel4Height: "height:373px;",
					tabPanel5Height: "height:373px;",
					tabPanel6Height: "height:373px;"
				});
			} else if (screenCssPixelRatio <= 1.11) {
				zoomLevel = "1";
				this.state.set({
					tabPanel1Height: "",
					tabPanel2Height: "",
					tabPanel3Height: "height:373px;",
					tabPanel4Height: "height:373px;",
					tabPanel5Height: "height:373px;",
					tabPanel6Height: "height:373px;"
				});
			} else if (screenCssPixelRatio <= 1.26) {
				zoomLevel = "2";
				this.state.set({
					tabPanel1Height: "",
					tabPanel2Height: "",
					tabPanel3Height: "height:373px;",
					tabPanel4Height: "height:373px;",
					tabPanel5Height: "height:373px;",
					tabPanel6Height: "height:373px;"
				});
			} else if (screenCssPixelRatio <= 1.51) {
				zoomLevel = "3";
				this.state.set({
					tabPanel1Height: "",
					tabPanel2Height: "",
					tabPanel3Height: "height:373px;",
					tabPanel4Height: "height:373px;",
					tabPanel5Height: "height:373px;",
					tabPanel6Height: "height:373px;"
				});
			} else {
				zoomLevel = "unknown";
			}

		} else {

			if (screenCssPixelRatio >= .46 && screenCssPixelRatio <= .81) {
				zoomLevel = "-5";
				this.state.set({
					tabPanel1Height: "",
					tabPanel2Height: "",
					tabPanel3Height: "height:650px",
					tabPanel4Height: "",
					tabPanel5Height: "",
					tabPanel6Height: ""
				});
			} else if (screenCssPixelRatio <= .91) {
				zoomLevel = "-1";
				this.state.set({
					tabPanel1Height: "",
					tabPanel2Height: "",
					tabPanel3Height: "height:288px;",
					tabPanel4Height: "",
					tabPanel5Height: "",
					tabPanel6Height: ""
				});
			} else if (screenCssPixelRatio <= 1.01) {
				zoomLevel = "0";
				this.state.set({
					tabPanel1Height: "",
					tabPanel2Height: "",
					tabPanel3Height: "height:288px;",
					tabPanel4Height: "",
					tabPanel5Height: "",
					tabPanel6Height: "height:288px;"
				});
			} else if (screenCssPixelRatio <= 1.11) {
				zoomLevel = "1";
				this.state.set({
					tabPanel1Height: "",
					tabPanel2Height: "",
					tabPanel3Height: "height:288px;",
					tabPanel4Height: "",
					tabPanel5Height: "",
					tabPanel6Height: "height:288px;"
				});
			} else if (screenCssPixelRatio <= 1.26) {
				zoomLevel = "2";
				this.state.set({
					tabPanel1Height: "",
					tabPanel2Height: "",
					tabPanel3Height: "height:288px;",
					tabPanel4Height: "",
					tabPanel5Height: "",
					tabPanel6Height: "height:288px;"
				});
			} else if (screenCssPixelRatio <= 1.51) {
				zoomLevel = "3";
				this.state.set({
					tabPanel1Height: "",
					tabPanel2Height: "",
					tabPanel3Height: "height:288px;",
					tabPanel4Height: "height:288px;",
					tabPanel5Height: "height:288px;",
					tabPanel6Height: "height:288px;"
				});
			} else {
				zoomLevel = "unknown";
			}
		}
	},

	/**
	 * Method that gets called when view is first displayed,
	 * fetching the full list of cases for selection
	 */
	loadCaseList() {
		this.collection = new CaseCollection();
		return this.collection.fetch(authHeaders())
			.then(() => this.initTypeahead());
	},

	initTypeahead() {

		let activeCases1 = this.collection;

		// Debug
		//console.log(JSON.stringify(activeCases1));

		// Set the data for the various filter types since this data is all null inexplicably

		// Specialty
		activeCases1.models[0].set("surgerySpecialty", "Neurological Surgery");
		activeCases1.models[1].set("surgerySpecialty", "Ophthalmic Surgery");
		activeCases1.models[2].set("surgerySpecialty", "Orthopaedic Surgery");
		activeCases1.models[3].set("surgerySpecialty", "Thoracic Surgery");

		// Provider
		activeCases1.models[0].set("attendingProvider", "Anatoly Vekrenov - Provider");
		activeCases1.models[1].set("attendingProvider", "Bernard Barley - Provider");
		activeCases1.models[2].set("attendingProvider", "David Smith - Provider");
		activeCases1.models[3].set("attendingProvider", "Zlatan Moncho - Provider");

		// Staff Member
		activeCases1.models[0].set("primarySurgeon", "Otto Kint - PS");
		activeCases1.models[1].set("primarySurgeon", "Olajide Ochefu - PS");
		activeCases1.models[2].set("primarySurgeon", "Pinar Yilmaz - PS");
		activeCases1.models[3].set("primarySurgeon", "Wilson Martinez - PS");

		activeCases1.models[0].set("attendingSurgeon", "Adam Karbacher - AS");
		activeCases1.models[1].set("attendingSurgeon", "Carlos Litbarski - AS");
		activeCases1.models[2].set("attendingSurgeon", "Gunther Thomas - AS");
		activeCases1.models[3].set("attendingSurgeon", "William Horst - AS");

		activeCases1.models[0].set("firstAsst", "Adan Swackhammer - FA");
		activeCases1.models[1].set("firstAsst", "Will Sikahema - FA");
		activeCases1.models[2].set("firstAsst", "Julio Velasco - FA");
		activeCases1.models[3].set("firstAsst", "Amanda Wilhite - FA");

		activeCases1.models[0].set("secondAsst", "William Wells - SA");
		activeCases1.models[1].set("secondAsst", "Ryan Flanagan - SA");
		activeCases1.models[2].set("secondAsst", "July Brown - SA");
		activeCases1.models[3].set("secondAsst", "Fulgencio Soto - SA");

		// ORs
		activeCases1.models[0].set("orRoom", "EC-2241");
		activeCases1.models[1].set("orRoom", "G-211");
		activeCases1.models[2].set("orRoom", "OC-3254A");
		activeCases1.models[3].set("orRoom", "RAD-325B");

		// SSLQI
		// Reasons for Unavailability

		let typeaheadQuery = null;
		let filterTypeSelected = this.state.get("filterTypeValue");

		this.getUI("typeaheadField").typeahead({
				minLength: 3,
				highlight: true
			},
			{
				name: "active-case-data",
				source(query, callback) {

					// The typeahead query will have to be known globally
					typeaheadQuery = query;

					// Filters out CaseModel objects which match the query above
					const results = activeCases1.filter((activeCase) => {

						let ck1 = null;
						let ck2 = null;
						let ck3 = null;
						let ck4 = null;

						// This code block returns either true or false
						if (filterTypeSelected == "Specialty") {
							ck1 = activeCase.get("surgerySpecialty").toLowerCase().indexOf(query.toLowerCase()) > -1;
							return ck1;
						} else if (filterTypeSelected == "Provider") {
							ck1 = activeCase.get("attendingProvider").toLowerCase().indexOf(query.toLowerCase()) > -1;
							return ck1;
						} else if(filterTypeSelected == "Staff Member") {
							ck1 = activeCase.get("primarySurgeon").toLowerCase().indexOf(query.toLowerCase()) > -1;
							ck2 = activeCase.get("attendingSurgeon").toLowerCase().indexOf(query.toLowerCase()) > -1;
							ck3 = activeCase.get("firstAsst").toLowerCase().indexOf(query.toLowerCase()) > -1;
							ck4 = activeCase.get("secondAsst").toLowerCase().indexOf(query.toLowerCase()) > -1;
							return ck1 | ck2 | ck3 | ck4;
						} else if(filterTypeSelected == "ORs") {
							ck1 = activeCase.get("orRoom").toLowerCase().indexOf(query.toLowerCase()) > -1;
							return ck1;
						}

					});

					let theMap = null;

					// Debug
					//console.log("Matching active cases", results);

					// Certain filter criteria require searching through more than one data item inside each case model object
					// filtered through to results above. When that happens, as with Staff Member, each data item searched for is
					// put into an temporary array, and then all the temporary arrays are lumped into one. Then all empty elements
					// in the consolidated array are gotten rid of. For filter criteria with one case model data item, its
					// processing is simpler.
					if(filterTypeSelected == "Specialty")
						theMap = results.map(r => (r.get("surgerySpecialty")));
					else if(filterTypeSelected == "Provider")
						theMap = results.map(r => (r.get("attendingProvider")));
					else if(filterTypeSelected == "Staff Member") {

						// Need to determine which one of the filter criteria actually contains the query
						let tempMap1 = results.map(r => ( (r.get("primarySurgeon").toLowerCase().indexOf(typeaheadQuery.toLowerCase()) > -1) ? r.get("primarySurgeon") : "") );
						let tempMap2 = results.map(r => ( (r.get("attendingSurgeon").toLowerCase().indexOf(typeaheadQuery.toLowerCase()) > -1) ? r.get("attendingSurgeon") : "") );
						let tempMap3 = results.map(r => ( (r.get("firstAsst").toLowerCase().indexOf(typeaheadQuery.toLowerCase()) > -1) ? r.get("firstAsst") : "") );
						let tempMap4 = results.map(r => ( (r.get("secondAsst").toLowerCase().indexOf(typeaheadQuery.toLowerCase()) > -1) ? r.get("secondAsst") : "") );
						let tempMap = [tempMap1, tempMap2, tempMap3, tempMap4];
						theMap = [].concat.apply([], tempMap);

						// Get rid of empty array elements
						theMap = jQuery.grep(theMap, function(n, i){
							return (n !== "" && n != null);
						});

					} else if(filterTypeSelected == "ORs")
						theMap = results.map(r => (r.get("orRoom")));

					callback(theMap);
				}
			});
	}
});

let hcJSONDataResponse = jQuery.parseJSON(`[
{
  "ien": "15",
  "caseCreateDate": "2016-09-14T05:00:00.000Z",
  "caseDateTime": "2016-10-14T05:00:00.000Z",
  "cancelDate": "2016-11-14T05:00:00.000Z",
  "patient": "Todd Lowry",
  "hospitalAdmissionStatus": "Admitted",
  "serviceConnected": "Hospital service",
  "generalComments": "Not SPD comments?",
  "otherProcedures": "Liver biopsy",
  "plannedPostopCare": "Physical Therapy",
  "primaryCancelReason": "N/A",
  "principalProcedure": "Amygdalohippocampectomy",
  "plannedPrincOpProcCPT": "AM1234",
  "procedureLaterality": "Additional OR preparation",
  "associatedClinic": "Santa Fe Clinic",
  "resSupCode": "RS8765",
  "anesCareTimeBlock": "4 hours",
  "princAnesthetist": "Scott Thurman",
  "anesthesiologistSupvr": "Paul Coehlo",
  "anesSuperviseCode": "AN0987",
  "anesthesiaTechnique": "Intravenous",
  "otherPostopDiags": "Fatty liver",
  "asaClass": "Class A",
  "diagnosticTherapeutic": "Therapeutic 107",
  "principalDiagnosis": "Amigdalitis",
  "specimens": "Throat culture",
  "operationIndications": "None specified",
  "operativeFindings": "None specified",
  "briefClinHistory": "5 previous incidents of severe inflamation",
  "plannedPrinDiagCode": "PPD463456",
  "location": "VAMS Surgery Center - Spokane",
  "procedureDate": "2016-10-24T05:00:00.000Z",
  "procedureStartTime": "08:00",
  "procedureEndTime": "11:00",
  "provider": "Cigna",
  "attendingProvider": "Stoppel and Associates",
  "medicalSpecialty": "Throat surgeon",
  "procedureOccurence": "Once",
  "dictatedSummary": "None specified",
  "orRoom": "OR230-5",
  "caseScheduleType": "Ledgered",
  "caseScheduleOrder": "CSO-5325432",
  "surgerySpecialty": "Ophthalmic Surgery",
  "preOpInfection": "None",
  "primarySurgeon": "Pinar Yilmaz",
  "firstAsst": "Robert Valdellon",
  "secondAsst": "Allison Welker",
  "attendingSurgeon": "Nathan Thorne",
  "otherPreopDiagnosis": "None",
  "reqAneTechnique": "Intravenous",
  "reqFrozSect": "None",
  "reqPreopXray": "No",
  "intraoperativeXrays": "Yes",
  "reqPhoto": "Yes",
  "reqBloodKind": "Yes",
  "principalPreOpDiagnosis": "Amigdalitis",
  "prinPreOpDiagICD": "AM346546",
  "requestBloodAvailability": "Yes",
  "bloodProductInfo": "Specified",
  "preAdmissionTesting": "Throat culture analysis",
  "surgeryPosition": "None specified",
  "principalPreOpICD": "PPO-ICD-32452453",
  "palliation": "Yes",
  "specialEquipment": "N/A",
  "plannedProsthImplant": "No",
  "specialSupplies": "None",
  "specialInstruments": "None needed",
  "pharmacyItems": "100 mg Vicadun",
  "plannedAdmissionStatus": "Set",
  "referringPhysician": "Leonard Capriatti"
},
{
  "ien": "12",
  "caseCreateDate": null,
  "caseDateTime": "2016-11-11T14:05:00.000Z",
  "cancelDate": null,
  "patient": null,
  "hospitalAdmissionStatus": null,
  "serviceConnected": null,
  "generalComments": null,
  "otherProcedures": null,
  "plannedPostopCare": null,
  "primaryCancelReason": null,
  "principalProcedure": "Otoplasty",
  "plannedPrincOpProcCPT": "EA887564",
  "procedureLaterality": null,
  "associatedClinic": null,
  "resSupCode": null,
  "anesCareTimeBlock": null,
  "princAnesthetist": null,
  "anesthesiologistSupvr": null,
  "anesSuperviseCode": null,
  "anesthesiaTechnique": null,
  "otherPostopDiags": null,
  "asaClass": null,
  "diagnosticTherapeutic": null,
  "principalDiagnosis": null,
  "specimens": null,
  "operationIndications": null,
  "operativeFindings": null,
  "briefClinHistory": null,
  "plannedPrinDiagCode": null,
  "location": null,
  "procedureDate": null,
  "procedureStartTime": null,
  "procedureEndTime": null,
  "provider": null,
  "attendingProvider": null,
  "medicalSpecialty": null,
  "procedureOccurence": null,
  "dictatedSummary": null,
  "orRoom": null,
  "caseScheduleType": null,
  "caseScheduleOrder": null,
  "surgerySpecialty": "Orthopaedic Surgery",
  "preOpInfection": null,
  "primarySurgeon": "Otto Kint",
  "firstAsst": null,
  "secondAsst": null,
  "attendingSurgeon": "Valerio Vega",
  "otherPreopDiagnosis": null,
  "reqAneTechnique": null,
  "reqFrozSect": null,
  "reqPreopXray": null,
  "intraoperativeXrays": null,
  "reqPhoto": null,
  "reqBloodKind": null,
  "principalPreOpDiagnosis": null,
  "prinPreOpDiagICD": null,
  "requestBloodAvailability": null,
  "bloodProductInfo": null,
  "preAdmissionTesting": null,
  "surgeryPosition": "Senior ER Staff2",
  "principalPreOpICD": null,
  "palliation": null,
  "specialEquipment": "None Needed",
  "plannedProsthImplant": null,
  "specialSupplies": null,
  "specialInstruments": null,
  "pharmacyItems": null,
  "plannedAdmissionStatus": null,
  "referringPhysician": null
},
{
  "ien": "48",
  "caseCreateDate": null,
  "caseDateTime": "2016-11-11T14:05:00.000Z",
  "cancelDate": null,
  "patient": null,
  "hospitalAdmissionStatus": null,
  "serviceConnected": null,
  "generalComments": null,
  "otherProcedures": null,
  "plannedPostopCare": null,
  "primaryCancelReason": null,
  "principalProcedure": "Rhinoplasty",
  "plannedPrincOpProcCPT": "RP32452",
  "procedureLaterality": null,
  "associatedClinic": null,
  "resSupCode": null,
  "anesCareTimeBlock": null,
  "princAnesthetist": null,
  "anesthesiologistSupvr": null,
  "anesSuperviseCode": null,
  "anesthesiaTechnique": null,
  "otherPostopDiags": null,
  "asaClass": null,
  "diagnosticTherapeutic": null,
  "principalDiagnosis": null,
  "specimens": null,
  "operationIndications": null,
  "operativeFindings": null,
  "briefClinHistory": null,
  "plannedPrinDiagCode": null,
  "location": null,
  "procedureDate": null,
  "procedureStartTime": null,
  "procedureEndTime": null,
  "provider": null,
  "attendingProvider": null,
  "medicalSpecialty": null,
  "procedureOccurence": null,
  "dictatedSummary": null,
  "orRoom": null,
  "caseScheduleType": null,
  "caseScheduleOrder": null,
  "surgerySpecialty": "Neurological Surgery",
  "preOpInfection": null,
  "primarySurgeon": "Olajide Ochefu",
  "firstAsst": null,
  "secondAsst": null,
  "attendingSurgeon": "Wilson Martinez",
  "otherPreopDiagnosis": null,
  "reqAneTechnique": null,
  "reqFrozSect": null,
  "reqPreopXray": null,
  "intraoperativeXrays": null,
  "reqPhoto": null,
  "reqBloodKind": null,
  "principalPreOpDiagnosis": null,
  "prinPreOpDiagICD": null,
  "requestBloodAvailability": null,
  "bloodProductInfo": null,
  "preAdmissionTesting": null,
  "surgeryPosition": "Senior ER Staff3",
  "principalPreOpICD": null,
  "palliation": null,
  "specialEquipment": "Pulse Oximeter",
  "plannedProsthImplant": null,
  "specialSupplies": null,
  "specialInstruments": null,
  "pharmacyItems": null,
  "plannedAdmissionStatus": null,
  "referringPhysician": null
},
{
  "ien": "47",
  "caseCreateDate": null,
  "caseDateTime": "2016-11-11T14:05:00.000Z",
  "cancelDate": null,
  "patient": null,
  "hospitalAdmissionStatus": null,
  "serviceConnected": null,
  "generalComments": null,
  "otherProcedures": null,
  "plannedPostopCare": null,
  "primaryCancelReason": null,
  "principalProcedure": "Tonsillectomy",
  "plannedPrincOpProcCPT": "TS353245",
  "procedureLaterality": null,
  "associatedClinic": null,
  "resSupCode": null,
  "anesCareTimeBlock": null,
  "princAnesthetist": null,
  "anesthesiologistSupvr": null,
  "anesSuperviseCode": null,
  "anesthesiaTechnique": null,
  "otherPostopDiags": null,
  "asaClass": null,
  "diagnosticTherapeutic": null,
  "principalDiagnosis": null,
  "specimens": null,
  "operationIndications": null,
  "operativeFindings": null,
  "briefClinHistory": null,
  "plannedPrinDiagCode": null,
  "location": null,
  "procedureDate": null,
  "procedureStartTime": null,
  "procedureEndTime": null,
  "provider": null,
  "attendingProvider": null,
  "medicalSpecialty": null,
  "procedureOccurence": null,
  "dictatedSummary": null,
  "orRoom": null,
  "caseScheduleType": null,
  "caseScheduleOrder": null,
  "surgerySpecialty": "Thoracic Surgery",
  "preOpInfection": null,
  "primarySurgeon": "Srikanth Krishnavarapu",
  "firstAsst": null,
  "secondAsst": null,
  "attendingSurgeon": "Grace Kim",
  "otherPreopDiagnosis": null,
  "reqAneTechnique": null,
  "reqFrozSect": null,
  "reqPreopXray": null,
  "intraoperativeXrays": null,
  "reqPhoto": null,
  "reqBloodKind": null,
  "principalPreOpDiagnosis": null,
  "prinPreOpDiagICD": null,
  "requestBloodAvailability": null,
  "bloodProductInfo": null,
  "preAdmissionTesting": null,
  "surgeryPosition": "Senior ER Staff4",
  "principalPreOpICD": null,
  "palliation": null,
  "specialEquipment": "Doppler",
  "plannedProsthImplant": null,
  "specialSupplies": null,
  "specialInstruments": null,
  "pharmacyItems": null,
  "plannedAdmissionStatus": null,
  "referringPhysician": null
},
 {
 "ien": "75",
 "caseCreateDate": "2016-09-14T05:00:00.000Z",
 "caseDateTime": "2016-10-14T05:00:00.000Z",
 "cancelDate": "2016-11-14T05:00:00.000Z",
 "patient": "Todd Lowry",
 "hospitalAdmissionStatus": "Admitted",
 "serviceConnected": "Hospital service",
 "generalComments": "Not SPD comments?",
 "otherProcedures": "Liver biopsy",
 "plannedPostopCare": "Physical Therapy",
 "primaryCancelReason": "N/A",
 "principalProcedure": "Ventriculostomy",
 "plannedPrincOpProcCPT": "AM1234",
 "procedureLaterality": "Additional OR preparation",
 "associatedClinic": "Santa Fe Clinic",
 "resSupCode": "RS8765",
 "anesCareTimeBlock": "4 hours",
 "princAnesthetist": "Scott Thurman",
 "anesthesiologistSupvr": "Paul Coehlo",
 "anesSuperviseCode": "AN0987",
 "anesthesiaTechnique": "Intravenous",
 "otherPostopDiags": "Fatty liver",
 "asaClass": "Class A",
 "diagnosticTherapeutic": "Therapeutic 107",
 "principalDiagnosis": "Amigdalitis",
 "specimens": "Throat culture",
 "operationIndications": "None specified",
 "operativeFindings": "None specified",
 "briefClinHistory": "5 previous incidents of severe inflamation",
 "plannedPrinDiagCode": "PPD463456",
 "location": "VAMS Surgery Center - Spokane",
 "procedureDate": "2016-10-24T05:00:00.000Z",
 "procedureStartTime": "08:00",
 "procedureEndTime": "11:00",
 "provider": "Cigna",
 "attendingProvider": "Stoppel and Associates",
 "medicalSpecialty": "Throat surgeon",
 "procedureOccurence": "Once",
 "dictatedSummary": "None specified",
 "orRoom": "OR230-5",
 "caseScheduleType": "Ledgered",
 "caseScheduleOrder": "CSO-5325432",
 "surgerySpecialty": "Obstetrics and Gynecology",
 "preOpInfection": "None",
 "primarySurgeon": "Selma Dyer",
 "firstAsst": "Robert Valdellon",
 "secondAsst": "Allison Welker",
 "attendingSurgeon": "Nathan Thorne",
 "otherPreopDiagnosis": "None",
 "reqAneTechnique": "Intravenous",
 "reqFrozSect": "None",
 "reqPreopXray": "No",
 "intraoperativeXrays": "Yes",
 "reqPhoto": "Yes",
 "reqBloodKind": "Yes",
 "principalPreOpDiagnosis": "Amigdalitis",
 "prinPreOpDiagICD": "AM346546",
 "requestBloodAvailability": "Yes",
 "bloodProductInfo": "Specified",
 "preAdmissionTesting": "Throat culture analysis",
 "surgeryPosition": "None specified",
 "principalPreOpICD": "PPO-ICD-32452453",
 "palliation": "Yes",
 "specialEquipment": "N/A",
 "plannedProsthImplant": "No",
 "specialSupplies": "None",
 "specialInstruments": "None needed",
 "pharmacyItems": "100 mg Vicadun",
 "plannedAdmissionStatus": "Set",
 "referringPhysician": "Leonard Capriatti"
 },
 {
 "ien": "99",
 "caseCreateDate": "2016-09-14T05:00:00.000Z",
 "caseDateTime": "2016-10-14T05:00:00.000Z",
 "cancelDate": "2016-11-14T05:00:00.000Z",
 "patient": "Todd Lowry",
 "hospitalAdmissionStatus": "Admitted",
 "serviceConnected": "Hospital service",
 "generalComments": "Not SPD comments?",
 "otherProcedures": "Liver biopsy",
 "plannedPostopCare": "Physical Therapy",
 "primaryCancelReason": "N/A",
 "principalProcedure": "Corneal transplantation",
 "plannedPrincOpProcCPT": "AM1234",
 "procedureLaterality": "Additional OR preparation",
 "associatedClinic": "Santa Fe Clinic",
 "resSupCode": "RS8765",
 "anesCareTimeBlock": "4 hours",
 "princAnesthetist": "Scott Thurman",
 "anesthesiologistSupvr": "Paul Coehlo",
 "anesSuperviseCode": "AN0987",
 "anesthesiaTechnique": "Intravenous",
 "otherPostopDiags": "Fatty liver",
 "asaClass": "Class A",
 "diagnosticTherapeutic": "Therapeutic 107",
 "principalDiagnosis": "Amigdalitis",
 "specimens": "Throat culture",
 "operationIndications": "None specified",
 "operativeFindings": "None specified",
 "briefClinHistory": "5 previous incidents of severe inflamation",
 "plannedPrinDiagCode": "PPD463456",
 "location": "VAMS Surgery Center - Spokane",
 "procedureDate": "2016-10-24T05:00:00.000Z",
 "procedureStartTime": "08:00",
 "procedureEndTime": "11:00",
 "provider": "Cigna",
 "attendingProvider": "Stoppel and Associates",
 "medicalSpecialty": "Throat surgeon",
 "procedureOccurence": "Once",
 "dictatedSummary": "None specified",
 "orRoom": "OR230-5",
 "caseScheduleType": "Ledgered",
 "caseScheduleOrder": "CSO-5325432",
 "surgerySpecialty": "Oral and Maxillofacial Surgery",
 "preOpInfection": "None",
 "primarySurgeon": "Selma Dyer",
 "firstAsst": "Robert Valdellon",
 "secondAsst": "Allison Welker",
 "attendingSurgeon": "Nathan Thorne",
 "otherPreopDiagnosis": "None",
 "reqAneTechnique": "Intravenous",
 "reqFrozSect": "None",
 "reqPreopXray": "No",
 "intraoperativeXrays": "Yes",
 "reqPhoto": "Yes",
 "reqBloodKind": "Yes",
 "principalPreOpDiagnosis": "Amigdalitis",
 "prinPreOpDiagICD": "AM346546",
 "requestBloodAvailability": "Yes",
 "bloodProductInfo": "Specified",
 "preAdmissionTesting": "Throat culture analysis",
 "surgeryPosition": "None specified",
 "principalPreOpICD": "PPO-ICD-32452453",
 "palliation": "Yes",
 "specialEquipment": "N/A",
 "plannedProsthImplant": "No",
 "specialSupplies": "None",
 "specialInstruments": "None needed",
 "pharmacyItems": "100 mg Vicadun",
 "plannedAdmissionStatus": "Set",
 "referringPhysician": "Leonard Capriatti"
 }
]`);

/*
 {
 "ien": "75",
 "caseCreateDate": "2016-09-14T05:00:00.000Z",
 "caseDateTime": "2016-10-14T05:00:00.000Z",
 "cancelDate": "2016-11-14T05:00:00.000Z",
 "patient": "Todd Lowry",
 "hospitalAdmissionStatus": "Admitted",
 "serviceConnected": "Hospital service",
 "generalComments": "Not SPD comments?",
 "otherProcedures": "Liver biopsy",
 "plannedPostopCare": "Physical Therapy",
 "primaryCancelReason": "N/A",
 "principalProcedure": "Ventriculostomy",
 "plannedPrincOpProcCPT": "AM1234",
 "procedureLaterality": "Additional OR preparation",
 "associatedClinic": "Santa Fe Clinic",
 "resSupCode": "RS8765",
 "anesCareTimeBlock": "4 hours",
 "princAnesthetist": "Scott Thurman",
 "anesthesiologistSupvr": "Paul Coehlo",
 "anesSuperviseCode": "AN0987",
 "anesthesiaTechnique": "Intravenous",
 "otherPostopDiags": "Fatty liver",
 "asaClass": "Class A",
 "diagnosticTherapeutic": "Therapeutic 107",
 "principalDiagnosis": "Amigdalitis",
 "specimens": "Throat culture",
 "operationIndications": "None specified",
 "operativeFindings": "None specified",
 "briefClinHistory": "5 previous incidents of severe inflamation",
 "plannedPrinDiagCode": "PPD463456",
 "location": "VAMS Surgery Center - Spokane",
 "procedureDate": "2016-10-24T05:00:00.000Z",
 "procedureStartTime": "08:00",
 "procedureEndTime": "11:00",
 "provider": "Cigna",
 "attendingProvider": "Stoppel and Associates",
 "medicalSpecialty": "Throat surgeon",
 "procedureOccurence": "Once",
 "dictatedSummary": "None specified",
 "orRoom": "OR230-5",
 "caseScheduleType": "Ledgered",
 "caseScheduleOrder": "CSO-5325432",
 "surgerySpecialty": "ENT",
 "preOpInfection": "None",
 "primarySurgeon": "Selma Dyer",
 "firstAsst": "Robert Valdellon",
 "secondAsst": "Allison Welker",
 "attendingSurgeon": "Nathan Thorne",
 "otherPreopDiagnosis": "None",
 "reqAneTechnique": "Intravenous",
 "reqFrozSect": "None",
 "reqPreopXray": "No",
 "intraoperativeXrays": "Yes",
 "reqPhoto": "Yes",
 "reqBloodKind": "Yes",
 "principalPreOpDiagnosis": "Amigdalitis",
 "prinPreOpDiagICD": "AM346546",
 "requestBloodAvailability": "Yes",
 "bloodProductInfo": "Specified",
 "preAdmissionTesting": "Throat culture analysis",
 "surgeryPosition": "None specified",
 "principalPreOpICD": "PPO-ICD-32452453",
 "palliation": "Yes",
 "specialEquipment": "N/A",
 "plannedProsthImplant": "No",
 "specialSupplies": "None",
 "specialInstruments": "None needed",
 "pharmacyItems": "100 mg Vicadun",
 "plannedAdmissionStatus": "Set",
 "referringPhysician": "Leonard Capriatti"
 },
 {
 "ien": "99",
 "caseCreateDate": "2016-09-14T05:00:00.000Z",
 "caseDateTime": "2016-10-14T05:00:00.000Z",
 "cancelDate": "2016-11-14T05:00:00.000Z",
 "patient": "Todd Lowry",
 "hospitalAdmissionStatus": "Admitted",
 "serviceConnected": "Hospital service",
 "generalComments": "Not SPD comments?",
 "otherProcedures": "Liver biopsy",
 "plannedPostopCare": "Physical Therapy",
 "primaryCancelReason": "N/A",
 "principalProcedure": "Corneal transplantation",
 "plannedPrincOpProcCPT": "AM1234",
 "procedureLaterality": "Additional OR preparation",
 "associatedClinic": "Santa Fe Clinic",
 "resSupCode": "RS8765",
 "anesCareTimeBlock": "4 hours",
 "princAnesthetist": "Scott Thurman",
 "anesthesiologistSupvr": "Paul Coehlo",
 "anesSuperviseCode": "AN0987",
 "anesthesiaTechnique": "Intravenous",
 "otherPostopDiags": "Fatty liver",
 "asaClass": "Class A",
 "diagnosticTherapeutic": "Therapeutic 107",
 "principalDiagnosis": "Amigdalitis",
 "specimens": "Throat culture",
 "operationIndications": "None specified",
 "operativeFindings": "None specified",
 "briefClinHistory": "5 previous incidents of severe inflamation",
 "plannedPrinDiagCode": "PPD463456",
 "location": "VAMS Surgery Center - Spokane",
 "procedureDate": "2016-10-24T05:00:00.000Z",
 "procedureStartTime": "08:00",
 "procedureEndTime": "11:00",
 "provider": "Cigna",
 "attendingProvider": "Stoppel and Associates",
 "medicalSpecialty": "Throat surgeon",
 "procedureOccurence": "Once",
 "dictatedSummary": "None specified",
 "orRoom": "OR230-5",
 "caseScheduleType": "Ledgered",
 "caseScheduleOrder": "CSO-5325432",
 "surgerySpecialty": "ENT",
 "preOpInfection": "None",
 "primarySurgeon": "Selma Dyer",
 "firstAsst": "Robert Valdellon",
 "secondAsst": "Allison Welker",
 "attendingSurgeon": "Nathan Thorne",
 "otherPreopDiagnosis": "None",
 "reqAneTechnique": "Intravenous",
 "reqFrozSect": "None",
 "reqPreopXray": "No",
 "intraoperativeXrays": "Yes",
 "reqPhoto": "Yes",
 "reqBloodKind": "Yes",
 "principalPreOpDiagnosis": "Amigdalitis",
 "prinPreOpDiagICD": "AM346546",
 "requestBloodAvailability": "Yes",
 "bloodProductInfo": "Specified",
 "preAdmissionTesting": "Throat culture analysis",
 "surgeryPosition": "None specified",
 "principalPreOpICD": "PPO-ICD-32452453",
 "palliation": "Yes",
 "specialEquipment": "N/A",
 "plannedProsthImplant": "No",
 "specialSupplies": "None",
 "specialInstruments": "None needed",
 "pharmacyItems": "100 mg Vicadun",
 "plannedAdmissionStatus": "Set",
 "referringPhysician": "Leonard Capriatti"
 }
 */
