"use strict";
const config = require("../../../env.json");
const $ = require("jquery");
const Marionette = require("backbone.marionette");
const Radio = require("backbone.radio");
import moment from "moment";

const SelectCaseView = require("../../case/sel_case/view");
const SelectPatientView = require("../../sel_patient/view");
const SidebarBehavior = require("../../../behaviors/sidebar");
const TooltipBehavior = require("../../../behaviors/tooltip");

const NAView = require("../na_placeholder/view");
const storage = require("../../../utils/storage");
const _ = require("underscore");


const addDevelopOptions = function(data) {
	if ("develop" === config.NODE_ENV) {
		const AdminItems = data.availableViews.Administration.items;

		const gotDevMenus = $.grep(AdminItems, function(obj) {
			return obj.menuId === "OtherPOC2";
		});


		if (0 == gotDevMenus.length) {
			AdminItems.push({
				"menuId": "Other",
				"menuLabel": "Others not yet available",
				"eventName": "develop",
				"linkURL": "/admin_other_config/",
				"tip": "This is a custom tool tip"
			});
			AdminItems.push({
				"menuId": "OtherPOC",
				"menuLabel": "A Proof of Concept",
				"eventName": "develop",
				"linkURL": "/admin_other_poc/",
				"tip": "This is a custom tool tip"
			});
			AdminItems.push({
				"menuId": "OtherPOC2",
				"menuLabel": "Another Proof of Concept",
				"eventName": "develop",
				"linkURL": "/admin_other_cases/",
				"tip": "This is a custom tool tip"
			});
		}
		data.availableViews.Color = {
			"menuId": "Colors",
			"menuIcon": "fa-folder-open-o",
			"menuLabel": "Color Swatches",
			"collapsible": true,
			"collapsed": true,
			"roles": [
				"Surgery Coder"
			],
			"items": [{
				"menuId": "O_OI",
				"menuLabel": "Operation Information",
				"linkURL": "",
				"eventName": ""
			}]
		};
	}
	return data;
};

module.exports = Marionette.StateView.extend({
	template: require("./template.hbs"),
	id: "sidebar",
	className: "flexIt flexIt-container",

	stateEvents: {
		"change": "render"
	},

	regions: {
		sidebar: "#r-sidebar",
		body: "#r-body"
	},

	events: {
		"click .sidebarMenu-link-sub": "onClickSubLink"
	},

	triggers: {
		"click .js-Logout": "select:Logout",
		"click .js-select-case": "select:case",
		"click .js-select-SML-Patient": "select:smlPatient",
		"click .js-select-SML-Cases": "select:smlCases",
		"click .js-select-SML-SSLQI": "select:smlSslqi",
		"click .js-select-SML-ORs": "select:smlOperatingRooms",
		"click .js-select-SML-Dates": "select:smlDates",
		"click .js-select-SML-ORM": "select:smlOrm",
		"click .js-select-SML-SCC": "select:smlScc",
		"click .js-select-SML-NYA": "select:smlNya"
	},

	behaviors: {
		SidebarBehavior,
		TooltipBehavior
	},

	initialize() {
		// Needed when using StateView - calls its constructor
		Marionette.StateView.prototype.initialize.apply(this, arguments);

		// Collect relevant data from various Radio channels
		Object.assign(this.options, {
			user: Radio.request("auth", "user"),
			facility: Radio.request("store", "get", "activeFacility"),
			patient: Radio.request("store", "get", "activePatient"),
			activeCase: Radio.request("store", "get", "activeCase"),
			viewAccess: Radio.request("store", "get", "viewAccess")
		});
	},

	templateContext() {
		const user = this.getOption("user");
		const facility = this.getOption("facility");
		const patient = this.getOption("patient");
		const viewAccess = this.getOption("viewAccess");

		const activeCase = undefined;
		const ERRactiveCase = this.getOption("activeCase");

		let data = {
			develop_mode: ("develop" === config.NODE_ENV),
			user: user ? user.toJSON() : undefined,
			facility: facility ? this.facilityConversion(facility) : undefined,
			patient: patient ? this.patientConversion(patient) : undefined,
			activeCase: activeCase ? activeCase.toJSON() : undefined,
			availableViews: viewAccess.getPermittedForUser(user)
		};
		data = addDevelopOptions(data);

		return data;
	},

	getTemplate() {
		const template = require("./template.hbs");
		return this.getOption("user") ? template : false;
	},

	facilityConversion(f) {
		const facility = f.toJSON();
		return {
			"Facility": storage.getItem("facilityName"),
			"Station ID": facility.stationID,
			"Complexity Level": facility.complexityLevel,
			"Facility Size": facility.facilitySize,
			"State": facility.state,
			"VISN": facility.visn
		};
	},

	patientConversion(p) {
		const patient = p.toJSON();
		const dob = patient.dob;
		const display_dob = moment(dob).format("MM/DD/YYYY");
		const ret = {
			"Name": patient.name,
			"Age": patient.age,
			"DOB": display_dob,
			"Gender": patient.gender,
			"SSN": patient.ssn,
			"Address": patient.addrStreet1,
			"City, State/Zip": patient.addrCityStateZip
		};

		if (patient.hasOwnProperty("provider")) {
			ret.Provider = patient.provider;
		}

		return ret;
	},

	handleLogin(user) {
		Object.assign(this.options, { user });
		this.render();
	},

	updateFacility(facility) {
		Object.assign(this.options, { facility });
		this.render();
	},

	updatePatient(patient) {
		if (patient) {
			Object.assign(this.options, { patient });
			this.listenTo(patient, "change", this.render);
			if (patient.isLoaded()) this.render();
		}
	},

	updateCase(activeCase) {
		Object.assign(this.options, { activeCase: activeCase });
		this.render();
	},

	onAttach() {
		// Backbone event handler
		// Listento - the model defined in the "auth" radio channel
		// for the "user:authenticate" event (which is triggered in the globalAPI/Auth.js module)
		// and call the "handleLogin" method (below)
		// using the "this" scope.
		this.listenTo(Radio.channel("auth"), "user:authenticate", this.handleLogin, this);

		this.listenTo(Radio.channel("store"), {
			"change:activeFacility": this.updateFacility,
			"change:activePatient": this.updatePatient,
			"change:activeCase": this.updateCase
		}, this);
	},

	onRender() {
		// Note: User Info, Facility Info, and Patient Info (if a patient has been selected) are all expanded by default
		// All other collapsible menus are expanded or collapsed by default depending on the "collapsed" parameter in the menu definition
		// located in the "view_roles_map.json" file
		if (this.getOption("user")) {
			$("#collapseUserInfo").collapse("show");
		}
		if (this.getOption("facility")) {
			$("#collapseFacilityInfo").collapse("show");
		}
		if (this.getOption("viewAccess")) {
			this.getOption("viewAccess").each((model) => {
				const id = model.get("menuId");
				const label = model.get("menuLabel");
				const collapsed = model.get("collapsed");
				const collapsible = model.get("collapsible");
				const theUser = this.getOption("user");
				let sMenu = model.get("secondaryMenu");
				if (sMenu && theUser) {
					const sMenu4Convert = sMenu.menuLabel4Conversion;
					if (sMenu4Convert) {
						let newLabel = sMenu4Convert.replace("%u", `${theUser.get("firstname")} ${theUser.get("lastname")}`);
						sMenu = Object.assign(sMenu, {"menuLabel": newLabel});
						model.set("secondaryMenu", sMenu);
					}
				}
				if ("PatientManagement" === id) {
					if(this.getOption("patient")) {
						$(`#collapse${id}`).collapse("show");
					}
				}
				else {
					if (collapsible && !collapsed) {
						$(`#collapse${id}`).collapse("show");
					}
					else {
						$(`#collapse${id}`).collapse("hide");
					}
				}
			});
		}
	},

	onBeforeDestroy() {
		this.stopListening(Radio.channel("auth"));
		this.stopListening(Radio.channel("store"));
	},

	onSelectLogout() {
		storage.clear();
		location.reload();
	},

	onSelectSmlPatient() {
		Radio.request("root", "modal", { view: new SelectPatientView() });
	},

	onSelectSmlSslqi() { // pass the type of the selected item (or any other data we want to show up) to the view
		Radio.request("root", "modal", { view: new NAView({"type": "SSLQI"}) });
	},

	onSelectSmlOperatingRooms() {
		Radio.request("root", "modal", { view: new NAView({"type": "Operating Rooms"}) });
	},

	onSelectSmlDates() {
		Radio.request("root", "modal", { view: new NAView({"type": "Dates"}) });
	},

	onSelectSmlOrm() {
		Radio.request("root", "modal", { view: new NAView({"type": "ORM"}) });
	},

	onSelectSmlScc() {
		Radio.request("root", "modal", { view: new NAView({"type": "SCC"}) });
	},

	onSelectSmlNya() {
		Radio.request("root", "modal", { view: new NAView({"type": "NYA"}) });
	},

	onClickSubLink(e) {
		e.stopPropagation();
		e.preventDefault();

		const href = $(e.currentTarget).data("href");
		if (href) {
			Radio.request("global", "navigate", href);
		}
	}
});
