const Backbone = require("backbone");
const Marionette = require("backbone.marionette");
const _ = require("underscore");
const API = require("../../api");
const Radio = require("backbone.radio");
const authHeaders = require("../../utils/authHeaders");
const CaseModel = require("../../entities/case/model");
const SpecialtiesCollection = require("../../entities/administration/specialty/collection");
const PersonCollection = require("../../entities/administration/person/collection");
const PatientModel = require("../../entities/patient/model");
const SelectPatientView = require("../sel_patient/view");
const CancelCaseView = require("./cancel_case/view");
const CloseCaseView = require("./close_case/view");
const SearchSSLQIView = require("../../modules/administration/sslqi_management/search/view");
const SSLQIModel = require("../../entities/administration/sslqi_management/model");

const TabView = require("../../components/tabs/view");
const CaseTabButtons = require("./components/case_tab_buttons");
const CaseCommonMixin = require("./components/case_common");

module.exports = Marionette.StateView.extend(_.extend(CaseCommonMixin, {
	id: "casePanel",
	className: "flexIt flexIt-container",
	template: require("./template.hbs"),

	regions: {
		tabRegion: "#tab-tops",
		buttonRegion: "#tabButtonsDiv",
		tabPanel: "#tab"
	},

	ui: {
		formxxx: "#cfForm",
		filterType: "#filterType option:selected",
		search: "#search",
		surgerySpecialty: "#surgerySpecialty",
		cancel: "#cancel"
	},

	defaultState() {
		return {
			caseScreenTitle: "Case",
			showTabsPanel: true,
			showResultMessage: false,
			resultMessage: "",
			caseTypeValue: "",
			selectedTab: this.isModeNew() ? "tab2": "tab0",
			filterTypeText: "Specialty",
			currentIen: null
		};
	},

	childViewEvents: {
		"action": "formAction"
	},

	stateEvents: {
		"change": "render"
	},

	events: {
		"click @ui.cancel": () => Radio.request("global", "navigate", "/"),

		// ENTER key clicks Search button
		"keyup": function(e) {
			if(e.which === 13) {
				this.getUI("search").click();
			}
		},

		// Other filter typeahead fields
		"click #surgerySpecialty":  function (e) {
			this.state.set("filterTypeText", "Specialty");
		}
	},

	templateContext() {
		const activePatient = Radio.request("store", "get", "activePatient");

		return {
			activePatient: activePatient && activePatient.toJSON()
		};
	},

	isModeNew() {
		return this.options.mode === "new";
	},
	isModeView() {
		return this.options.mode === "view";
	},

	/**
	 * Order of execution:
	 * 1: intialize
	 * 2: onRender
	 * 3: onDomRefresh
	 * 4: onAttach
	 */
	initialize() {
		// Needed when using StateView - calls its constructor
		Marionette.StateView.prototype.initialize.apply(this, arguments);

		let specialties = new SpecialtiesCollection();
		specialties.fetch(authHeaders());
		Radio.request("store", "set", "specialties", specialties);

		let people = new PersonCollection();
		people.fetch(authHeaders());
		Radio.request("store", "set", "people", people);

		// Collect relevant data from Radio channel
		this.options.activePatient = Radio.request("store", "get", "activePatient");
		const case_attrs = {};
		if (this.options.mode === "view") {
			case_attrs.ien = this.options.case_id;
		}
		this.options.currentCase = new CaseModel(case_attrs);
		if (this.isModeView()) {
			this.options.currentCase.fetch(authHeaders());
		}
		if (this.isModeNew() && this.options.activePatient) {
			this.options.currentCase.set("patient", this.options.activePatient.id);
		}

		this.listenTo(Radio.channel("store"), {
			"change:activePatient": this.updatePatient
		}, this);
	},

	onRender: function () {
		const selectedTab = this.state.get("selectedTab");

		if (this.options.mode !== "filter") {
			this.$("#caseFilterPanel").hide();
			this.$(".filter-ec-button-div").hide();
			this.$("h1").text("Case");
		}

		if (this.options.mode === "new" && !this.options.activePatient) {
			Radio.request("root", "modal", { view: new SelectPatientView() });

			this.$("#tab").text("Please select a patient before creating a new case.");
			return;
		}

		this.showChildView("tabRegion", new TabView({
			tabs: this.isModeNew() ? _.omit(this.tabs, "tab0", "tab1", "tab6", "tab7") : this.tabs,
			selected: selectedTab
		}));

		this.showChildView("tabPanel", this.getTabPanel(selectedTab));

		if (selectedTab !== "tab0" && selectedTab !== "tab1") {
			this.showChildView("buttonRegion", new CaseTabButtons({
				closeOrCancel: "clear"
			}));
		}
	},

	onAttach: function() {
		this.ui.formxxx.show();
	},

	loadPatientCase() {
		let activePatient = Radio.request("store", "get", "activePatient");
		if (!this.options.currentCase.id) {
			this.options.currentCase = new CaseModel({patient: activePatient.id});
		}
	}
}));
