import _ from "underscore";
import Backbone from "backbone";
import Marionette from "backbone.marionette";
import Radio from "backbone.radio";
import moment from "moment";

import FormView from "../../../components/form/view";
import PeopleSelect from "../../../utils/entity-select/people";
import ListsSelect from "../../../utils/entity-select/lists";

import ListCollections from "../../../entities/lists/collections";

import API from "../../../api";
import authHeaders from "../../../utils/authHeaders";

import CPTCollection from "../../../entities/case/cpt/collection";


function cleanupTime(rawTime) {
	const try1 = moment(rawTime, "MM/DD/YYYY HH:mm");
	if (try1.isValid()) {
		return try1.toISOString();
	}
	return moment(rawTime).toISOString();
}

const tasFormFields = [
	{
		label: "Date of Operation <i class=\"fa fa-asterisk required\" aria-hidden=\"true\" data-toggle=\"tooltip\" title=\"Required\"></i>",
		type: "date",
		field: "dateOfOperation",
		width: 2
	}, {
		label: "Surgical Specialty",
		type: "select",
		selSrc: ListsSelect.Specialty,
		field: "surgerySpecialty",
		width: 4
	}, {
		label: "Primary Surgeon",
		field: "primarySurgeon",
		type: "select",
		selSrc: PeopleSelect,
		width: 3
	}, {
		label: "Attending Surgeon",
		field: "attendingSurgeon",
		type: "select",
		selSrc: PeopleSelect,
		width: 3
	}, {
		label: "First Assist",
		field: "firstAsst",
		type: "select",
		selSrc: PeopleSelect,
		valueKey: "id",
		labelKey: "name"
	}, {
		label: "Second Assist",
		field: "secondAsst",
		type: "select",
		selSrc: PeopleSelect,
		valueKey: "id",
		labelKey: "name"
	}, {
		label: "Case Status",
		type: "read-only",
		field: "caseStatus"
	}, {
		label: "Case Schedule Order",
		type: "text",
		field: "caseScheduleOrder"
	}, {
		label: "Scheduled Start Time",
		type: "time",
		field: "scheduledStartTime",
		width: 3,
		onGet(v, o) {
			console.log("onGet:", v, o);
			let m = moment(v);
			console.log(m, m.format("H:mm a"));
			if (v) {
				return m.format("H:mm a");
			} else {
				return v;
			}
		},
		onSet(v, o) {
			console.log("onSet:", v, o);
			let m = moment(v, "H:mm a");
			console.log("m:", m);
			if (v) {
				return m.toISOString();
			} else {
				return v;
			}
		},
		getVal($el, e, o) {
			console.log("getVal:", $el, e, o);
			return $el.val();
		},
		update($el, v, m, o) {
			console.log("update:", $el, v, m, o);
			$el.val(v);
		}
	}, {
		label: "Scheduled End Time",
		type: "time",
		field: "scheduledEndTime",
		width: 3,
		onGet(v, o) {
			console.log("onGet:", v, o);
			let m = moment(v);
			console.log(m, m.format("H:mm a"));
			if (v) {
				return m.format("H:mm a");
			} else {
				return v;
			}
		},
		onSet(v, o) {
			console.log("onSet:", v, o);
			let m = moment(v, "H:mm a");
			console.log("m:", m);
			if (v) {
				return m.toISOString();
			} else {
				return v;
			}
		}
	}, {
		label: "Duration",
		type: "read-only",
		field: "calculatedDuration",
		width: 3
	}, {
		label: "Estimated Overtime",
		type: "text",
		field: "estimatedOverTime",
		width: 3
	}
];

const TimeAndStaffForm = FormView.extend({
	formFields: tasFormFields,
	className: "case-tab-panel"
});


const cptDurationFormFields = [
	{
		label: "Surgeon",
		type: "text",
		field: "surgeonDuration",
		width: 3
	}, {
		label: "Site",
		type: "text",
		field: "siteDuration",
		width: 3
	}, {
		label: "Specialty",
		type: "text",
		field: "specialtyDuration",
		width: 3
	}, {
		label: "National",
		type: "text",
		field: "nationalDuration",
		width: 3
	}, {
		type: "empty"
	}, {
		type: "button",
		width: 2,
		"btnId": "recalcDuration",
		"btnXtraCase": "recalc-duration",
		"btnName": "recalcDuration",
		"btnTitle": "Recalc. Duration",
		"btnLabel": "Recalc. Duration"
	}
];

const CptDurationForm = FormView.extend({
	formFields: cptDurationFormFields,
	className: "case-tab-panel",
	events: {
		"click button#recalcDuration": "recalcClick"
	},
	initialize() {
		FormView.prototype.initialize.apply(this, arguments);
		if (!this.options.cptReference) {
			throw new Error("CptDurationForm requires cptReference option.");
		}
		this.cptReference = this.options.cptReference;
		if (!this.options.case) {
			throw new Error("CptDurationForm requires case option.");
		}
		this.case = this.options.case;
	},
	recalcClick(evt) {
		let cptVal = this.case.get("plannedPrincOpProcCPT");
		let cptCode = this.cptReference.get(cptVal).get("cptCode");
		let url = API.getURL("CPTDurationCalculate") + "/" + cptCode;
		let options = _.extend(authHeaders(), {
			method: "POST",
			success: () => {
				Radio.request("root", "dialog", {
					type: "info",
					title: "Recalc Duration Success",
					message: "Duration re-calculation successfully requested."
				});
			},
			error: () => {
				Radio.request("root", "dialog", {
					type: "error",
					title: "Recalc Duration Error",
					message: "Please try again later."
				});
			}
		});
		$.ajax(url, options);
	},
	onRender() {
		if (!this.model) {
			return;
		}
	}
});


const CPTDurationSearch = Backbone.Model.extend({
	url: API.getURL("CPTDurationSearch"),
	sync(method, model, options) {
		return Backbone.sync.call(this, "create", model, options);
	}
});

module.exports = Marionette.View.extend({
	className: "case-tab-panel",
	template: require("./time_staff_panel.hbs"),
	regions: {
		tasForm: "#tas-form",
		cptForm: "#cpt-form"
	},
	initialize() {
		this.cptReference = new CPTCollection();
		this.cptReference.fetch(authHeaders()).done( (data) => {
			let deferred = this.durationsQuery();
			deferred.done( () => {
				this.showCptDurations();
			});
		});
	},
	showCptDurations() {
		let reg = this.getRegion("cptForm");
		if (!reg) {
			/*
			   This can happen if this function is called after we've
			   already left the tab, which can happen since it is
			   called from an ajax load callback.
			 */
			return;
		}
		reg.show(new CptDurationForm({model: this.cptDurationResults,
			case: this.model,
			cptReference: this.cptReference}));
	},
	durationsQuery() {
		let specialty = this.model.get("surgerySpecialty");
		let surgeon = this.model.get("primarySurgeon");
		let cptVal = this.model.get("plannedPrincOpProcCPT");

		let optData = {
				surgeon: surgeon,
				specialty: specialty
		};

		if (cptVal) {
			let cptCode = this.cptReference.get(cptVal).get("cptCode");
			if (cptCode) {
				optData.cptCode = cptCode;
			}
		}

		let options = _.extend(authHeaders(), {
			dataType: "json",
			method: "POST",
			type: "POST",
			contentType: "application/json",
			data: JSON.stringify(optData)
		});


		this.cptDurationResults = new CPTDurationSearch();
		let deferred = this.cptDurationResults.fetch(options);
		return deferred;
	},
	onRender() {
		this.showChildView("tasForm",
			new TimeAndStaffForm({model: this.model}));
	}
});
