const Backbone = require("backbone");
const Marionette = require("backbone.marionette");
const Radio = require("backbone.radio");
const authHeaders = require("../../../utils/authHeaders");
const API = require("../../../api");
const TabView = require("../../../components/tabs/view");
const IndexView = require("../../../modules/index/view");
const SSLQIModel = require("../../../entities/administration/sslqi_management/search/model");
const SSLQIReservationModel = require("../../../entities/administration/sslqi_management/reservation/model");

module.exports = Marionette.StateView.extend({

	template: require("./sslqi_tab_content_panel.hbs"),

	ui: {
		reserveBtn: "#reserve",
		saveBtn: "#save",
		deleteBtn: "#delete",
		actionResult: "#actionResult",
		item: "#item",
		inventoryNumber: "#inventoryNumber",
		itemCategory: "#itemCategory",
		itemType: "#itemType",
		classification: "#classification",
		specialty: "#specialty",
		startDT: "#startDT",
		endDT: "#endDT",
		tabContentInnerPanel: "#tabContentInnerPanel"
	},

	defaultState: {
		selectedTab: "tab0"
	},

	stateEvents: {
		"change": "render"
	},

	regions: {
		tabTopsRegion: "#tab-tops"
	},

	events: {
		"click #save":  function (e) {
			this.saveUpdateSSLQI();
		},

		"click #reserve":  function (e) {
			this.reserveUpdateSSLQI();
		},

		"click #delete":  function (e) {
			if(this.getOption("parent").getOption("isChildView")) {
				this.deleteReservedSSLQI();
			} else {
				this.deleteSSLQI();
			}
		},

		"click #cancel":  function (e) {
			this.cancelOutOfScreen();
		}
	},

	templateContext() {

		const currentSSLQI = this.getOption("currentSSLQI");
		const sslqiFormStructure = this.getOption("sslqiFormStructure");
		const parent = this.getOption("parent");
		const parentIsChildView = parent.getOption("isChildView");
		const isNewSSLQI = this.getOption("isNewSSLQI");

		const ctx = {

			currentSSLQI: currentSSLQI ? currentSSLQI.toJSON() : undefined,
			sslqiFormStructure: sslqiFormStructure ? sslqiFormStructure.toJSON() : undefined,
			parent: parent,
			isNewSSLQI: isNewSSLQI,
			parentIsChildView: parentIsChildView
		};

		console.log("Rendering from sslqi_tab_content_panel component view:\n", ctx);
		return ctx;
	},

	initialize: function() {

		// Needed when using StateView - calls its constructor
		Marionette.StateView.prototype.initialize.apply(this, arguments);

		// If changing rows in SSLQI list, remain on the same tab as previous row
		if(!this.getOption("isNewSSLQI")) {
			let storedSelectedTab = Radio.request("store", "get", "sslqiSelectedTab");
			storedSelectedTab = (storedSelectedTab !== undefined) ? storedSelectedTab : "tab0";
			this.state.set({
				selectedTab: storedSelectedTab
			},{silent: true});
		}

		if(this.getOption("parent").getOption("isChildView")) {

			// Start off with Reservation tab if the parent view is a child view
			this.state.set({selectedTab: "tab3"}, {silent: true});

			console.log(this.captureAllFormData());

			// This is used to maintain the state of the SSLQI tabs (including the Reservation tab) across either
			// the selection of the Cases (search) screen expand/collapse Case Filtering panel, or the selection of
			// a new case. The current SSLQI (row) id is set here but depending on certain conditions, but there is
			// also a bit of code in the SSLQI search view onAttach event that plays into its regulation. The
			// currentSSLQI objects's data is kept as-is (whether full or empty of data) across the expansion and
			// collapse of the Case Filtering panel. The currentSSLQI object's data is emptied if a new case is
			// selected in the Cases (search) screen.
			if(Radio.request("store", "get", "isExpandCollapse")) {
				let currSSLQI = Radio.request("store", "get", "currSSLQI");
				if (currSSLQI === undefined || currSSLQI === null || currSSLQI.get("classification") === "") {
					Radio.request("store", "set", "currSSLQI", new SSLQIModel());
					Radio.request("store", "set", "currentSSLQIId", null);
				}

				Radio.request("store", "set", "isExpandCollapse", false);
			} else {
				if (Radio.request("store", "get", "isNewIen")) {

					Radio.request("store", "set", "currSSLQI", new SSLQIModel());
					Radio.request("store", "set", "currentSSLQIId", null);
					Radio.request("store", "set", "isNewIen", false);
				}
			}

			Object.assign(this.options, {currentSSLQI: Radio.request("store", "get", "currSSLQI")});
		}

		this.loadSSLQIFormStructure();
	},

	onRender: function () {
		const isChildView = this.getOption("parent").getOption("isChildView");
		const selectedTab = this.state.get("selectedTab");

		const tabs = {
			tab0: "Details",
			tab1: "Usage",
			tab2: "Safety"
		};

		if (isChildView) {
			tabs.tab3 = "Reservation";
		}

		// TODO: Ask Joshua why tabs variable needs to be global. Tried it as template context options var and failed.
		this.showChildView("tabTopsRegion", new TabView({
			// Maintains state in between tab clicks
			tabs,
			selected: selectedTab
		}));

		// Add visual mini calendaring functionality to date fields
		this.$("#startDT,#endDT").datetimepicker({
			//format: "MM/DD/YYYY h:mm a", // Got error saying failed to get computed style for (date) element
			format: "MM/DD/YYYY",
			formatDate: "MM/DD/YYYY"
		});

		// Reload the state variable that contains the data to be displayed in the Reservation tab
		if(this.getOption("parent").getOption("isChildView")) {
			this.state.set({
				activeReservedSSLQI: Radio.request("store", "get", "activeReservedSSLQI")
			}, {silent: true});
		}
	},

	onAttach: function() {
		this.loadSSLQIFormStructure();
	},

	onChildviewTabSelected(tab) {
		this.options.currentSSLQI.set(this.captureAllFormData());
		Radio.request("store", "set", "sslqiSelectedTab", tab);
		this.state.set("selectedTab", tab);
	},

	/**
	 * Method called when view is first displayed, fetching the metadata for form structure from a JSON file inside
	 * the data_files directory. The API for it is configured inside the api.js file.
	 */
	loadSSLQIFormStructure() {
		let SSLQIFormStructureModel  = Backbone.Model.extend({
			url: API.getLocalURL("SSLQIs")
		});

		this.sslqiFormStructure = new SSLQIFormStructureModel();

		return this.sslqiFormStructure.fetch()
			.then(() => {
				console.log("Loaded sslqiFormStructure:\n" + this.sslqiFormStructure);
				this.render();
			})
			.catch((err) => console.error("Failed to retrieve sslqiFormStructure:", err));
	},

	/**
	 * Method called when a user wants to save or update an association of a current
	 * case ID (i.e., ien) with an existing SSLQI record
	 */
	reserveUpdateSSLQI() {

		if(this.validateForm()) {

			this.getOption("parent").state.set({childAction: "reserve"}, {silent: true});
			let activeCase = Radio.request("store", "get", "activeCase");
			let enteredSSLQIReservation = new SSLQIReservationModel();

			// Gather Reservation tab form values (including classification which is not in the Reservation tab)
			enteredSSLQIReservation.set(this.captureReservationFormData());

			// Check whether this is going to be a save or an update (i.e., a POST or a PUT)
			let currentVistaId = this.getOption("currentSSLQI").get("vistaId");
			let currentClassification = enteredSSLQIReservation.get("classification");
			let reservationId = this.scanForExistingReservationId(currentVistaId, currentClassification);
			if(reservationId !== null && reservationId !== undefined && reservationId !== "") {
				enteredSSLQIReservation.set({_id: reservationId});
			}

			//Set values for fields not in Reservation tab form
			enteredSSLQIReservation.set({
				caseId: activeCase.get("ien"),
				vistaId: currentVistaId
			});

			enteredSSLQIReservation.save(null, authHeaders())
				.then(() => {
					console.log("Successful reservation of record:\n" + enteredSSLQIReservation);
					Radio.request("store", "set", "activeReservedSSLQI", enteredSSLQIReservation);
					this.getOption("parent").getReservedSSLQIs();
					this.showResultMessage("info", "SSLQI Reservation Status", "Record successfully reserved or updated");
				})
				.catch(err => {
					console.error("Error reserving SSLQI", err);
					this.showResultMessage("error", "SSLQI Reservation Status", "Error reserving record");
				});
		}
	},

	/**
	 * Method called when a user wants to save a new or update an existing SSLQI record
	 */
	saveUpdateSSLQI() {

		if(this.validateForm()) {

			this.getOption("parent").state.set({childAction: "saveUpdate"}, {silent: true});
			this.options.currentSSLQI.set(this.captureAllFormData());
			this.options.currentSSLQI.save(null, authHeaders())
				.then(() => {
					console.log("Successful saving of record:\n" + this.options.currentSSLQI);
					if (!this.getOption("isNewSSLQI")) {this.getOption("parent").getSSLQIs();}
					this.showResultMessage("info", "SSLQI Status", "Record successfully saved or updated");
					Radio.request("store", "set", "activeSSLQI", this.options.currentSSLQI);
				})
				.catch(err => {
					console.error("Error saving SSLQI", err);
					this.showResultMessage("error", "SSLQI Status", "Error saving record");
				});
		}
	},

	/**
	 * Method called when a user wants to delete existing SSLQI form data. It is
	 * deleted by ID.
	 */
	deleteSSLQI() {

		this.getOption("parent").state.set({childAction: "delete"}, {silent: true});
		const currentSSLQI = this.getOption("currentSSLQI");
		currentSSLQI.idAttribute = "_id";

		if (currentSSLQI.isNew()){
			return;
		}

		this.captureAllFormData();
		return currentSSLQI.destroy(authHeaders())
			.then(() => {
				console.log("Successful deletion of record:", currentSSLQI);

				if(!this.getOption("isNewSSLQI")) {
					this.getOption("parent").getSSLQIs();
				}

				this.showResultMessage("info", "SSLQI Status", "Record successfully deleted");
			})
			.catch(err => {
				console.error("Error deleting SSLQI", err);
				this.showResultMessage("error", "SSLQI Status", "Error deleting record");
			});
	},

	// TODO: Answer question - What happens if an SSLQI is deleted from the administration while a user has this SSLQI
	// associated with the case they are viewing? Are we to presume that this will be enforced by the database?
	// Because it was not just now 4/5/2017.
	/**
	 * Method called when a user wants to delete an SSLQI reservation (that is associated with a case ID)
	 */
	deleteReservedSSLQI() {

		this.getOption("parent").state.set({childAction: "delete"}, {silent: true});
		let selectedSSLQIReservation = new SSLQIReservationModel();

		// Gather Reservation tab form values (including classification which is not in the Reservation tab)
		selectedSSLQIReservation.set(this.captureReservationFormData());

		// Check whether the reservation ID exists for this case. If so, then delete. Otherwise, alert user to
		// there not being a reservation ID associated with the current case ID (ien).
		let currentVistaId = this.getOption("currentSSLQI").get("vistaId");
		let currentClassification = selectedSSLQIReservation.get("classification");
		let reservationId = this.scanForExistingReservationId(currentVistaId, currentClassification);

		if(reservationId !== null && reservationId !== undefined && reservationId !== "") {

			selectedSSLQIReservation.set({_id: reservationId});
			return selectedSSLQIReservation.destroy(authHeaders())
				.then(() => {
					console.log("Successful deletion of record:", selectedSSLQIReservation);
					this.getOption("parent").getReservedSSLQIs();
					this.showResultMessage("info", "SSLQI Reservation Status", "Record successfully deleted");
				})
				.catch(err => {
					console.error("Error deleting SSLQI for case", err);
					this.showResultMessage("error", "SSLQI Reservation Status", "Error deleting record");
				});
		} else {
			this.showResultMessage("error", "SSLQI Deletion Error", "This SSLQI is not deletable as it is not associated with the case being viewed");
		}
	},

	/**
	 * Method called to validate search fields prior to performing a search based on their entered values
	 */
	validateForm() {

		let itemValue = this.getUI("item").val();
		let inventoryNumberValue = this.getUI("inventoryNumber").val();
		let itemCategoryValue = this.getUI("itemCategory").val();
		let itemTypeValue = this.getUI("itemType").val();
		let classificationValue = this.getUI("classification").val();
		let specialtyValue = this.getUI("specialty").val();
		let startDTValue = this.getUI("startDT").val();
		let endDTValue = this.getUI("endDT").val();
		let fResultMessage = "";
		let fResultMessageAdditional = "";
		let isValidated = false;

		if(this.getOption("parent").getOption("isChildView")) {

			if (itemValue === "" ||
				inventoryNumberValue === "" ||
				itemCategoryValue === "" ||
				itemTypeValue === "" ||
				classificationValue === "" ||
				specialtyValue === "" ||
				startDTValue === "" ||
				endDTValue === "") {

				if(itemValue === "" )
					fResultMessageAdditional += " Item,";

				if(inventoryNumberValue === "" )
					fResultMessageAdditional += " Inventory Number,";

				if(itemCategoryValue === "" )
					fResultMessageAdditional += " Item Category,";

				if(itemTypeValue === "" )
					fResultMessageAdditional += " Item Type,";

				if(classificationValue === "" )
					fResultMessageAdditional += " Classification Value,";

				if(specialtyValue === "" )
					fResultMessageAdditional += " Specialty Value,";

				if(startDTValue === "" )
					fResultMessageAdditional += " Start Date,";

				if(endDTValue === "" )
					fResultMessageAdditional += " End Date,";

				// Strip off last comma
				fResultMessageAdditional = fResultMessageAdditional.slice(0, -1);

				fResultMessage = "Please enter appropriate required data:";
				fResultMessage += fResultMessageAdditional;
				isValidated = false;

			} else {
				isValidated = true;
			}

		} else {

			if (itemValue === "" ||
				inventoryNumberValue === "" ||
				itemCategoryValue === "" ||
				itemTypeValue === "" ||
				classificationValue === "" ||
				specialtyValue === "") {

				if(itemValue === "" )
					fResultMessageAdditional += " Item,";

				if(inventoryNumberValue === "" )
					fResultMessageAdditional += " Inventory Number,";

				if(itemCategoryValue === "" )
					fResultMessageAdditional += " Item Category,";

				if(itemTypeValue === "" )
					fResultMessageAdditional += " Item Type,";

				if(classificationValue === "" )
					fResultMessageAdditional += " Classification Value,";

				if(specialtyValue === "" )
					fResultMessageAdditional += " Specialty Value,";

				// Strip off last comma
				fResultMessageAdditional = fResultMessageAdditional.slice(0, -1);

				fResultMessage = "Please enter required data:";
				fResultMessage += fResultMessageAdditional;
				isValidated = false;

			} else {
				isValidated = true;
			}
		}

		if(!isValidated) {
			this.showResultMessage("error", "SSLQI Validation Error", fResultMessage);
		}

		return isValidated;
	},

	captureAllFormData() {
		const sslqiFormData = {};
		$("select, input").each(function(key, element){

			console.log("sslqiFormData:", key, element);
			if (element.id) {

				// Exclude Case form fields
				if(element.id !== "filterType" && element.id !== "filterData" && element.id !== "fromDate" &&
					element.id !== "toDate" && element.id !== "category" && element.id !== "item_Type" &&
					element.id !== "itemSpecialty" && element.id !== "nonORCB" && element.id !== "patientCB" &&
					element.id !== "placeholderCB" && element.id !== "requestedCB" && element.id !== "scheduledCB") {

					sslqiFormData[element.id] = element.value;
				}
			}
		});

		return sslqiFormData;
	},

	captureReservationFormData() {
		const sslqiFormData = {};
		$("select, input").each(function(key, element){

			console.log("sslqiFormData:", key, element);
			if (element.id) {

				// Include only reservation model object fields (mostly in Reservation tab)
				if(element.id === "classification" || element.id === "room" || element.id === "requester" ||
					element.id === "startDT" || element.id === "endDT" || element.id === "reservationComments") {

					sslqiFormData[element.id] = element.value;
				}
			}
		});

		return sslqiFormData;
	},

	/**
	 * This method is needed given that when an SSLQI Reservation is first saved, it does not return the
	 * SSLQI Reservation object model with an "_id" field. Therefore, the state SSLQI Reservation objec that
	 * populates the Reservation tab data does not get stored with an "_id" field value. Therefore, it cannot
	 * be used to get the current SSLQI Reservation object's "_id".
	 */
	scanForExistingReservationId(currentVistaId, currentClassification) {

		let reservationId = null;
		let activeReservedSSLQIs = Radio.request("store", "get", "activeReservedSSLQIs");

		$.each(activeReservedSSLQIs.models, function (key, value) {
			if(value.get("vistaId") === currentVistaId && value.get("classification") === currentClassification) {
				reservationId = value.get("_id");
			}
		});

		return reservationId;
	},

	showResultMessage(type, title, message) {

		// Maintain display of any data that might have been entered
		this.options.currentSSLQI.set(this.captureAllFormData());

		Radio.request("root", "dialog", {
			type: type,
			title: title,
			message: message
		});
	},

	cancelOutOfScreen() {
		Backbone.history.navigate( "/", { trigger: false, replace: true } );
		Radio.request("root", "body", new IndexView({}));
	}
});
