const _ = require("underscore");
const Marionette = require("backbone.marionette");
const Radio = require("backbone.radio");

import FoldingView from "../../../components/folding_view";
import PatientInformationPanel from "./patient_demographic_panel.js";

import FormView from "../../../components/form/view";

import CptSelect from "../../../utils/entity-select/cpt_procedure";
import FieldsSelect from "../../../utils/entity-select/fields";
import ListsSelect from "../../../utils/entity-select/lists";
import PeopleSelect from "../../../utils/entity-select/people";

/* The final view will consist of three primary parts.

   First, there is a folding subview for the patient personal information.
   Second, there is a Form that is mort of the patient & procedure panel.
   Third, the CPT field (actually sits between the prior two).

   The CPT field is seperate from the form because it has special
   behaviour on value changes that isn't part of the normal form to
   model binding.  The main point is that when you change the cpt
   field, two model fields get changed.

*/

const ppFormFields = [
	{
		label: "Laterality of Procedure",
		field: "procedureLaterality",
		type: "select",
		selSrc: FieldsSelect.ProcedureLaterality
	}, {
		label: "Principal Preoperative <abbr data-placement=\"bottom\" placement=\"bottom\" data-toggle=\"tooltip\" title=\"International Classification of Diseases\">ICD</abbr> Diagnosis Code",
		field: "prinPreOpDiagICD",
		type: "text"
	}, {
		label: "Principal Preoperative Diagnosis",
		field: "principalPreOpDiagnosis",
		type: "text"
	}, {
		label: "Other Procedures",
		field: "otherProcedures",
		type: "text"
	}, {
		label: "Multiple Team Procedures",
		field: "concurrentCase",
		type: "text"
	}, {
		label: "Other Preoperative Diagnosis",
		field: "otherPreopDiagnosis",
		type: "text"
	}, {
		label: "Other Preoperative <abbr data-placement=\"bottom\" placement=\"bottom\" data-toggle=\"tooltip\" title=\"International Classification of Diseases\">ICD</abbr> Diagnosis Code",
		field: "otherPreoperativeDiagnosisCode",
		type: "text"
	}, {
		label: "Surgical Risk (ASRC)",
		field: "surgicalRiskAsrc",
		type: "text"
	}, {
		label: "Related to: Service Connected Status/Environmental Factors",
		field: "serviceConnected",
		type: "yes-no"
	}, {
		label: "Planned Admission Status",
		field: "plannedAdmissionStatus",
		type: "select",
		selSrc: FieldsSelect.AdmissionStatus
	}, {
		label: "Palliation",
		field: "palliation",
		type: "yes-no"
	}, {
		label: "Referring Physician",
		field: "referringPhysician",
		type: "select",
		selSrc: PeopleSelect
	}, {
		label: "Brief Clinical History",
		field: "briefClinHistory",
		type: "text-area",
		width: 12
	}, {
		label: "Indications for Operation",
		field: "operationIndications",
		type: "text-area",
		width: 12
	}
];

const PatientForm = FormView.extend({
	formFields: ppFormFields,
	className: "case-tab-panel",
	onRender() {
		if (!this.model) {
			return;
		}
	}
});

module.exports = Marionette.StateView.extend({

	className: "case-tab-panel",
	template: require("./patient_panel.hbs"),

	regions: {
		patientInformationPanel: "#patientInformationDiv",
		patientForm: "#patient-form",
		cptCombo: "#cptComboPlaceHolder"
	},

	defaultState: {
		activeFilter: "principalProcedure",

		typeAheadCPTFieldsList: [{
			"field": {
				"uiName": "principalProcedure" }
		}, {
			"field": {
				"uiName": "plannedPrincOpProcCPT" }
		}]
	},

	stateEvents: {
		"change": "render"
	},

	templateContext() {

		const currentSSLQI = this.getOption("model");
		const cptCodes = this.getOption("cptCodes");
		const ctx = {
			currentSSLQI: currentSSLQI ? currentSSLQI.toJSON() : undefined,
			cptCodes: cptCodes ? cptCodes.toJSON() : undefined
		};
		console.log("Rendering from patient_panel view:\n", ctx);
		return ctx;
	},

	initialize: function() {

		// Needed when using StateView - calls its constructor
		Marionette.StateView.prototype.initialize.apply(this, arguments);
	},
	onRender() {
		this.showChildView("patientForm", new PatientForm({
			model: this.model
		}));

		const selectView = new CptSelect({
			selected: [this.model.get("plannedPrincOpProcCPT")]
		});
		this.showChildView("cptCombo", selectView);

		this.listenTo(selectView, "change", (val) => {
			console.log("cpt select view change:", val);
			this.model.set("plannedPrincOpProcCPT", val);
			console.log("bbb:", selectView.collection.get(val));
			this.model.set("principalProcedure",
						selectView.collection.get(val).get("name"));
		});
		this.listenTo(this.model, "change:plannedPrincOpProcCPT", (mod) => {
			let val = mod.get("plannedPrincOpProcCPT");
			console.log("cpt Test oot:", val, selectView.$("select").val());
			if (val !== selectView.$("select").val()) {
				selectView.$("select").selectpicker("val", val);
			}
		});

		this.showChildView("patientInformationPanel", new FoldingView({
			id: "CaseDataPatientInformation",
			title: "Patient Information",
			containedView: new PatientInformationPanel({
				model: this.model.getPatient()
			})
		}));

		if (!this.model) {
			return;
		}
	}
});
