const _ = require("underscore");
const Backbone = require("backbone");
const Marionette = require("backbone.marionette");
const Radio = require("backbone.radio");
const CPTCollection = require("../../../entities/case/cpt/collection");
const authHeaders = require("../../../utils/authHeaders");

const CaseSummary = require("./case_summary");
const SurgeonSummary = require("./surgeon_summary");
const PatientPanel = require("./patient_panel");
const TimeStaffPanel = require("./time_staff_panel");
const IntraOpPanel = require("./intraop_panel");
const SpecialCasePanel = require("./special_case_panel");
const OperationsPanel = require("./operations/operations_panel");
const RiskAssessmentPanel = require("./risk_assessment/risk_assessment_panel");

const SearchSSLQIView = require("../../../modules/administration/sslqi_management/search/view");
const SSLQIModel = require("../../../entities/administration/sslqi_management/model");

module.exports = {
	/**
	 * Method for calling CPT codes for surgical procedures
	 */

	tabs: {
		tab0: "Case Summary",
		tab1: "Surgeon Summary",
		tab2: "Patient &amp; Procedure <i class=\"fa fa-asterisk required\" aria-hidden=\"true\" data-toggle=\"tooltip\" title=\"Contains Required Fields\"></i>",
		tab3: "Time And Staff <i class=\"fa fa-asterisk required\" aria-hidden=\"true\" data-toggle=\"tooltip\" title=\"Contains Required Fields\"></i>",
		tab4: "Intraoperative",
		tab5: "Special Case Needs",
		tab6: "SSLQI",
		tab7: "Operations",
		tab8: "Risk Assessment"
	},

	getCurrentTabRegion() {
		const selectedTab = this.state.get("selectedTab");
		return this.getRegion("tabPanel");
	},

	onChildviewTabSelected(tab) {
		this.state.set("selectedTab", tab);
	},

	getTabPanel(tab) {
		const currentCase = this.getOption("currentCase");
		const activePatient = this.getOption("activePatient");
		const roomTypes = this.room_types;

		switch(tab) {
		case "tab0": return new CaseSummary({ model: currentCase });
		case "tab1": return new SurgeonSummary({ model: currentCase });
		case "tab2": return new PatientPanel({ model: currentCase, patient: activePatient });
		case "tab3": return new TimeStaffPanel({ model: currentCase });
		case "tab4": return new IntraOpPanel({ model: currentCase, room_types: roomTypes });
		case "tab5": return new SpecialCasePanel({ model: currentCase });
		case "tab6":
			return new SearchSSLQIView({
				model: new SSLQIModel(),
				currentSSLQI: new SSLQIModel(),
				isChildView: true
			});

		case "tab7": return new OperationsPanel({ model: currentCase });
		case "tab8": return new RiskAssessmentPanel({ model: currentCase });
		default: throw new Error(`No panel for tab: "${tab}"`);
		}
	},

	updatePatient(activePatient) {
		console.log("Update Patient:", activePatient);
		if(activePatient != undefined) {

			this.options.activePatient = activePatient;

			this.loadPatientCase();
			this.render();
		}
	},

	formAction(action) {
		console.log("fA:", action);
		if (action === "save") {
			console.log("Saving...");
			Radio.request("store", "set", "activeCase", this.options.currentCase);
			const isNew = !this.options.currentCase.id;

			let saveOpts = authHeaders();
			saveOpts.success = (data) => {
				Radio.request("root", "dialog", {
					type: "info",
					title: "Saved Case",
					message: "Case Saved.  You may continue editing."
				});
				if (isNew) {
					Radio.request("global", "navigate", "case/" + data.id + "/");
					this.render();
				}
			};
			saveOpts.error = (model, errors) => {
				Radio.request("root", "dialog", {
					type: "error",
					title: "Error Saving Case",
					message: "Please check your values."
				});
			};
			this.options.currentCase.save(null, saveOpts);
			return;
		}

		if (action === "close") {
			Radio.request("root", "modal", { view: new CloseCaseView() });
			return;
		}

		if (action === "cancel") {
			Radio.request("root", "modal", {
				view: new CancelCaseView({
					model: this.options.currentCase
				})
			});
			return;
		}

		if (action === "clear") {
			this.options.currentCase = new CaseModel({
				patient: Radio.request("store", "get", "activePatient").id});

			this.state.set({
				filterTypeText: "Specialty"
			});

			this.render();
			return;
		}

		if (action === "exit") {
			Radio.request("global", "navigate", "/");
		}
	},

};
