const Backbone = require("backbone");
const Marionette = require("backbone.marionette");
const _ = require("underscore");
const API = require("../../../api");
const Radio = require("backbone.radio");
const authHeaders = require("../../../utils/authHeaders");
const CaseCollection = require("../../../entities/case/collection");
const CaseSearchModel = require("../../../entities/case/search/model");
const CaseSearchCollection = require("../../../entities/case/search/collection");
const PatientCaseCollection = require("../../../entities/case/patient_cases/collection");
const SpecialtiesCollection = require("../../../entities/administration/specialty/collection");
const CaseModel = require("../../../entities/case/model");
const PatientModel = require("../../../entities/patient/model");
const IndexView = require("../../../modules/index/view");
const SaveCaseView = require("../close_case/view");
const CancelCaseView = require("../cancel_case/view");
const CloseCaseView = require("../close_case/view");
const CaseView = require("../view");
const SSLQIModel = require("../../../entities/administration/sslqi_management/model");

const TabView = require("../../../components/tabs/view");
const CaseTabButtons = require("../components/case_tab_buttons");
const CaseCommonMixin = require("../components/case_common");

module.exports = Marionette.StateView.extend(_.extend(CaseCommonMixin, {
	template: require("../template.hbs"),

	regions: {
		caseFilterPanel: "#caseFilterPanel",
		tabRegion: "#tab-tops",
		buttonRegion: "#tabButtonsDiv",
		tabPanel: "#tab"
	},

	ui: {
		formxxx: "#cfForm",
		filterType: "#filterType option:selected",
		titleAddition: "#titleAddition",
		fromDate: "#fromDate",
		toDate: "#toDate",
		dateError: "#dateError",
		search: "#search",
		filteringCaret: "#filter-expand-collapse",
		// TypeAhead fields - To add another typeahead field
		// 1. Add it in this section
		// 2. Add it to typeAheadFieldsList JSON state variable further below
		// 3. Add a click (accessabiilty-wise, this might be another event later) event for the field and
		// set filterSelectedText state variable as appropriate to one of the relevant dropdown filter types. That's it.
		twitterTypeAheadSpan: ".twitter-typeahead",
		filterData: "#filterData",
		surgerySpecialty: "#surgerySpecialty"
	},

	defaultState: {
		loading: true,
		titleAddition: "",
		toggleECClass: true,
		toggleTabPanelClass: true,
		showSubtitlePanel: true,
		showCaseFilterPanel: true,
		showResultsPanel: false,
		showTabsPanel: false,
		resultsObj: new Object(),
		filterTypeValue: "surgerySpecialty",
		filterTypeText: "Specialty",
		filterDataValue: "",
		patientCBDisabled: false,
		placeholderCBDisabled: false,
		patientCBChecked: false,
		placeholderCBChecked: false,
		requestedCBChecked: false,
		scheduledCBChecked: false,
		nonORCBChecked: false,
		fromDateValue: "",
		toDateValue: "",
		showDateError: false,
		showResultMessage: false,
		resultMessage: "",
		resultMessageAttr: "ResultMsg-Error",
		currentIen: null,
		caseTypeValue: "",
		selectedTab: "tab0",

		typeAheadFilterFieldsList: [{
			"field": {
				"uiName": "filterData" }
		}, {
			"field": {
				"uiName": "surgerySpecialty" }
		}],

		filterTypes: [{
			"option": {
				"value": "surgerySpecialty",
				"text": "Specialty"}
		}, {
			"option": {
				"value": "attendingProvider",
				"text": "Provider"}
		}, {
			"option": {
				"value": "primarySurgeon",
				"text": "Staff Member"}
		},  {
			"option": {
				"value": "orRoom",
				"text": "ORs"}
		}, {
			"option": {
				"value": "sslqi",
				"text": "SSLQI"}
		}, {
			"option": {
				"value": "primaryCancelReason",
				"text": "Reasons for Unavailability"}
		}],

		caseTypes: [{
			"option": {
				"value": "Requested",
				"text": "Requested" }
		},  {
			"option": {
				"value": "Scheduled",
				"text": "Scheduled" }
		}, {
			"option": {
				"value": "Placeholder",
				"text": "Placeholder" }
		}, {
			"option": {
				"value": "Non OR",
				"text": "Non OR" }
		}]
	},
	childViewEvents: {
		"action": "formAction"
	},
	stateEvents: {
		"change": "render"
	},

	toggleFilterPanelCollapse: function() {
		this.state.set({
			showCaseFilterPanel: !this.state.get("showCaseFilterPanel"),
			toggleECClass: !this.state.get("toggleECClass"),
			toggleTabPanelClass: !this.state.get("toggleTabPanelClass")
		});
	},

	events: {
		"click #caseScreenSubTitle": function(e) {
			this.toggleFilterPanelCollapse();

			// This flag is to inform the SSLQI children views that this event has occurred.
			Radio.request("store", "set", "isExpandCollapse", true);
		},

		"click #patientCB": function (e) {
			this.state.set({
				patientCBChecked:  !this.state.get("patientCBChecked"),
				placeholderCBDisabled: !this.state.get("placeholderCBDisabled")
			});
		},

		"click #placeholderCB": function (e) {

			let phSelectedTab = this.state.get("selectedTab");
			let phCaseTypeValue = "";

			if(!this.state.get("placeholderCBChecked"))
				phCaseTypeValue = "Placeholder";

			this.state.set({
				placeholderCBChecked:  !this.state.get("placeholderCBChecked"),
				patientCBDisabled: !this.state.get("patientCBDisabled"),
				caseTypeValue: phCaseTypeValue,
				selectedTab: phSelectedTab
			});
		},

		"click #requestedCB": function (e) {

			let rCaseTypeValue = "";

			if(!this.state.get("requestedCBChecked"))
				rCaseTypeValue = "Requested";

			this.state.set({
				requestedCBChecked: !this.state.get("requestedCBChecked"),
				caseTypeValue: rCaseTypeValue,
				fromDateValue: "",
				toDateValue: ""
			});
		},

		"click #scheduledCB": function (e) {

			let sCaseTypeValue = "";

			if(!this.state.get("scheduledCBChecked"))
				sCaseTypeValue = "Scheduled";

			this.state.set({
				scheduledCBChecked: !this.state.get("scheduledCBChecked"),
				caseTypeValue: sCaseTypeValue,
				fromDateValue: "",
				toDateValue: ""
			});
		},

		"click #nonORCB": function (e) {

			let noCaseTypeValue = "";

			if(!this.state.get("nonORCBChecked"))
				noCaseTypeValue = "Non OR";

			this.state.set({
				nonORCBChecked: !this.state.get("nonORCBChecked"),
				caseTypeValue: noCaseTypeValue
			});
		},

		"change #fromDate": function (e) {
			if(this.state.get("toDateValue") != "" && e.target.value > this.state.get("toDateValue")) {
				this.state.set({
					fromDateValue: "",
					showDateError: true
				});

				this.render();
				this.getUI("dateError").fadeOut(4000, () => {
					this.state.set("showDateError", false);
					this.render();
				});

			} else {
				this.state.set("fromDateValue", this.getUI("fromDate").val());
			}
		},

		"change #toDate": function (e) {
			if(this.state.get("fromDateValue") != "" && e.target.value < this.state.get("fromDateValue")) {
				this.state.set({
					toDateValue: "",
					showDateError: true
				});

				this.render();
				this.getUI("dateError").fadeOut(4000, () => {
					this.state.set("showDateError", false);
					this.render();
				});

			} else {
				this.state.set("toDateValue", this.getUI("toDate").val());
			}
		},

		"change #filterType": function (e) {

			this.state.set({
				filterTypeValue: e.target.value,
				filterTypeText: e.target.selectedOptions[0].innerText,
				filterDataValue: ""
			});
		},

		"click #cancel": () => Radio.request("global", "navigate", "/"),

		"click #clear": function (e) {
			this.options.currentCase = new CaseModel();

			this.state.set({
				showResultMessage: false,
				showResultsPanel: false,
				showTabsPanel: false,
				titleAddition: "",
				patientCBChecked: false,
				placeholderCBChecked: false,
				requestedCBChecked: false,
				scheduledCBChecked: false,
				nonORCBChecked: false,
				fromDateValue: "",
				toDateValue: "",
				filterTypeValue: "surgerySpecialty",
				filterTypeText: "Specialty",
				filterDataValue: "",
				currentIen: null
			});

			// This would not clear if put in above set call as an empty string
			this.state.unset("filterDataValue");

			Radio.request("store", "set", "currSSLQI", new SSLQIModel());
			Radio.request("store", "set", "currentSSLQIId", null);
		},

		// ENTER key clicks Search button
		"keyup": function(e) {
			if(e.which === 13 && !this.state.get("showTabsPanel")) {
				this.getUI("search").click();
			}
		},

		"click #search": function(e) {

			let liFilterTypeValue = this.state.get("filterTypeValue");
			let liFilterDataValue = this.getUI("filterData").val();
			let liResultsObj = new Object();
			let liResultMessage = "";
			let liShowResultMessage = false;
			let liShowResultsPanel = false;
			let liResultMessageAttr = "ResultMsg-Info";

			let liFromDate = this.getUI("fromDate").val();
			let liToDate = this.getUI("toDate").val();

			if (!this.state.get("patientCBChecked") &&
				!this.state.get("placeholderCBChecked") &&
				!this.state.get("requestedCBChecked") &&
				!this.state.get("scheduledCBChecked") &&
				!this.state.get("nonORCBChecked") &&
				liFilterDataValue == "" ) {

				/*
				 // Remove error div (if there), show error message, and hide other page components
				 liResultMessage = "Please enter filter data";

				 liShowResultMessage = true;
				 liShowResultsPanel = false;
				 liResultMessageAttr = "ResultMsg-Error";
				 */

				// Prepare list object with ien as key and case model as value
				$.each(this.activeSearchCases.models, function (key, value) {
					liResultsObj[value.get("ien")] = value;
				});

			} else if((liFromDate != "" && liToDate == "") || (liFromDate == "" && liToDate != "")) {

				liResultMessage = "Please enter an appropriate date range";
				liShowResultMessage = true;
				liShowResultsPanel = false;
				liResultMessageAttr = "ResultMsg-Error";

			} else if (liFilterDataValue == "all") {

				// Prepare list object with ien as key and case model as value
				//$.each(this.activeCases.models, function (key, value) {
				$.each(this.activeSearchCases.models, function (key, value) {
					liResultsObj[value.get("ien")] = value;
				});

				// Step into here if a patient has been selected and the search is within their cases
			} else if (this.state.get("patientCBChecked")) {

				const searchResults = this.activePatientCases.models.filter((circulatingCase) => {

					let rCk = true;
					let sCk = true;
					let noCk = true;
					let dateCk = true;
					let fdCk = true;

					if(this.state.get("requestedCBChecked")) {
						rCk = circulatingCase.get("caseScheduleType") == null;
					}
					if(this.state.get("scheduledCBChecked")) {
						sCk = circulatingCase.get("caseScheduleType") != null;
					}
					if(this.state.get("nonORCBChecked")) {
						noCk = circulatingCase.get("orRoom") == null;
					}
					if(liFromDate != "" && liToDate != "") {
						dateCk = liFromDate < circulatingCase.get("caseCreateDate") && liToDate > circulatingCase.get("caseCreateDate");
					}
					if(liFilterDataValue != "") {
						fdCk = circulatingCase.get(liFilterTypeValue) == liFilterDataValue;
					}

					return rCk && sCk && noCk && dateCk && fdCk;
				});

				// Prepare list object with ien as key and case model as value
				$.each(searchResults, function (key, value) {
					liResultsObj[value.get("ien")] = value;
				});

			} else {

				const searchResults = this.activeSearchCases.models.filter((circulatingCase) => {

					let phCk = true;
					let rCk = true;
					let sCk = true;
					let noCk = true;
					let dateCk = true;
					let fdCk = true;

					if(this.state.get("requestedCBChecked"))
						{rCk = circulatingCase.get("caseScheduleType") == null;}
					if(this.state.get("scheduledCBChecked"))
						{sCk = circulatingCase.get("caseScheduleType") != null;}
					if(this.state.get("nonORCBChecked"))
						{noCk = circulatingCase.get("orRoom") == null;}
					if(liFromDate != "" && liToDate != "")
						{dateCk = liFromDate < circulatingCase.get("caseCreateDate") && liToDate > circulatingCase.get("caseCreateDate");}
					if(liFilterDataValue != "")
						{fdCk = circulatingCase.get(liFilterTypeValue) == liFilterDataValue;}

					return phCk && rCk && sCk && noCk && dateCk && fdCk;
				});

				// Prepare list object with ien as key and case model as value
				$.each(searchResults, function (key, value) {
					liResultsObj[value.get("ien")] = value;
				});
			}

			// If no search results were generated, prepare custom result message
			let liResultMessageAdditional = `your search: ["${liFilterDataValue}"]`;
			if(Object.keys(liResultsObj).length == 0 && this.state.get("patientCBChecked"))
				liResultMessageAdditional = `cases for patient: [${this.getOption("activePatient").get("name")}]`;
			else if(Object.keys(liResultsObj).length == 0 && liFromDate != "" && liToDate != "" && liFilterDataValue == "")
				liResultMessageAdditional = `the dates selected: [${liFromDate} to ${liToDate}]`;
			else if(Object.keys(liResultsObj).length == 0 && liFromDate != "" && liToDate != "" && liFilterDataValue != "")
				liResultMessageAdditional = "your search";

			if(liResultMessageAttr != "ResultMsg-Error")
				liResultMessage = `No matching results found for ${liResultMessageAdditional}`;


			// Prepare the view state based on search result data
			if (Object.keys(liResultsObj).length > 0) {
				liShowResultMessage = false;
				liShowResultsPanel = true;
			} else {
				liShowResultMessage = true;
				liShowResultsPanel = false;
				liFilterDataValue = "";
			}

			// Re-render the view with result data
			this.state.set({
				resultsObj: liResultsObj,
				resultMessage: liResultMessage,
				showResultsPanel: liShowResultsPanel,
				showTabsPanel: false,
				showResultMessage: liShowResultMessage,
				resultMessageAttr: liResultMessageAttr,
				fromDateValue: liFromDate,
				toDateValue: liToDate,
				filterDataValue: liFilterDataValue
			});

			// Reset subtitle
			this.getUI("titleAddition").text("").removeClass("cf-title-addition");
		},

		"click #list": function(e) {

			// Set up a CaseModel object for the current case and put it into options
			let selectedCaseObj = this.state.get("resultsObj")[e.target.id];
			this.options.currentCase = new CaseModel(selectedCaseObj.attributes);
			let fetchOpts = authHeaders();
			fetchOpts.async = false;
			this.options.currentCase.fetch(fetchOpts);
			this.options.currentCase.fetchExtended();

			console.log("SELECTING CASE:", this.options.currentCase.id);
			Radio.request("store", "set", "activeCase", this.options.currentCase);
			Radio.request("store", "set", "activePatient", this.options.currentCase.getPatient());

			// Check whether current case is scheduled
			let liScheduledCBChecked = false;
			if(selectedCaseObj.attributes.caseScheduleType != null)
				liScheduledCBChecked = true;

			// Check whether a new case was selected. Used by SSLQI form downstream.
			let storedIen = Radio.request("store", "get", "currentIen");
			if(storedIen === undefined || storedIen === e.target.id)
				Radio.request("store", "set", "isNewIen", false);
			else
				Radio.request("store", "set", "isNewIen", true);

			Radio.request("store", "set", "currentIen", e.target.id);


			// TODO: #584 - Need one of the above for requested case but have no Case model field to base it on
			//let liRequestedCBChecked = false;
			//if(selectedCaseObj.attributes.caseType != null)
			//	liRequestedCBChecked = true;
			console.log("click list:", e.target);

			// Show the actual tab panel below the case case and display the principal procedure near title
			// Collapse the list selector when the case is expanded
			this.toggleFilterPanelCollapse();
			this.state.set({

				currentIen: e.target.id,
				titleAddition: `${e.target.id} - ${this.getOption("currentCase").get("principalProcedure")}`,
				scheduledCBChecked: liScheduledCBChecked,
				showTabsPanel: true
			});
		},

		"click #surgerySpecialty": function (e) {
			this.state.set("filterTypeText", "Specialty");
		}
	},

	templateContext() {
		const activePatient = this.getOption("activePatient");
		const activeCases = this.getOption("activeCases");
		const activeSearchCases = this.getOption("activeSearchCases");
		const activePatientCases = this.getOption("activePatientCases");
		const currentCase = this.getOption("currentCase");

		let patientName = undefined;
		if (currentCase) {
			let patient = currentCase.getPatient();
			if (patient) patientName = patient.get("name");
		}
		const ctx = {
			activePatient: activePatient ? activePatient.toJSON() : undefined,
			currentCase: currentCase ? currentCase.toJSON() : undefined
		};

		console.log("templateContext from case_list view:\n", ctx);
		return ctx;
	},

	/**
	 * Order of execution:
	 * 1: initialize
	 * 2: onRender
	 * 3: onAttach
	 * 4: onDomRefresh
	 */
	initialize: function() {

		// Needed when using StateView - calls its constructor
		Marionette.StateView.prototype.initialize.apply(this, arguments);

		this.room_types = new SpecialtiesCollection();
		this.room_types.fetch(authHeaders());

		// Collect relevant data from Radio channel
		Object.assign(this.options, {
			activePatient: Radio.request("store", "get", "activePatient"),
			activeCases: new CaseCollection(),
			activeSearchCases: new CaseSearchCollection(),
			activePatientCases: new PatientCaseCollection(),
			currentCase: new CaseModel()
		});

		// This flag is for the SSLQI children views. Setting it to true allows the SSLQI form to be displayed
		// upon clicking on an SSLQI from the search list.
		Radio.request("store", "set", "isExpandCollapse", true);
	},

	onAttach: function() {
		this.ui.formxxx.show();

		this.listenTo(Radio.channel("store"), {
			"change:activePatient": this.updatePatient
		}, this);

		this.loadCaseList();
		this.loadSearchCaseList();
	},

	onRender: function () {
		const selectedTab = this.state.get("selectedTab");

		this.$("#toDate,#fromDate").datetimepicker({
			format: "MM/DD/YYYY",
			formatDate: "MM/DD/YYYY"
		});
		if (this.state.get("toggleTabPanelClass")) {
			this.$("#tabPanel").addClass("tab-panel-top-margin slide-up");
		}

		const caret = this.getUI("filteringCaret");
		if (this.state.get("toggleECClass")) {
			caret.addClass("fa-caret-down").removeClass("fa-caret-right");
		} else {
			caret.removeClass("fa-caret-down").addClass("fa-caret-right");
		}

		this.state.get("toggleECClass") || this.$("#caseFilterPanel").hide();

		// After this everything is for the actual case view, so skip if not showing case.
		if (!this.state.get("showTabsPanel")) { return; }

		this.showChildView("tabRegion", new TabView({
			tabs: this.tabs,
			selected: selectedTab
		}));

		this.showChildView("tabPanel", this.getTabPanel(selectedTab));

		// TODO: Hide Time and Staff (ref tab3) when placeholder.
		let cancelOrClose = "";
		if (this.state.get("requestedCBChecked")) {
			cancelOrClose = "close";
		}

		if (this.state.get("scheduledCBChecked")) {
			cancelOrClose = "cancel";
		}

		if (selectedTab !== "tab0" && selectedTab !== "tab1" && this.state.get("showTabsPanel")) {
			this.showChildView("buttonRegion", new CaseTabButtons({
				closeOrCancel: cancelOrClose
			}));
		}
	},

	/**
	 * Method called when view is first displayed, fetching the full case of cases for selection (GET)
	 */
	loadCaseList() {
		this.activeCases = new CaseCollection();
		return this.activeCases.fetch(authHeaders())
			.then(() => {
				console.log("Active Cases - ", this.activeCases);
				this.state.set("loading", false);
			});
	},

	/**
	 * Method called when view is first displayed, fetching the full case of cases for selection (POST)
	 * An empty input gets back all cases.
	 */
	loadSearchCaseList() {
		let postInput = new CaseSearchModel();
		return postInput.save(null, authHeaders())
			.then(() => {
				this.activeSearchCases = new CaseSearchCollection(postInput.get("results"));
				this.initFilterTypeAhead();
			});
	},

	/**
	 * Method for calling cases pertaining to a patient
	 */
	loadPatientCaseList() {
		this.activePatientCases = new CaseCollection();
		this.activePatientCases.url = API.getURL("PatientId", `${this.getOption("activePatient").get("dfn")}/cases`);

		// TODO: You are going to have to append {patientID}/cases to the URL for this API before making the fetch call

		return this.activePatientCases.fetch(authHeaders())
			.then(() => console.log(this.activePatientCases));
	},

	initFilterTypeAhead() {

		if(this.activeSearchCases != undefined && this.activeSearchCases.length > 0) {

			//let activeCaseModels = this.activeCases.models;
			let activeCaseModels = this.activeSearchCases.models;
			let typeaheadQuery = null;
			let filterTypeSelected = this.state.get("filterTypeText");

			$.each(this.state.get("typeAheadFilterFieldsList"), (key, value) => {
				this.getUI(value.field.uiName).typeahead({
					minLength: 3,
					highlight: true
				},
					{
						name: "active-case-data",
						source: function (query, callback) {
							// The typeahead query will have to be known globally
							typeaheadQuery = query;

							// Screens out CaseModel objects which match the query above
							const results = activeCaseModels.filter((activeCase) => {

								let ck1 = null;
								let ck2 = null;
								let ck3 = null;
								let ck4 = null;

								// This code block returns either true or false
								if (filterTypeSelected == "Specialty") {
									ck1 = activeCase.get("surgerySpecialty").toLowerCase().indexOf(query.toLowerCase()) > -1;
									return ck1;
								} else if (filterTypeSelected == "Provider") {
									ck1 = activeCase.get("attendingProvider").toLowerCase().indexOf(query.toLowerCase()) > -1;
									return ck1;
								} else if (filterTypeSelected == "Staff Member") {
									ck1 = activeCase.get("primarySurgeon").toLowerCase().indexOf(query.toLowerCase()) > -1;
									ck2 = activeCase.get("attendingSurgeon").toLowerCase().indexOf(query.toLowerCase()) > -1;
									ck3 = activeCase.get("firstAsst").toLowerCase().indexOf(query.toLowerCase()) > -1;
									ck4 = activeCase.get("secondAsst").toLowerCase().indexOf(query.toLowerCase()) > -1;
									return ck1 | ck2 | ck3 | ck4;
								} else if (filterTypeSelected == "ORs") {
									ck1 = activeCase.get("orRoom").toLowerCase().indexOf(query.toLowerCase()) > -1;
									return ck1;
								} else if (filterTypeSelected == "Reasons for Unavailability") {
									ck1 = activeCase.get("primaryCancelReason").toLowerCase().indexOf(query.toLowerCase()) > -1;
									return ck1;
								}

							});

							let theMap = null;

							// Debug
							//console.log("Matching active cases", results);

							// Certain filter criteria require searching through more than one data item inside each case model object
							// filtered through to results above. When that happens, as with Staff Member, each data item searched for is
							// put into an temporary array, and then all the temporary arrays are lumped into one. Then all empty elements
							// in the consolidated array are gotten rid of. For filter criteria with one case model data item, its
							// processing is simpler.
							if (filterTypeSelected == "Specialty")
								theMap = results.map(r => (r.get("surgerySpecialty")));
							else if (filterTypeSelected == "Provider")
								theMap = results.map(r => (r.get("attendingProvider")));
							else if (filterTypeSelected == "Staff Member") {

								// Need to determine which one of the filter criteria actually contains the query
								let tempMap1 = results.map(r => ( (r.get("primarySurgeon").toLowerCase().indexOf(typeaheadQuery.toLowerCase()) > -1) ? r.get("primarySurgeon") : ""));
								let tempMap2 = results.map(r => ( (r.get("attendingSurgeon").toLowerCase().indexOf(typeaheadQuery.toLowerCase()) > -1) ? r.get("attendingSurgeon") : ""));
								let tempMap3 = results.map(r => ( (r.get("firstAsst").toLowerCase().indexOf(typeaheadQuery.toLowerCase()) > -1) ? r.get("firstAsst") : ""));
								let tempMap4 = results.map(r => ( (r.get("secondAsst").toLowerCase().indexOf(typeaheadQuery.toLowerCase()) > -1) ? r.get("secondAsst") : ""));
								let tempMap = [tempMap1, tempMap2, tempMap3, tempMap4];
								theMap = [].concat.apply([], tempMap);

								// Get rid of empty array elements
								theMap = theMap.filter(function (n) {
									return n;
								});

							} else if (filterTypeSelected == "ORs")
								theMap = results.map(r => (r.get("orRoom")));
							else if (filterTypeSelected == "Reasons for Unavailability")
								theMap = results.map(r => (r.get("primaryCancelReason")));

							callback(theMap);
						}
					});
			});
		}
	}
}));
