const Backbone = require("backbone");
const Marionette = require("backbone.marionette");
const Radio = require("backbone.radio");
const authHeaders = require("../../../utils/authHeaders");
const IndexView = require("../../../modules/index/view");
const CancelReasonsCollection = require("../../../entities/case/cancel_reasons/collection");
const TooltipBehavior = require("../../../behaviors/tooltip");

import ListsSelect from "../../../utils/entity-select/lists";
import FormView from "../../../components/form/view";

const CancelForm = FormView.extend({
	formFields: [
		{
			label: "Primary Cancelation Reason",
			field: "primaryCancelReason",
			type: "select",
			selSrc: ListsSelect.CaseCancelReasons
		}, {
			label: "Comments",
			field: "cancelComments",
			type: "text-area",
			width: 12
		}
	],
	className: "case-tab-panel",
	onRender() {
		this.$(".form-group").removeClass("row");
	}
});

module.exports = Marionette.View.extend({
	template: require("./template.hbs"),

	regions: {
		cancelForm: "#cancel-form"
	},

	ui: {
		ok: "#ok",
		cancel: "#cancel",
		close: "#close",
		question: "#question"
	},
	triggers: {
		"click @ui.ok": "ok",
		"click @ui.cancel": "cancel",
		"click @ui.close": "close",
		"click @ui.question": "question"
	},
	behaviors: { TooltipBehavior },
	onOk() {
		Radio.request("root", "close:modal");
		if (this.model.get("primaryCancelReason")) {
			this.model.set("cancelDate", (new Date()).toISOString());
			let saveOpts = authHeaders();
			saveOpts.success = () => {
				Radio.request("root", "dialog", {
					type: "info",
					title: "Canceled Case",
					message: "Case Canceled.  You may continue."
				});
			};
			saveOpts.error = () => {
				Radio.request("root", "dialog", {
					type: "error",
					title: "Error Saving Canceled Case",
					message: "Please try again."
				});
			};
			this.model.save(null, saveOpts);
		}
		Radio.request("global", "navigate", "/");
	},

	onCancel() {
		Radio.request("root", "close:modal");
	},

	onClose() {
		Radio.request("root", "close:modal");
	},

	onQuestion() {
		Radio.request("root", "close:modal");
		let href = "https://ehmp.vistacore.us/help/eHMP_Page%20Not%20Found.htm";
		let left = (screen.width/2)-(900/2);
		let top = (screen.height/2)-(600/2);
		window.open(href, "Widget Help", `top=${top}, left=${left}, width=800, height=450`);
	},

	/**
	 * Order of execution:
	 * 1: initialize
	 * 2: onRender
	 * 3: onAttach
	 * 4: onDomRefresh
	 */
	initialize() {

		// Collect relevant data from Radio channel
		Object.assign(this.options, {
			cancelReasons: new CancelReasonsCollection()
		});
	},

	onRender() {
		this.showChildView("cancelForm", new CancelForm({
			model: this.model
		}));
	},

	onAttach() {
		console.log("Attaching Cancel view");
	}
});
