const Backbone = require("backbone");
const Marionette = require("backbone.marionette");
const Radio = require("backbone.radio");
const _ = require("underscore");
const authHeaders = require("../../../utils/authHeaders");
const API = require("../../../api");
const CaseSearchModel = require("../../../entities/case/search/model");
const CaseSearchCollection = require("../../../entities/case/search/collection");
require("bootstrap-select");
import SearchBySelect from "../../../utils/entity-select/search_by";
import PeopleSelect from "../../../utils/entity-select/people";
import ListsSelect from "../../../utils/entity-select/lists";

module.exports = Marionette.StateView.extend({

	template: require("./template.hbs"),

	regions: {
		searchByField: "#search-by-field",
		selectedField: "#selected-field"
	},

	ui: {
		formxxx: "#calendarSearchForm",
		selectFieldLabel: "#selectFieldLabel"
	},

	defaultState: {
		showSelectedFieldLabel: false,
		selectedProviderId: null,
		startDate: null,
		endDate: null
	},

	stateEvents: {
		"change": "render"
	},

	events: {

		"change #search-by-field":  function (e) {

			// Reset calendar
			//this.getOption("parent").loadGraphicalCalendar(null, null, null);

			if(e.target.value === "all") {
				$("#selected-field").hide();
				$("label[for='selected-field']").hide();
				this.loadSearchCaseList();
			} else if(e.target.value === "staffMember") {
				$("label[for='selected-field']").text("Staff Member").show();
				this.showChildView("selectedField", new PeopleSelect());
				$("#selected-field").show();
			} else if(e.target.value === "provider") {
				$("label[for='selected-field']").text("Provider").show();
				this.showChildView("selectedField", new PeopleSelect());
				$("#selected-field").show();
			} else {
				$("#selected-field").hide();
				$("label[for='selected-field']").hide();
			}
		},

		"change #selected-field":  function (e) {
			this.state.set({selectedProviderId: e.target.value}, {silent: true});
			this.getActiveCasesForProvider();
		}
	},

	templateContext() {

		const activeSearchCases = this.getOption("activeSearchCases");
		const calendarCases = this.getOption("calendarCases");
		const activeCasesForProvider = this.getOption("activeCasesForProvider");
		const workTimesForProvider = this.getOption("workTimesForProvider");
		const allWorkTimesForProvider = this.getOption("allWorkTimesForProvider");
		const parent = this.getOption("parent");

		const ctx = {
			//activeCasesForProvider: activeCasesForProvider ? activeCasesForProvider.toJSON() : undefined,
			//workTimesForProvider: workTimesForProvider ? workTimesForProvider.toJSON() : undefined,
			//parent: parent
		};

		console.log("Rendering people from calendar view:\n", ctx);
		return ctx;
	},

	initialize: function() {
		// Needed when using StateView - calls its constructor
		Marionette.StateView.prototype.initialize.apply(this, arguments);
	},

	onRender() {
		this.showChildView("searchByField", new SearchBySelect());
	},

	onAttach() {
		this.loadSearchCaseList();
	},

	/**
	 * Method called when calendar first loads. It loads all cases in the calendar across all
	 * time for now.
	 */
	loadSearchCaseList() {
		let searchInput = new CaseSearchModel();
		return searchInput.save(null, authHeaders())
			.then(() => {
				this.activeSearchCases = new CaseSearchCollection(searchInput.get("results"));
				this.workTimesForProvider = null;
				this.getAllCurrentWeekProviderWorkSchedules();
			});
	},

	/**
	 * Method called after getActiveCasesForProvider which would have been called when a user
	 * selects a staff member
	 */
	getAllCurrentWeekProviderWorkSchedules() {

		this.setCurrentWeekDateRange();
		let searchFromDate = this.state.get("startDate");
		let searchToDate = this.state.get("endDate");
		let calendarCases = this.qualityAssureCases(this.activeSearchCases);

		let AllCurrentWeekProviderWorkScheduleModel = Backbone.Model.extend({
			//url: API.getURL("Provider") + "/" + providerId + "/" + searchFromDate + "/" + searchToDate,
			url: API.getLocalURL("AllProviderWorkTimesLocal"),
			defaults: {
				"date": "",
				"schedule": []
			},

			parse(data) {
				return _.isArray(data) ? { results: data } : data;
			}
		});

		this.allWorkTimesForProvider = new AllCurrentWeekProviderWorkScheduleModel();
		return this.allWorkTimesForProvider.fetch(authHeaders())
			.then(() => {
				this.getOption("parent").loadGraphicalCalendar(calendarCases, this.workTimesForProvider, this.allWorkTimesForProvider);
			});
	},

	/**
	 * Method called when a user selects a staff member or provider as of now.
	 */
	getActiveCasesForProvider() {

		let providerId = this.state.get("selectedProviderId");
		this.setCurrentWeekDateRange();
		let searchFromDate = this.state.get("startDate");
		let searchToDate = this.state.get("endDate");

		this.activeCasesForProvider = new CaseSearchCollection();
		this.activeCasesForProvider.url = API.getURL("CaseByProvider") + "/" + providerId + "/" + searchFromDate + "/" + searchToDate;
		return this.activeCasesForProvider.fetch(authHeaders())
			.then(() => {
				Radio.request("store", "set", "activeCasesForProvider", this.activeCasesForProvider);
				this.calendarCases = this.qualityAssureCases(this.activeCasesForProvider);

				if(this.calendarCases.models.length === 0) {
					this.getOption("parent").loadGraphicalCalendar(null, null, this.allWorkTimesForProvider);
					this.showResultMessage("info", "Search Result", "No case results matched your search");
				} else {
					this.getCurrentWeekProviderWorkSchedule(providerId);
				}
			});
	},

	/**
	 * Method called from getActiveCasesForProvider which would have been called when a user
	 * selects a staff member
	 */
	getCurrentWeekProviderWorkSchedule(providerId) {

		let searchFromDate = this.state.get("startDate");
		let searchToDate = this.state.get("endDate");

		let CurrentWeekProviderWorkScheduleModel = Backbone.Model.extend({
			//url: API.getURL("Provider") + "/" + providerId + "/" + searchFromDate + "/" + searchToDate,
			url: API.getLocalURL("ProviderWorkTimesLocal"),
			defaults: {
				"start": "",
				"end": ""
			},

			parse(data) {
				return _.isArray(data) ? { results: data } : data;
			}
		});

		this.workTimesForProvider = new CurrentWeekProviderWorkScheduleModel();
		return this.workTimesForProvider.fetch(authHeaders())
			.then(() => {
				Radio.request("store", "set", "activeCasesForProvider", this.workTimesForProvider);

				// Now call the parent to re-render the graphical calendar view with blocked times for this provider
				//this.getOption("parent").loadGraphicalCalendar(this.calendarCases, this.workTimesForProvider, null);

				this.getAllCurrentWeekProviderWorkSchedules();
			});
	},



	/**
	 * Method that calculates the current week's date range (i.e., from Sunday to Saturday) and
	 * stores the calculated dates in a state variable.
	 */
	setCurrentWeekDateRange() {

		// Figure out difference between today and start of current week, Sunday
		let today = new Date();
		let day = today.getDay(),
			diff = today.getDate() - day;

		// Change Sunday start date into proper format (i.e., yyyy-mm-dd)
		let sundayDate = new Date(today.setDate(diff));
		let sundayDayOfMonth = sundayDate.getDate();
		let sundayMonth = ("0" + (sundayDate.getMonth() + 1)).slice(-2);
		let sundayYear = sundayDate.getFullYear();
		this.state.set({startDate: sundayYear + "-" + sundayMonth + "-" + sundayDayOfMonth}, {silent: true});

		// Now calculate end of week day, Saturday, and change into proper format (i.e., yyyy-mm-dd)
		let nextSaturdayDate = sundayDate;
		nextSaturdayDate.setDate(nextSaturdayDate.getDate() + 6);
		let saturdayDayOfMonth = nextSaturdayDate.getDate();
		let saturdayMonth = ("0" + (nextSaturdayDate.getMonth() + 1)).slice(-2);
		let saturdayYear = nextSaturdayDate.getFullYear();
		this.state.set({endDate: saturdayYear + "-" + saturdayMonth + "-" + saturdayDayOfMonth}, {silent: true});
	},

	/**
	 * Weed out cases with no scheduled start or end times
	 */
	qualityAssureCases(currentCollection) {

		let calendarCases = new CaseSearchCollection();

		$.each(currentCollection.models, (key, value) => {
			if(value.get("scheduledStartTime") !== undefined && value.get("scheduledStartTime") !== null && value.get("scheduledStartTime") !== "" &&
				value.get("scheduledEndTime") !== undefined && value.get("scheduledEndTime") !== null && value.get("scheduledEndTime") !== "") {

				// Remove current element from activeSearchCases if no start and end times
				calendarCases.models.push(value);
			}
		});

		return calendarCases;
	},

	/**
	 * Method that shows popup result message display
	 */
	showResultMessage(type, title, message) {

		Radio.request("root", "dialog", {
			type: type,
			title: title,
			message: message
		});
	}
});
