const Marionette = require("backbone.marionette");
const Radio = require("backbone.radio");
const authHeaders = require("../../../../utils/authHeaders");
const SSLQIModel = require("../../../../entities/administration/sslqi_management/model");
const SSLQICollection = require("../../../../entities/administration/sslqi_management/collection");
const SSLQISearchModel = require("../../../../entities/administration/sslqi_management/search/model");
const SSLQISearchCollection = require("../../../../entities/administration/sslqi_management/search/collection");
const SSLQIReservationModel = require("../../../../entities/administration/sslqi_management/reservation/model");
const SSLQIReservedCollection = require("../../../../entities/administration/sslqi_management/reservation/collection");
const SSLQITabContentPanel = require("../../../case/components/sslqi_tab_content_panel");
const NewSSLQIView = require("../view");

module.exports = Marionette.StateView.extend({

	template: require("./template.hbs"),

	regions: {
		tabPanelSSLQIRegion: "#tabPanelSSLQI"
	},

	ui: {
		formxxx: "#sslqiSearchForm",
		category: "#category",
		item_Type: "#item_Type", // Field id is "item_Type" to not to conflict with its tabbed form counterpart of id "itemType"
		itemSpecialty: "#itemSpecialty",
		searchResultsList: "#list",
		reservedSearchResultsList: "#reservedlist"
	},

	defaultState: {
		showTabsPanel: false,
		activeFilter: "category",
		item_TypeDisabled: true,
		itemSpecialtyDisabled: true,
		topTableVisible: false,
		categoryValue: "",
		item_TypeValue: "",
		itemSpecialtyValue: "",
		current_id: null,
		resultsObj: new Object(),
		searchResults: null,
		childAction: "",
		listScrollHeight: 0,
		reservedListScrollHeight: 0,
		doClientSideSearch: false,
		c1stOptionDisabled: false,
		it1stOptionDisabled: false,
		its1stOptionDisabled: false,
		childIsLoaded: false
	},

	stateEvents: {
		"change": "render"
	},

	events: {

		"click #category":  function (e) {

			// Detects click on same option - a different option clicked triggers change event
			if(e.offsetY < 0 && e.target.value !== "all"){
				this.state.set({
					item_TypeDisabled: false,
					categoryValue: e.target.value,
					showTabsPanel: false
				});

				Radio.request("store", "set", "sslqiShowTabsPanel", false);
			} else {
				// This code block executes when a user clicks on the select box to expand it
				Radio.request("store", "set", "sslqiSearchResults", null);
			}
		},

		"change #category":  function (e) {

			this.state.set({c1stOptionDisabled: true}, {silent: true});

			if (e.target.value !== "all" && e.target.value !== "Select Category") {
				this.state.set({
					item_TypeDisabled: false,
					categoryValue: this.getUI("category").val(),
					showTabsPanel: false
				});

				this.getUI("item_Type").focus();
				Radio.request("store", "set", "sslqiShowTabsPanel", false);

			} else if (e.target.value.toLowerCase() === "all") {
				this.state.set({
					categoryValue: e.target.value.toLowerCase(),
					item_TypeDisabled: true,
					itemSpecialtyDisabled: true,
					item_TypeValue: "",
					itemSpecialtyValue: "",
					it1stOptionDisabled: false,
					its1stOptionDisabled: false,
					showTabsPanel: false
				});

				Radio.request("store", "set", "sslqiShowTabsPanel", false);
				this.search();
			}
		},

		"click #item_Type":  function (e) {

			// Detects click on same option - a different option clicked triggers change event
			if(e.offsetY < 0 && e.target.value !== "Select Item Type"){

				this.state.set({
					itemSpecialtyDisabled: false,
					item_TypeValue: this.getUI("item_Type").val(),
					showTabsPanel: false
				});

				Radio.request("store", "set", "sslqiShowTabsPanel", false);
			} else {
				// This code block executes when a user clicks on the select box to expand it
				Radio.request("store", "set", "sslqiSearchResults", null);
			}
		},

		"change #item_Type":  function (e) {

			if (this.getUI("category").val().toLowerCase() !== "all" && e.target.value !== "Select Item Type") {
				this.state.set({
					it1stOptionDisabled: true,
					itemSpecialtyDisabled: false,
					item_TypeValue: this.getUI("item_Type").val(),
					showTabsPanel: false
				});

				this.getUI("itemSpecialty").focus();
				Radio.request("store", "set", "sslqiShowTabsPanel", false);
			}
		},

		// Detect if option currently selected has been clicked again (no change event would be triggered by this)
		"click #itemSpecialty":  function (e) {

			// Detects click on same option - a different option clicked triggers change event
			if(e.offsetY < 0){
				//this.search();
			} else {
				// This code block executes when a user clicks on the select box to expand it
				Radio.request("store", "set", "sslqiSearchResults", null);
			}
		},

		"change #itemSpecialty":  function (e) {
			this.state.set({its1stOptionDisabled: true}, {silent: true});
			this.search();
		},

		"click #list": function(e) {

			// Keep the scroll position of the list where it was
			this.state.set({
				listScrollHeight: this.getUI("searchResultsList").scrollTop()
			}, {silent: true});

			// Set up a SSLQIModel object for the current SSLQI and put it into options
			let selectedSSLQIObj = this.state.get("resultsObj")[e.target.id];
			Object.assign(this.options, {
				currentSSLQI: new SSLQIModel(selectedSSLQIObj.attributes)
			});

			// Enable this flag in order to show the view which goes through one onRender event. When first loaded,
			// this view goes through two of the onRender events.
			this.state.set({childIsLoaded: true},{silent: true});

			// Whatever selection is made in the SSLQI in the SSLQI search list, find the corresponding SSLQI
			// Reservation record based on the composite vistaId and classification foreign key. Store the matching
			// SSLQI Reservation record if there is one for later data population in the Reservation tab fields.
			if(this.getOption("isChildView")) {

				// Store various states for keeping state across expanding/collapsing, new case selection, and
				// navigating away from and back into tabs.
				Radio.request("store", "set", "currSSLQI", new SSLQIModel(selectedSSLQIObj.attributes)); // To be used in collapse/expansion of Case Filtering reload of this view
				Radio.request("store", "set", "sslqiShowTabsPanel", true);
				Radio.request("store", "set", "currentSSLQIId", e.target.id); // To be used in collapse/expansion of Case Filtering reload of this view
				Radio.request("store", "set", "activeReservedSSLQI", new SSLQIReservationModel()); // Reset this object in case user navigates to another row in SSLQI search list

				$.each(this.activeReservedSSLQIs.models, (key, value) => {
					if(value.get("vistaId") === selectedSSLQIObj.attributes.vistaId &&
						value.get("classification") === selectedSSLQIObj.attributes.classification) {

						Radio.request("store", "set", "activeReservedSSLQI", new SSLQIReservationModel(value.attributes));
					}
				});
			}

			// Show the actual tab panel below the case case and display the principal procedure near title
			this.state.set({
				current_id: e.target.id,
				showTabsPanel: true
			});
		},

		"click #newSSSLQILink":  function (e) {
			//Backbone.history.navigate( "/", { trigger: false, replace: true } );
			Radio.request("root", "body", new NewSSLQIView({}));
		}

		// TODO: Need to implement ENTER key to trigger search if it's itemSpecialty and a user
		// has arrowed down to selection
		// Pressing ENTER key triggers search
		//"keyup": function(e) { if(e.which === 13) { this.search(); } }
	},

	templateContext() {
		const activeSSLQIs = this.getOption("activeSSLQIs");
		const activeReservedSSLQIs = this.getOption("activeReservedSSLQIs");
		const currentSSLQI = this.getOption("currentSSLQI");
		const searchFormStructure = this.getOption("searchFormStructure");

		// Need this here for title not showing up if this screen is a child view
		const isChildView = this.getOption("isChildView");

		const ctx = {
			activeSSLQIs: activeSSLQIs ? activeSSLQIs.toJSON() : undefined,
			activeReservedSSLQIs: activeReservedSSLQIs ? activeReservedSSLQIs.toJSON() : undefined,
			currentSSLQI: currentSSLQI ? currentSSLQI.toJSON() : undefined,

			//TODO: Put this into a data_files JSON file retrievable through API call?
			searchFormStructure: {
				"fields": {
					"column1": [{
						"id": "category",
						"type": "select",
						"modelField": "category",
						"label": "Category",
						"title": "Enter category - \"All\" retrieves all SSLQIs",
						"placeholder": "Enter category"
					},
					{
						"id": "item_Type",
						"type": "select",
						"modelField": "itemType",
						"label": "Item Type",
						"title": "Enter item type",
						"placeholder": "Enter item type"
					},
					{
						"id": "itemSpecialty",
						"type": "select",
						"modelField": "itemSpecialty",
						"label": "Item Specialty",
						"title": "Enter item specialty - its selection will generate search results",
						"placeholder": "Enter item specialty"
					}]
				}
			},

			isChildView: isChildView
		};

		console.log("Rendering from sslqi_management view:\n", ctx);
		return ctx;
	},

	// Called first once at the beginning
	initialize: function() {

		// Needed when using StateView - calls its constructor
		Marionette.StateView.prototype.initialize.apply(this, arguments);
	},

	// Called every time screen changes state. It's called twice when first loaded just to populate drop downs.
	onRender: function () {

		// The childAction state value would have been set by the child view's delete action if it executed. A parent
		// "this" object is passed down to the child to have access to this state variable.
		if(this.state.get("childAction") === "delete") {
			Object.assign(this.options, {
				currentSSLQI: new SSLQIModel()
			});

			Radio.request("store", "set", "sslqiShowTabsPanel", false);
			Radio.request("store", "set", "currentSSLQIId", null);

			this.state.set({
				showTabsPanel: false,
				current_id: null,
				childAction: "read"
			}, {silent: false});
		}

		if (this.state.get("showTabsPanel")) {

			// This ensures that the child view is loaded just once (onRender being called twice to allow
			// drop downs to populate - remember this view is itself a child view the Cases (search) screen)
			if(this.state.get("childIsLoaded")) {
				this.showChildView("tabPanelSSLQIRegion", new SSLQITabContentPanel({
					currentSSLQI: this.options.currentSSLQI,
					parent: this,
					isNewSSLQI: false
				}));
			} else {
				this.state.set({childIsLoaded: true},{silent: true}); // Make it so it loads on second pass-through
			}
		}

		// Keep the scroll position of the list where it was - listScrollHeight was captured before rendering
		this.getUI("searchResultsList").scrollTop(this.state.get("listScrollHeight"));

		//if(this.getOption("isChildView")) {
		//	this.getUI("reservedSearchResultsList").scrollTop(this.state.get("reservedListScrollHeight")); // Not being used yet
		//}
	},

	// Called at the beginning after first loading screen DOM
	onAttach: function() {
		this.ui.formxxx.show();

		this.getSSLQIs();

		// If there are search results already in storage and it's the child view SSLQI screen inside a Cases screen,
		// show them. If you want to maintain this state through out all SSLQI screens, then comment out the first if.
		if(this.getOption("isChildView")) {

			this.searchResults = Radio.request("store", "get", "sslqiSearchResults");
			if (this.searchResults !== undefined &&
					this.searchResults !== null &&
					this.searchResults.length > 0) {
				this.activeSSLQIs = Radio.request("store", "get", "activeSSLQIs");

				let liCategoryValue = Radio.request("store", "get", "sslqiItemCategoryValue");
				let liItem_TypeValue = Radio.request("store", "get", "sslqiItemTypeValue");
				let liItemSpecialtyValue = Radio.request("store", "get", "sslqiItemSpecialtyValue");
				let liItem_TypeDisabled = false;
				let liItemSpecialtyDisabled = false;

				liCategoryValue = (liCategoryValue !== null && liCategoryValue !== "") ? liCategoryValue : "";
				liItem_TypeValue = (liItem_TypeValue !== null && liItem_TypeValue !== "") ? liItem_TypeValue : "";
				liItemSpecialtyValue = (liItemSpecialtyValue !== null && liItemSpecialtyValue !== "") ? liItemSpecialtyValue : "";

				if(liCategoryValue === "all") {
					liItem_TypeDisabled = true;
					liItemSpecialtyDisabled = true;
				}

				// If it's a new case, it will cycle through here, then go to the case list view #list
				// clicking event where the isNewIen flag is set, and then it will come back to this
				// block of code where currentSSLQIId is set to null. Otherwise, just get its value from
				// storage.
				let currentSSLQIId = Radio.request("store", "get", "currentSSLQIId");
				if (Radio.request("store", "get", "isNewIen")) {
					currentSSLQIId = null;
				}

				this.state.set({
					topTableVisible: true,
					categoryValue: liCategoryValue,
					item_TypeValue: liItem_TypeValue,
					itemSpecialtyValue: liItemSpecialtyValue,
					item_TypeDisabled: liItem_TypeDisabled,
					itemSpecialtyDisabled: liItemSpecialtyDisabled,
					current_id: currentSSLQIId,
					showTabsPanel: Radio.request("store", "get", "sslqiShowTabsPanel")
				}, {silent: true});
			}
		}
	},

	/**
	 * Method called when view is first displayed, fetching the full list of SSLQIs using the GET method
	 */
	getSSLQIs() {
		this.activeSSLQIs = new SSLQICollection();
		return this.activeSSLQIs.fetch(authHeaders())
			.then(() => {
				Radio.request("store", "set", "activeSSLQIs", this.activeSSLQIs);
				this.refreshSearchResults();
				if(this.getOption("isChildView"))
					this.getReservedSSLQIs();
				else
					this.render(); // This is a double render to get drop-downs to populate (1st render is from refreshSearchResults)
			});
	},

	/**
	 * Method called when view is first displayed if it's a child view, fetching SSLQIs associated with a caseId
	 */
	getReservedSSLQIs() {
		this.activeReservedSSLQIs = new SSLQIReservedCollection();
		return this.activeReservedSSLQIs.fetch(authHeaders())
			.then(() => {

				Radio.request("store", "set", "activeReservedSSLQIs", this.activeReservedSSLQIs);
				let liResultsObj = new Object();
				let activeCase = Radio.request("store", "get", "activeCase");
				let currentCaseId = parseInt(activeCase.get("ien"));
				let orRoom = activeCase.get("orRoom");
				let allActiveSSLQIs = Radio.request("store", "get", "activeSSLQIs");

				$.each(this.activeReservedSSLQIs.models, (key, value) => {

					// Get only SSLQIs associated with active case
					if(value.attributes["caseId"] === currentCaseId) {

						let r_id = value.get("_id");
						let rVistaId = value.get("vistaId");
						let rClassification = value.get("classification");

						liResultsObj[r_id] = value;

						// Get the item category that will serve as the description field for the case
						// SSLQI list. Also, get the orRoom from the active case. It will be the same as
						// the SSLQI reservation room which is read only field in the Reservation tab form.
						$.each(allActiveSSLQIs.models, (key, value) => {

							let aVistaId = value.get("vistaId");
							let aClassification = value.get("classification");

							if(rVistaId === aVistaId && rClassification === aClassification) {
								liResultsObj[r_id].attributes.itemCategory = value.get("itemCategory");

								// Set SSLQI reservation room with active case orRoom if it has a value
								if(orRoom !== undefined && orRoom !== null && orRoom !== "")
									liResultsObj[r_id].attributes.room = orRoom;
							}
						});
					}
				});

				// This is a double render so that the drop-downs can populate (1st render is from refreshSearchResults)
				this.state.set({
					reservedResultsObj: liResultsObj
				}, {silent: false});
			});
	},

	/**
	 * Method that determines which searching mechanism to use (i.e., client or server)
	 */
	search() {

		// Choose which kind of search to do
		if(this.state.get("doClientSideSearch"))
			this.searchThroughClient();
		else
			this.searchThroughAPI();
	},

	/**
	 * Method that searches entire loaded SSLQI list for records matching the entered search criteria
	 */
	searchThroughClient() {

		let liCategoryValue = this.getUI("category").val();
		let liItem_TypeValue = this.getUI("item_Type").val();
		let liItemSpecialtyValue = this.getUI("itemSpecialty").val();

		if(this.validateSearch()) {

			if (liCategoryValue == "all") {
				this.searchResults = this.activeSSLQIs.models;
			} else {

				this.searchResults = this.activeSSLQIs.models.filter((circulatingSSLQI) => {

					if (circulatingSSLQI.get("itemCategory") !== null &&
						circulatingSSLQI.get("itemType") !== null &&
						circulatingSSLQI.get("specialty") !== null) {

						let catCk = circulatingSSLQI.get("itemCategory").toLowerCase().indexOf(liCategoryValue.toLowerCase()) > -1;
						let itCk = circulatingSSLQI.get("itemType").toLowerCase().indexOf(liItem_TypeValue.toLowerCase()) > -1;

						let isCk = false;
						let specialtyArray = circulatingSSLQI.get("specialty");
						$.each(specialtyArray, function (key, value) {
							isCk = value.toLowerCase().indexOf(liItemSpecialtyValue.toLowerCase()) > -1;
							if (isCk) {
								return false;
							} // break out of loop if match found
						});

						return catCk && itCk && isCk;
					}
				});
			}

			this.prepareSearchResultsDisplay();
		}
	},

	/**
	 * Method called when view is first displayed, fetching a complete set of SSLQIs via an API (POST call).
	 * An empty input gets back all SSLQIs.
	 */
	searchThroughAPI() {

		if(this.validateSearch()) {

			let request = new SSLQISearchModel();

			if(this.state.get("categoryValue") !== "all") {

				const sslqiFormData = {};
				$("select, input").each(function(key, element){
					console.log("sslqiFormData:", key, element);
					if (element.id) {

						if(element.id === "category")
							sslqiFormData["itemCategory"] = element.value;

						// Field id is "item_Type" to not to conflict with its tabbed form counterpart of id "itemType"
						if(element.id === "item_Type")
							sslqiFormData["itemType"] = element.value;

						if(element.id === "itemSpecialty")
							sslqiFormData["specialty"] = element.value;
					}
				});

				// TODO: Make classification data population dynamic in some way since it's required
				// Hard coding classification for now
				sslqiFormData["classification"] = "Supplies";

				request.set(sslqiFormData);
			}

			request.save(null, authHeaders())
				.then(() => {
					console.log("Successful search of SSLQI records:\n" + request);
					this.searchResults = (new SSLQISearchCollection(request.get("results"))).models;

					// Store search results for later retrieval across screen navigation changes
					Radio.request("store", "set", "sslqiSearchResults", this.searchResults);

					this.state.set({
						topTableVisible: true
					}, {silent: true});

					this.prepareSearchResultsDisplay();
				})
				.catch(err => {
					console.error("Error searching SSLQI records", err);
				});
		}
	},

	/**
	 * Method called to validate search fields prior to performing a search based on their entered values
	 */
	validateSearch() {

		let liCategoryValue = this.getUI("category").val();
		let liItem_TypeValue = this.getUI("item_Type").val();
		let liItemSpecialtyValue = this.getUI("itemSpecialty").val();
		let liResultMessage = "";
		let liResultMessageAdditional = "";
		let liTopTableVisible = false;
		let isValidated = false;

		if (this.activeSSLQIs === undefined || this.activeSSLQIs.length === 0) {
			liResultMessage = "SSLQI data not loaded yet. Please refresh the screen.";
			isValidated = false;

		} else if(liCategoryValue === "all") {
			liItem_TypeValue = "";
			liItemSpecialtyValue = "";
			isValidated = true;

		} else if (liCategoryValue === "Select Category" ||
				liItem_TypeValue === "Select Item Type" ||
				liItemSpecialtyValue === "Select Item Specialty") {

			if(liCategoryValue === "Select Category" )
				liResultMessageAdditional += " [not \"" + liCategoryValue + "\"]";

			if(liItem_TypeValue === "Select Item Type" )
				liResultMessageAdditional += " [not " + liItem_TypeValue + "\"]";

			if(liItemSpecialtyValue === "Select Item Specialty" )
				liResultMessageAdditional += " [not \"" + liItemSpecialtyValue + "\"]";

			liResultMessage = "Please enter appropriate search data:";
			liResultMessage += liResultMessageAdditional;
			isValidated = false;

		} else {
			isValidated = true;
		}

		if(!isValidated) {

			// TODO: Place this code somewhere where it's accessible by the various points it's currently implemented
			Radio.request("store", "set", "sslqiSearchResults", null);
			Radio.request("store", "set", "sslqiItemCategoryValue", liCategoryValue);
			Radio.request("store", "set", "sslqiItemTypeValue", liItem_TypeValue);
			Radio.request("store", "set", "sslqiItemSpecialtyValue", liItemSpecialtyValue);

			Radio.request("store", "set", "sslqiShowTabsPanel", false);

			this.state.set({
				topTableVisible: liTopTableVisible,
				categoryValue: liCategoryValue,
				item_TypeValue: liItem_TypeValue,
				itemSpecialtyValue: liItemSpecialtyValue,
				current_id: -1,
				showTabsPanel: false
			});

			this.showResultMessage("error", "SSLQI Validation Error", liResultMessage);
		}

		return isValidated;
	},

	/**
	 * Method that prepares search result display
	 */
	prepareSearchResultsDisplay() {

		let liCategoryValue = this.getUI("category").val();
		let liItem_TypeValue = this.getUI("item_Type").val();
		let liItemSpecialtyValue = this.getUI("itemSpecialty").val();
		let liResultsObj = new Object();
		let liResultMessage = "";
		let liShowResultMessage = false;
		let liTopTableVisible = false;

		// Prepare list object with the Mongo related "_id" field as the key and the corresponding
		// SSLQI model object as value
		if (this.searchResults !== null && this.searchResults.length > 0) {

			$.each(this.searchResults, function (key, value) {

				// Weed out bad data from search action
				if (value.get("itemCategory") !== "" &&
					value.get("itemCategory") !== null &&
					value.get("itemCategory") !== undefined) {

					liResultsObj[value.get("_id")] = value;
				}
			});

			// Store search results for later retrieval across screen navigation changes
			Radio.request("store", "set", "sslqiSearchResults", this.searchResults);
			Radio.request("store", "set", "sslqiItemCategoryValue", liCategoryValue);
			Radio.request("store", "set", "sslqiItemTypeValue", liItem_TypeValue);
			Radio.request("store", "set", "sslqiItemSpecialtyValue", liItemSpecialtyValue);
		}

		// Prepare the view state based on search result data
		if (Object.keys(liResultsObj).length > 0) {
			liTopTableVisible = true;
			liShowResultMessage = false;

			if(liCategoryValue.toLowerCase() === "all") {
				liItem_TypeValue = "";
				liItemSpecialtyValue = "";
			}

		} else {
			liTopTableVisible = false;
			liShowResultMessage = true;

			if(liCategoryValue === "all") {
				liResultMessage = "No matching results found for your search";

			} else {
				liResultMessage = "No matching results found for your search on " +
					"[Item Category: '" + liCategoryValue +
					"'], [Item Type: '" + liItem_TypeValue +
					"'], [Item Specialty: '" + liItemSpecialtyValue + "]";
			}
		}

		Radio.request("store", "set", "sslqiShowTabsPanel", false);

		// Re-render the view with result data
		this.state.set({
			resultsObj: liResultsObj,
			topTableVisible: liTopTableVisible,
			categoryValue: liCategoryValue,
			item_TypeValue: liItem_TypeValue,
			itemSpecialtyValue: liItemSpecialtyValue,
			current_id: -1,
			showTabsPanel: false
		});

		if(liShowResultMessage)
			this.showResultMessage("info", "SSLQI Status", liResultMessage);
	},

	/**
	 * Method that converts the latest search results into a state result object for the SSLQI search
	 * result list display. It updates that list and re-displays it while whittling out bad data records.
	 * This method is needed because a result object that has an id key and an SSLQI model object value
	 * is used by the post-display list selection mechanism. The activeSSLQIs object is used to retrieve
	 * the key id values. This method is called by getSSLQIs and does one of the two renderings
	 * that need to happen in order for SSLQI filter dropdowns to be populated fully. When getSSLQIs
	 * is called initially, most of the method is irrelevant, but it's the this.state.set call that
	 * matters. This method is also called through getSSLQIs from the child view's saving and deleting
	 * methods, where the code for the entire method is relevant. It updates the search results
	 * DISPLAY in real time.
	 */
	refreshSearchResults() {

		if(this.state.get("topTableVisible")) {
			let updatedSearchResults = [];
			let updatedResultsObj = new Object();

			$.each(this.searchResults, (key, value) => {

				let result = this.activeSSLQIs.models.filter((circulatingSSLQI) => {
					return circulatingSSLQI.get("_id").toLowerCase().indexOf(value.get("_id").toLowerCase()) > -1;
				});

				// Weed out bad data from buttons action
				if(result[0] !== undefined &&
					result[0].get("itemCategory") !== null &&
					result[0].get("itemType") !== null &&
					result[0].get("specialty") !== []) {

					updatedSearchResults.push(result[0]);
				}
			});

			this.searchResults = updatedSearchResults;

			$.each(this.searchResults, (key, value) => {
				if(value.id !== undefined) {
					updatedResultsObj[value.id] = value;
				}
			});

			this.state.set({
				resultsObj: updatedResultsObj
			},{silent: false});
		}
	},

	/**
	 * Method that shows popup result message display
	 */
	showResultMessage(type, title, message) {
		Radio.request("root", "dialog", {
			type: type,
			title: title,
			message: message
		});
	}
});
