const { StateView } = require("backbone.marionette");
const _ = require("underscore");
const Radio = require("backbone.radio");
const BaseCollection = require("../../../../entities/base/collection");
const SSLQICollection = require("../../../../entities/administration/sslqi_management/collection");
const SelectView = require("../../../../utils/select/view");
const cleanseArray = require("../../../../utils/cleanseArray");
const authHeaders = require("../../../../utils/authHeaders");
const SSLQIResultsView = require("./results/view");

module.exports = StateView.extend({
	className: "flexIt flexIt-container sslqiManagement",
	template: require("./template.hbs"),

	stateEvents: {
		"change:loading": "render",
		"change:filters": "handleFilterChange"
	},

	regions: {
		filterCategory: ".js-r-category",
		filterType: ".js-r-type",
		filterSpecialty: ".js-r-specialty",
		results: { el: ".js-r-results", replaceElement: true }
	},

	triggers: {
		"click .js-reset": "reset:filters",
		"click .js-create-new": "create"
	},

	collectionEvents: {
		sync: "handleSync"
	},

	initialize() {
		StateView.prototype.initialize.apply(this, arguments);
		this.collection = new SSLQICollection();
		this.collection.fetch(authHeaders());
	},

	defaultState() {
		const filters = this.getOption("filters") || {};

		return {
			loading: true,
			filters: {
				category: filters.category || null,
				type: filters.type || null,
				specialty: filters.specialty || null
			}
		};
	},

	getFilteredList() {
		const { category, type, specialty } = this.state.get("filters");
		return this.collection.filter((model) => {
			const matchesCategory = category ? model.get("itemCategory") === category : true;
			const matchesType = type ? model.get("itemType") === type : true;
			const matchesSpecialty = specialty ? _.contains(model.get("specialty"), specialty) : true;

			return matchesCategory && matchesType && matchesSpecialty;
		});
	},

	getAvailableCategories() {
		const categories = this.collection.map(model => model.get("itemCategory"));
		return cleanseArray(categories);
	},

	getAvailableTypes() {
		const types = this.getFilteredList().map(model => model.get("itemType"));
		return cleanseArray(types);
	},

	getAvailableSpecialties() {
		const specialties = this.getFilteredList().reduce((arr, model) => arr.concat(model.get("specialty")), []);
		return cleanseArray(specialties);
	},

	onRender() {
		this.renderFilters();

		if (!this.state.get("loading")) {
			this.showChildView("results", new SSLQIResultsView({
				collection: new SSLQICollection()
			}));

			this.updateResults();
		}
	},

	onChildviewChangeValue() {
		const category = this.getChildView("filterCategory").getValue();
		const type = this.getChildView("filterType").getValue();
		const specialty = this.getChildView("filterSpecialty").getValue();
		const currentFilters = this.state.get("filters");
		const categoryChanged = category !== currentFilters.category;
		const typeChanged = type !== currentFilters.type;

		this.state.set({
			filters: {
				category,
				type: (category && !categoryChanged && type) ? type : null,
				specialty: (category && !categoryChanged && type && !typeChanged && specialty) ? specialty : null
			}
		});
	},

	onChildviewSelectResult(sslqi) {
		Radio.request("global", "navigate", `/admin_sslqi/${sslqi.get("_id")}/`);
	},

	onCreate() {
		Radio.request("global", "navigate", "/admin_sslqi/new/");
	},

	onResetFilters() {
		this.state.set({
			filters: {
				category: null,
				type: null,
				specialty: null
			}
		});
	},

	handleSync() {
		this.state.set("loading", false);
	},

	handleFilterChange() {
		this.renderFilters();
		this.updateResults();
		this.updateUrl();
	},

	updateUrl() {
		const { category, type, specialty } = this.state.get("filters");

		let params = "";
		if (category) { params += `&category=${category}`; }
		if (type) { params += `&type=${type}`; }
		if (specialty) { params += `&specialty=${specialty}`; }
		params = params.replace("&", "?");

		Radio.request("global", "navigate", global.location.pathname + params, {
			trigger: false,
			replace: true
		});
	},

	renderFilters() {
		const { category, type, specialty } = this.state.get("filters");
		const isLoading = this.state.get("loading");
		const mapArrToSelectOptions = arr => arr.map(x => ({ id: x, dspData: x }));

		this.showChildView("filterCategory", new SelectView({
			collection: new BaseCollection(
				mapArrToSelectOptions(this.getAvailableCategories())
			),
			selected: category,
			title: "Filter by Category",
			disabled: isLoading
		}));

		this.showChildView("filterType", new SelectView({
			collection: new BaseCollection(
				mapArrToSelectOptions(this.getAvailableTypes())
			),
			selected: type,
			title: "Filter by Type",
			disabled: isLoading || !category
		}));

		this.showChildView("filterSpecialty", new SelectView({
			collection: new BaseCollection(
				mapArrToSelectOptions(this.getAvailableSpecialties())
			),
			selected: specialty,
			title: "Filter by Specialty",
			disabled: isLoading || !type
		}));
	},

	updateResults() {
		this.getChildView("results").collection.reset(this.getFilteredList());
	}
});
