const { history } = require("backbone");
const { StateView } = require("backbone.marionette");
const Radio = require("backbone.radio");
const TabView = require("../../../../components/tabs/view");
const DetailsForm = require("./details/view");
const UsageForm = require("./usage/view");
const SafetyForm = require("./safety/view");
const authHeaders = require("../../../../utils/authHeaders");

module.exports = StateView.extend({
	className: "flexIt flexIt-container sslqiManagement",
	template: require("./template.hbs"),

	defaultState: {
		currentTab: "details"
	},

	regions: {
		tabs: { el: ".js-r-tabs", replaceElement: true },
		form: { el: ".js-r-form", replaceElement: true }
	},

	triggers: {
		"click .js-save": "submit",
		"click .js-delete": "delete",
		"click .js-cancel": "back",
		"click .js-back": "back"
	},

	childViewEvents: {
		"tab:selected": "showForm",
		"form:submit": "onSubmit"
	},

	onRender() {
		this.showChildView("tabs", new TabView({
			tabs: {
				details: "Details",
				usage: "Usage",
				safety: "Safety"
			},
			selected: "details"
		}));

		this.showForm(this.state.get("currentTab"));
	},

	showForm(tab) {
		const getFormView = (tab) => {
			switch (tab) {
				case "details": return DetailsForm;
				case "usage": return UsageForm;
				case "safety": return SafetyForm;
				default: throw new Error(`No form specified for tab: "${tab}".`);
			}
		};

		const FormView = getFormView(tab);
		this.showChildView("form", new FormView({ model: this.model }));
		this.state.set("currentTab", tab);
	},

	onSubmit() {
		if (!this.model.isValid()) {
			return Radio.request("root", "dialog", {
				type: "danger",
				title: "Form Errors",
				message: "Please complete all required fields and fix fields with errors before saving."
			});
		}

		const updateUrl = this.model.isNew();

		return this.model.save(null, authHeaders())
			.then(() => {
				const id = this.model.get("_id");

				if (updateUrl && id) {
					// Update the URL to recognize the SSLQI is no longer new, will correctly
					// render if user refreshes
					Radio.request("app", "navigate", `/admin_sslqi/${id}`, { trigger: false, replace: true });

					// Re-render to add the Delete button
					this.render();
				}

				return Radio.request("root", "dialog", {
					type: "info",
					title: "Success!",
					message: "Changes to SSLQI have been saved successfully."
				});
			})
			.catch(() => Radio.request("root", "dialog", {
				type: "danger",
				title: "Error",
				message: "SSLQI changes failed to save. Please try again later."
			}));
	},

	onDelete() {
		return this.model.destroy(authHeaders())
			.then(() => {
				Radio.request("root", "dialog", {
					type: "info",
					title: "Success!",
					message: "The SSLQI was successfully deleted."
				});

				this.triggerMethod("back");
			})
			.catch(() => Radio.request("root", "dialog", {
				type: "danger",
				title: "Error",
				message: "Failed to remove SSLQI. Please try again later."
			}));
	},

	onBack() { history.history.back(); }
});
