const Backbone = require("backbone");
const BaseModel = require("../base/model");
import moment from "moment";
const API = require("../../api");
const _ = require("underscore");
const Patient = require("../patient/model");
const authHeaders = require("../../utils/authHeaders");
const getDateOnly = require("../../utils/format_date").getDateOnly;
const Radio = require("backbone.radio");

import ExtendedCase from "./extended";

function fmt_duration(dur) {
	const h = dur.hours(), m = dur.minutes();
	let ret = "";
	if (h) {
		ret = h + (h > 1 ? " hours " : " hour ");
	}
	if (m) {
		ret += m + (m > 1 ? " minutes" : " minute");
	}
	return ret.trim();
}

function get_duration(date_str, start_time, stop_time) {
	if (!start_time || !stop_time) {
		return "";
	}
	const m = moment(stop_time);
	const m2 = moment(start_time);

	let d = moment.duration(m.diff(m2));

	return fmt_duration(d);
}

module.exports = BaseModel.extend({
	idAttribute: "ien",
	urlRoot: API.getURL("CaseByID"),
	inSyncFetch: false,
	initialize() {
		BaseModel.prototype.initialize.apply(this, arguments);

		/* This part saves changes to extended case (if needed)
		   after the case itself is saved.
		 */
		this.listenTo(this, "sync", () => {
			if (!this.inSyncFetch) {
				this.saveExtended();
				this.inSyncFetch = true;
			} else {
				this.inSyncFetch = false;
			}
		});

		/* Wrap get and set on case to handle setting the
		   estimatedOvertime field on the extended case.
		   Backbone-computedfields won't do lazy evaluation of
		   computed fields.  Without that, using
		   backbone-computedfields for this results in the extended
		   case being loaded whenever a case object is created, which
		   is no good when loading a collection of cases (since that
		   results in individually loading all the extended cases that
		   correspond to that case collection and most likely we
		   aren't going to want the extended case in that context
		   anyway.
		*/

		this._originalGet = this.get;
		this.get = _.wrap(this.get, this.getWrapper);

		this._originalSet = this.set;
		this.set = _.wrap(this.set, this.setWrapper);

	},
	getWrapper(origGet) {
		const args = Array.prototype.slice.call(arguments, 1);
		console.log("model getWrapper:", args[0]);
		if (args[0] === "estimatedOverTime") {
			let extended = this.getExtended();
			if (extended) {
				return extended.get("estimatedOverTime");
			} else {
				return "";
			}
		}

		return origGet.apply(this, args);
	},
	setWrapper(origSet) {
		const args = Array.prototype.slice.call(arguments, 1);

		if (args[0] === "estimatedOverTime") {
			let extended = this.getExtended();
			return extended.set("estimatedOverTime", args[1]+"");
		}

		if (_.isObject(args[0]) && args[0].hasOwnProperty("estimatedOverTime")) {
			let extended = this.getExtended();
			extended.set("estimatedOverTime", args[0].estimatedOverTime+"");
			let newArgs = _.omit(args, "estimatedOverTime");
			return origSet.apply(this, newArgs);
		}

		return origSet.apply(this, args);
	},
	computed: {
		calculatedDuration: {
			depends: ["dateOfOperation",
				"scheduledStartTime",
				"scheduledEndTime"
			],
			get(fields) {
				return get_duration(fields.dateOfOperation,
									fields.scheduledStartTime,
									fields.scheduledEndTime);
			}
		}
	},
	fetchPatient() {
		const activePatient = Radio.request("store", "get", "activePatient");
		if (activePatient && activePatient.id === this.get("patient")) {
			this._patient = activePatient;
			return;
		}

		this._patient = new Patient({dfn: this.get("patient")});
		this._patient.fetch(authHeaders());
	},
	getPatient() {
		if (!this.get("patient")) return null;
		if (!this._patient) this.fetchPatient();
		if (!this._patient.id != this.get("patient")) this.fetchPatient();
		return this._patient;
	},
	getExtended() {
		if (!this._extendedCase) this.fetchExtended();
		return this._extendedCase;
	},
	fetchExtended() {
		this._extendedCase = new ExtendedCase({vistaId: this.id});
		let fetchOpts = authHeaders();
		/* Sync is dirty, but when a computed get or set is called,
		   those must be sync.
		*/
		fetchOpts.async = false;
		this._extendedCase.fetch(fetchOpts);
	},
	saveExtended() {
		let fetchOpts = authHeaders();
		this.fetch(fetchOpts);
		let extended = this.getExtended();
		console.log("CaseM oSe:", extended);
		extended.save(null, authHeaders());
	},
	blacklistFields: [
		"roomType",
		"otherPreoperativeDiagnosisCode",
		"surgicalRiskAsrc",
		"relatedToServiceConnectedStatusAndEnvironmentalFactors"
	],
	toJSON(options) {
		let ret = _.clone(this.attributes);
		ret = _.omit(ret, this.blacklistFields);

		ret = _.pick(ret, function(value, key, object) {
			return !_.isNull(value);
		});
		return ret;
	},

	convertCaseStatus(data) {
		if ("2" == data.caseStatus) {
			data.caseStatus = "Unknown Status Value = 2";
		}
		return data;
	},


	trimTimeFromDates(data) {
		data.caseCreateDate = getDateOnly(data.caseCreateDate);
		data.procedureDate = getDateOnly(data.procedureDate);
		data.cancelDate = getDateOnly(data.cancelDate);
		data.dateOfOperation = getDateOnly(data.dateOfOperation);
		if (!data.procedureDate && data.dateOfOperation) {
			data.procedureDate = data.dateOfOperation;
		}
		else if (data.procedureDate && !data.dateOfOperation) {
			data.dateOfOperation = data.procedureDate;
		}
		return data;
	},
	parse(data) {
		/* Current there is a bug in server where a directly fetched
		   single case comes wrapped in an array.  Write the parse to
		   continue working when in the future server presumably
		   returns expected result.
		*/
		const x = _.isArray(data);

		if (_.isArray(data)) {
			data = data[0];
		}
		// data = this.convertCaseStatus(data);
		data = this.trimTimeFromDates(data);
		return data;
	},
	validate(attrs, options) {
		let ret = [];
		if(attrs.scheduledStartTime && attrs.scheduledEndTime) {
			let start = attrs.scheduledStartTime;
			let stop = attrs.scheduledEndTime;
			if (start.split(":")[0].length === 1) {
				start = "0" + start;
			}
			if (stop.split(":")[0].length === 1) {
				stop = "0" + stop;
			}

			if (start >= stop ) {
				ret.push(["scheduledStartTime", "must be earlier than Scheduled End Time"]);
				ret.push(["scheduledEndTime", "must be later than Scheduled Start Time"]);
			}
		}

		let staff = {};

		if(attrs.primarySurgeon) {
//			staff[attrs.primarySurgeon] =
		}
			/*
primarySurgeon
attendingSurgeon
firstAsst
secondAsst
*/

		if (ret.length) {
			return ret;
		} // else return undefined

	}
});
