import "backbone.stickit";
import _ from "underscore";
import Marionette from "backbone.marionette";
import DateTimePicker from "../../behaviors/datetime-picker";

// We put these methods in a Behavior so that they won't interfere with
// extended views using their own lifecycle methods
const FormBehavior = Marionette.Behavior.extend({
	behaviors: {
		dateFields: {
			behaviorClass: DateTimePicker,
			selector: ".js-field-type-date",
			options: {
				format: "MM/DD/YYYY",
				formatDate: "MM/DD/YYYY",
				timepicker: false
			}
		},

		timeFields: {
			behaviorClass: DateTimePicker,
			selector: ".js-field-type-time",
			options: {
				format: "h:mm a",
				formatTime: "h:mm a",
				datepicker: false
			}
		},

		dateTimeFields: {
			behaviorClass: DateTimePicker,
			selector: ".js-field-type-date-time",
			options: {
				format: "MM/DD/YYYY h:mm a",
				formatTime: "h:mm a",
				formatDate: "MM/DD/YYYY"
			}
		}
	},

	initialize() {
		// Store the fields being mapped to SelectViews so we
		// don't need to calculate it each time
		this.selectFields = _.where(this.view.getOption("formFields"), { type: "select" });
	},

	onBeforeAttach() {
		// Create a region for each SelectView
		this.selectFields.forEach(({ field }) => {
			this.view.addRegion(`select-${field}`, `.js-field-region-${field}`);
		});
	},

	onDomRefresh() {
		// Render SelectView into each region, setting the active value
		this.selectFields.forEach((selectField) => {
			let field = selectField.field;
			let selSrc = selectField.selSrc;

			let selectOptions = _.omit(selectField,
										"field", "selSrc", "label", "type");

			selectOptions.selected = [this.view.model.get(field)];

			const selectView = new selSrc(selectOptions);

			// Update model on value change
			this.listenTo(selectView, "change",
				newVal => {
					if (_.isFunction(selectField.onSet)) {
						newVal = selectField.onSet(newVal,
											this.view.model);
					}
					this.view.model.set(field, newVal);
				}
			);

			// Update view on model change
			this.listenTo(this.view.model, `change:${field}`, () => {
				const newVal = this.view.model.get(field);
				if (selectView.getValue() === newVal) { return; }
				selectView.setValue(newVal);
			});

			this.view.showChildView(`select-${field}`, selectView);
		});

		// Initialize field bindings
		this.view.stickit();
	}
});

module.exports = Marionette.View.extend({
	tagName: "form",
	formFields: [],
	behaviors: { FormBehavior },
	template: require("./template.hbs"),

	initialize() {
		if (!this.model) {
			throw new Error("Forms require a model");
		}

		this.listenTo(this.model, "invalid", this.formErrors);
		this.listenTo(this.model, "sync", this.clearErrors);

	},

	triggers() {
		if (!this.getOption("includeSubmit")) { return; }

		return {
			submit: "form:submit"
		};
	},

	clearErrors() {
		this.$(".has-error").removeClass("has-error");
	},
	formErrors(model, errors) {
		console.log("FormView formErrors:", errors);
		for(let error of errors) {
			console.log("Flagging error on:", error[0]);
			this.$("#" + error[0]).parent().addClass("has-error");
		}
	},
	// Sets up a stickit binding for each field
	bindings() {
		const bindings = {};
		this.getOption("formFields").forEach((fieldObj) => {
			let field = fieldObj.field;
			if (!field) {
				return;
			}
			let binding = {
				observe: field
			};
			binding = _.extend(binding,
				_.pick(fieldObj, "onGet", "onSet", "getVal", "update"));
			bindings[`#${field}`] = binding;
		});
		return bindings;
	},

	// Mix in the field data for usage in the template. Putting this in
	// serializeData to avoid any potential collisions with extended views
	// using templateContext
	serializeData(...args) {
		return {
			...Marionette.View.prototype.serializeData.apply(this, args),
			_fields: this.getOption("formFields"),
			_includeSubmit: this.getOption("includeSubmit")
		};
	}
});
