const Backbone = require("backbone");
const Marionette = require("backbone.marionette");
const Radio = require("backbone.radio");
const Router = require("./router");
const RootView = require("./modules/global/root/view");
const Auth = require("./globalAPI/Auth");
const Store = require("./globalAPI/Store");
const FacilityModel = require("./entities/facilities/details/model");
const ViewAccessCollection = require("./entities/administration/view_access_config/collection");
const storage = require("./utils/storage");
const authHeaders = require("./utils/authHeaders");

module.exports = Marionette.Application.extend({
	region: "#app",

	initialize() {
		// We only listen to the first "authenticate" event as the only time
		// this function will need to run is on a refresh of the page. In this
		// case the user likely already had a session, or is logging-in to
		// start a new session.
		Radio.once("auth", "user:authenticate", this.fetchInitialData, this);
	},

	loadViewAccess() {
		const access = new ViewAccessCollection();
		return access.fetch()
			.then(() => Radio.request("store", "set", "viewAccess", access));
	},

	onStart() {
		this.router = new Router();
		this.store = new Store();
		this.auth = new Auth();

		// Need to load up the view access configuration before rendering anything
		return this.loadViewAccess().then(() => {
			this.showView(new RootView());
			Radio.reply("global", "navigate", this.navigate, this);

			// Invoke routing
			Backbone.Intercept.start();
			Backbone.history.start({pushState: true});
		});
	},

	navigate(uri, options = {trigger: true}) {
		return this.router.navigate(uri, options);
	},

	/**
	 * Searches for a stored ID in sessionStorage and a stored model
	 * in the global store, creating & fetching a new instance of the
	 * provided constructor if an ID but no model is found.
	 *
	 * @param storedId    Key the id is stored under in sessionStorage
	 * @param storedKey   Key the model is stored under in global store
	 * @param Ctor        Model constructor to use if no model found
	 * @returns {Promise}
	 * @private
	 */
	_fetchData(storedId, storedKey, Ctor) {
		const id = storage.getItem(storedId);
		const activeModel = this.store.getItem(storedKey);

		// We're only concerned if an ID has been stored but no matching
		// exists in the global store
		if (!id || activeModel) {
			return Promise.resolve();
		}

		const model = new Ctor();
		return model
			.set(model.idAttribute, id)
			.fetch(authHeaders())
			.then(() => this.store.setItem(storedKey, model));
	},

	/**
	 * Fetches previously set global data from last session on
	 * authentication event
	 * @returns {Promise}
	 */
	fetchInitialData() {
		return Promise.all([
			this._fetchData("facilityId", "activeFacility", FacilityModel) // ,

			// Need to determine how to fetch a single patient, currently
			// only coming from a collection.
			//
			// this._fetchData('patientId', 'activePatient', PatientModel),

			// Need to re-visit this once the select case module has been
			// refactored, currently it's setting the activeCase to a plain
			// Object and not an actual Case Model like it had been previously.
			//
			// this._fetchData('caseId', 'activeCase', CaseModel),
		]);
	}
});
