/** Contains default routes in application **/
'use strict';

require('dotenv').config();
let debug = require('debug')('http');
let jwt = require('jsonwebtoken'); // used to create, sign, and verify tokens
let authenticate = require('./app/services/authenticate');
let site = require('./app/services/cnf/site');
const facilities = require("./app/services/cnf/site");
const roles = require("./app/services/cnf/role");
const cptConfig = require("./app/controller/cptConfig");
// interceptor-audit libraries
const interceptor = require('express-interceptor');
let audit = require('./app/services/cnf/audit');


module.exports = function(app) {

    // Write audit log to DB
    app.use(updateAudit);

    app.get('/', function(req, res) {
        res.send('hello world');
    });

    /*
    app.use(function(req, res, next) {
        // in local we are going to add the access control headers to allow developers
        // to work locally 
        if (process.env.NODE_ENV == 'local') {
            // Headers for local development to work
            // Gui PORT site
            res.setHeader('Access-Control-Allow-Origin', '*');
            // Request methods allowed by GUI
            res.setHeader('Access-Control-Allow-Methods', 'GET, POST, OPTIONS, PUT, PATCH, DELETE');
            // Request headers allowed
            res.setHeader('Access-Control-Allow-Headers', 'authorization, X-Requested-With,content-type');
        }
        next();
    });*/

    /** 
    * middle ware for local or test environments
    **/
    app.use(function(req, res, next){
        if (process.env.NODE_ENV == 'local' || process.env.NODE_ENV == 'test') {
             // set access control allow origin header
            // Website you wish to allow to connect
            res.setHeader('Access-Control-Allow-Origin', '*');

            // Request methods you wish to allow
            res.setHeader('Access-Control-Allow-Methods', 'GET, POST, OPTIONS, PUT, PATCH, DELETE, OPTIONS');

            // Request headers you wish to allow
            res.setHeader('Access-Control-Allow-Headers', 'X-Requested-With,content-type,authorization');
        }
        next();
    });

    app.get("/facilities", function(req, res) {
        

        facilities.getFacilitiesList(function(err, result) {
            if (err) { res.json(err); } else { res.json(result); }

        });
    });

    /** Authenticate user and provide a JWT token **/
    app.post('/authenticate',
        function(req, res) {
            if (req.body.stationID === undefined){
                    res.status(401).json({
                    success: false,
                    message: 'Please specify a facility VistA instance to connect to.',
                    error: 'ERROR:  The facility station ID is missing, incorrect, or does not exist in the database. Please contact your local VistA administrator for support if the problem persists.'
                });
            }
            else if (req.body.access != undefined && req.body.verify != undefined){// && req.body.stationID != undefined) {
                site.getConnectionInfo(req.body.stationID, function(ipResult) {
                    if (ipResult != undefined && ipResult.ipAddress != undefined && ipResult.port != undefined) {
                        authenticate.verify(req.body, function(error, result) {
                            if (error) {
                                console.log(error);
                                if (error.message.includes("ECONNRESET")){
                                    res.status(421).json({
                                        success: false,
                                        message: 'Service temporarily down.',
                                        error: error.message
                                    });
                                }
                                else{
                                    res.status(401).json({
                                        success: false,
                                        message: 'Access code or verify code incorrect.',
                                        error: error.message
                                    });
                                }
                            } else {
                                if (result.duz != undefined) {
                                    // Retrieve list of roles
                                    authenticate.retrieveKeys(req.body, result.duz, function(err, userKeys) {
                                        if (err) {
                                            res.status(401).json({
                                                success: false,
                                                message: 'There was an error when attempting to access user roles. Please contact your local VistA administrator for support.',
                                                error: err.message
                                            });
                                        }
                                        else if (userKeys === undefined || userKeys === null || userKeys.length === 0){
                                            res.status(401).json({
                                                success: false,
                                                message: 'The current user does NOT have the appropriate security keys to access this system. Please contact your local VistA administrator for support.',
                                                error: 'ERROR:  The user that is attempting to login does not have the appropriate VistA security keys.'
                                            });
                                        }
                                        else {
                                            console.log('verified');

                                            roles.getRolesForKeys(userKeys.split('^'), function(error, userRoles) {
                                                let thirty_min_ahead = new Date();
                                                thirty_min_ahead.setMinutes(new Date().getMinutes() + 30);

                                                console.log("Thirty Min ahead: ", thirty_min_ahead);

                                                let payload = {
                                                    // the original access token
                                                    accessCode: req.body.access,
                                                    // original verify token
                                                    verify: req.body.verify,
                                                    // the station id of the facility
                                                    stationID: req.body.stationID,
                                                    // return date in unix epoch
                                                    exp: parseInt(thirty_min_ahead / 1000),
                                                    // issuer is who issued, we have an arbitrary issuer name
                                                    iss: 'vista-surgery-api',
                                                    // audience incase we have multiple audience members
                                                    aud: 'vista-surgery-api'
                                                        // add scope to support roles

                                                };

                                                // create a token
                                                let token = jwt.sign(payload, process.env.JWT_SECRET);
                                                let name = result.greeting.split(' ')[2];

                                                // Check CPT Duration Averages
                                                cptConfig.checkCPTDurations(req.body);

                                                authenticate.userInfo(req.body, function(err, data){
                                                    // return the information including token as JSON
                                                    res.status(200).json({
                                                        ien: result.duz,
                                                        success: true,
                                                        message: 'Enjoy your token!',
                                                        firstname: name.split(',')[1],
                                                        lastname: name.split(',')[0],
                                                        user_info: data,
                                                        roles: userRoles,
                                                        token: token
                                                    });
                                                });
                                                
                                            });
                                        }
                                    });
                                } else {
                                    res.status(401).json({
                                        success: false,
                                        message: 'The current user does not have the appropriate security keys for this system. Please contact your local VistA administrator for support.',
                                        error: 'ERROR:  The user that is attempting to login does not have the appropriate VistA security keys.'
                                    });
                                }

                            }
                        });
                    } else {
                        res.status(401).json({
                            success: false,
                            message: 'Facility connection information is incorrect or missing. Please contact your local VistA administrator for support.',
                            error: 'ERROR: The facility connection information is incorrect or missing.'
                        });
                    }
                });

            } else {
                res.status(401).json({
                    success: false,
                    message: 'Access or Verify code missing.'
                });
            }
        });


    /** 
	  	Authentication Middleware, place any routes you want to be authenticated, below this middle ware 
	  	*/
    app.use(function(req, res, next) {
        // skip authentication for local
        console.log("node env: " + (process.env.NODE_ENV !== 'local'));
        if (process.env.NODE_ENV !== 'local' && process.env.NODE_ENV !== 'test') {
            console.log('not local or test: ' + process.env.NODE_ENV);
            // check header or url parameters or post parameters for token
            var token = req.body.token || req.query.token || req.headers['authorization'];
            if (token) {
                // verifies secret and checks exp
                jwt.verify(token, process.env.JWT_SECRET, function(err, decoded) {
                    if (err) {
                        res.setHeader("WWW-Authenticate", 'JWT realm: /authenticate');
                        res.statusCode = 401;
                        res.end('Failed to authenticate token. ' + err);
                        return res;
                    } else {
                        // if everything is good, save to request for use in other routes
                        req.decoded = decoded;
                        next();
                    }
                });
            } else {
                console.log('where is the token');
                // if there is no token
                // return an error
                res.setHeader("WWW-Authenticate", 'JWT realm: /authenticate');
                res.statusCode = (401);
                res.end('No token provided.');
                return res;
            }
        } else {
            console.log('must be local or test');
            console.log('what are you: ' + process.env.NODE_ENV);
            // in test  we are just going to hardcode the access / verify code needed

            if (!req.decoded) {
                // set the access control header when its not set in local or test mode
                req.decoded = { accessCode: "VS2017", verifyCode: "VS2018***", stationID: "648" };
            }


            next();
        }
    });


    /**
     * Authenticated example
     */
    app.get("/test", function(req, res) {
        debug("Decoded Access / Verify", req.decoded);
        res.send('Authenticated: hello world');
    });


    /** load all authenticated routes here **/
    app.use('/v1', require('./app/routes/api_v1'));

    /**
     * Error handling
     */

    app.use(function(err, req, res, next) {
        // treat as 404
        if (err.message &&
            (~err.message.indexOf('not found') ||
                (~err.message.indexOf('Cast to ObjectId failed')))) {
            return next();
        }

        console.error(err.stack);

        if (err.stack.includes('ValidationError')) {
            res.status(422).render('422', {
                error: err.stack
            });
            return;
        }

        // error page
        res.status(500).render('500', {
            error: err.stack
        });
    });

    // assume 404 since no middleware responded
    app.use(function(req, res) {
        res.status(404).json({
            "invalid": "url"
        });
    });
}

let updateAudit = interceptor (function (req, res)
{
    return {

        isInterceptable: function()
        {
            return (req.method === "POST" || req.method === "PUT" ) && /application\/json/.test(res.get('Content-Type'));
        },
        intercept: function(body, send)
        {
            if (req.decoded)
            {
                let user = req.decoded.accessCode;
                let jsonBody = JSON.parse(body);
                let id;
                if (req.method === "POST" && req.originalUrl.includes("/v1/") && !req.originalUrl.includes("/search")) {
                    id = jsonBody._id || jsonBody.ien;
                    if (jsonBody._id !== undefined) {
                        // Mongo data
                        audit.createNewEntry(id, JSON.stringify(req.body), req.originalUrl, user, audit.CREATE, audit.MDB_DT, function(err, data){
                            if (!err){send(body);}
                            else{
                                console.log("ERROR: " + err);
                                send(body);
                            }
                        });
                    }
                    else if (jsonBody.ien !== undefined) {
                        // VistA data
                        audit.createNewEntry(id, JSON.stringify(req.body), req.originalUrl, user, audit.CREATE, audit.VISTA_DT, function(err, data){
                            if (!err){send(body);}
                            else{
                                console.log("ERROR: " + err);
                                send(body);
                            }
                        });
                    }
                    else{
                        audit.createNewEntry(null, JSON.stringify(req.body), req.originalUrl, user, audit.CREATE, audit.UNKNOWN_DT, function(err, data){
                            if (!err){send(body);}
                            else{
                                console.log("ERROR: " + err);
                                send(body);
                            }
                        });
                    }
                }
                else if (req.method === "PUT") {
                    id = req.body._id || req.body.ien || undefined;
                    if (id !== undefined) {
                        audit.createNewEntry(id, JSON.stringify(req.body), req.originalUrl, user, audit.UPDATE, function (err, data) {
                            if (!err) {
                                send(body);
                            }
                            else {
                                console.log("ERROR: " + err);
                                send(body);
                            }
                        });
                    }
                    else {send(body);}
                }
                else if (req.method === "DELETE") {
                    id = req.body._id || req.body.ien || req.body.id || undefined;
                    if (id !== undefined) {
                        audit.createNewEntry(id, JSON.stringify(req.body), req.originalUrl, user, audit.DELETE, function (err, data) {
                            if (!err) {
                                send(body);
                            }
                            else {
                                console.log("ERROR: " + err);
                                send(body);
                            }
                        });
                    }
                    else {send(body);}
                }
                else
                {
                    send(body);
                }
            }
            else
            {
                send(body);
            }
        }
    };

})